/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalize.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration properties for the solution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SolutionConfig implements SdkPojo, Serializable, ToCopyableBuilder<SolutionConfig.Builder, SolutionConfig> {
    private static final SdkField<String> EVENT_VALUE_THRESHOLD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SolutionConfig::eventValueThreshold)).setter(setter(Builder::eventValueThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventValueThreshold").build())
            .build();

    private static final SdkField<HPOConfig> HPO_CONFIG_FIELD = SdkField.<HPOConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(SolutionConfig::hpoConfig)).setter(setter(Builder::hpoConfig)).constructor(HPOConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hpoConfig").build()).build();

    private static final SdkField<Map<String, String>> ALGORITHM_HYPER_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(SolutionConfig::algorithmHyperParameters))
            .setter(setter(Builder::algorithmHyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithmHyperParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> FEATURE_TRANSFORMATION_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(SolutionConfig::featureTransformationParameters))
            .setter(setter(Builder::featureTransformationParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("featureTransformationParameters")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<AutoMLConfig> AUTO_ML_CONFIG_FIELD = SdkField.<AutoMLConfig> builder(MarshallingType.SDK_POJO)
            .getter(getter(SolutionConfig::autoMLConfig)).setter(setter(Builder::autoMLConfig))
            .constructor(AutoMLConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoMLConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_VALUE_THRESHOLD_FIELD,
            HPO_CONFIG_FIELD, ALGORITHM_HYPER_PARAMETERS_FIELD, FEATURE_TRANSFORMATION_PARAMETERS_FIELD, AUTO_ML_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String eventValueThreshold;

    private final HPOConfig hpoConfig;

    private final Map<String, String> algorithmHyperParameters;

    private final Map<String, String> featureTransformationParameters;

    private final AutoMLConfig autoMLConfig;

    private SolutionConfig(BuilderImpl builder) {
        this.eventValueThreshold = builder.eventValueThreshold;
        this.hpoConfig = builder.hpoConfig;
        this.algorithmHyperParameters = builder.algorithmHyperParameters;
        this.featureTransformationParameters = builder.featureTransformationParameters;
        this.autoMLConfig = builder.autoMLConfig;
    }

    /**
     * <p>
     * Only events with a value greater than or equal to this threshold are used for training a model.
     * </p>
     * 
     * @return Only events with a value greater than or equal to this threshold are used for training a model.
     */
    public String eventValueThreshold() {
        return eventValueThreshold;
    }

    /**
     * <p>
     * Describes the properties for hyperparameter optimization (HPO).
     * </p>
     * 
     * @return Describes the properties for hyperparameter optimization (HPO).
     */
    public HPOConfig hpoConfig() {
        return hpoConfig;
    }

    /**
     * Returns true if the AlgorithmHyperParameters property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAlgorithmHyperParameters() {
        return algorithmHyperParameters != null && !(algorithmHyperParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Lists the hyperparameter names and ranges.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAlgorithmHyperParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Lists the hyperparameter names and ranges.
     */
    public Map<String, String> algorithmHyperParameters() {
        return algorithmHyperParameters;
    }

    /**
     * Returns true if the FeatureTransformationParameters property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public boolean hasFeatureTransformationParameters() {
        return featureTransformationParameters != null && !(featureTransformationParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Lists the feature transformation parameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFeatureTransformationParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Lists the feature transformation parameters.
     */
    public Map<String, String> featureTransformationParameters() {
        return featureTransformationParameters;
    }

    /**
     * <p>
     * The <a>AutoMLConfig</a> object containing a list of recipes to search when AutoML is performed.
     * </p>
     * 
     * @return The <a>AutoMLConfig</a> object containing a list of recipes to search when AutoML is performed.
     */
    public AutoMLConfig autoMLConfig() {
        return autoMLConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventValueThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(hpoConfig());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmHyperParameters());
        hashCode = 31 * hashCode + Objects.hashCode(featureTransformationParameters());
        hashCode = 31 * hashCode + Objects.hashCode(autoMLConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SolutionConfig)) {
            return false;
        }
        SolutionConfig other = (SolutionConfig) obj;
        return Objects.equals(eventValueThreshold(), other.eventValueThreshold())
                && Objects.equals(hpoConfig(), other.hpoConfig())
                && Objects.equals(algorithmHyperParameters(), other.algorithmHyperParameters())
                && Objects.equals(featureTransformationParameters(), other.featureTransformationParameters())
                && Objects.equals(autoMLConfig(), other.autoMLConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SolutionConfig").add("EventValueThreshold", eventValueThreshold()).add("HpoConfig", hpoConfig())
                .add("AlgorithmHyperParameters", algorithmHyperParameters())
                .add("FeatureTransformationParameters", featureTransformationParameters()).add("AutoMLConfig", autoMLConfig())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eventValueThreshold":
            return Optional.ofNullable(clazz.cast(eventValueThreshold()));
        case "hpoConfig":
            return Optional.ofNullable(clazz.cast(hpoConfig()));
        case "algorithmHyperParameters":
            return Optional.ofNullable(clazz.cast(algorithmHyperParameters()));
        case "featureTransformationParameters":
            return Optional.ofNullable(clazz.cast(featureTransformationParameters()));
        case "autoMLConfig":
            return Optional.ofNullable(clazz.cast(autoMLConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SolutionConfig, T> g) {
        return obj -> g.apply((SolutionConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SolutionConfig> {
        /**
         * <p>
         * Only events with a value greater than or equal to this threshold are used for training a model.
         * </p>
         * 
         * @param eventValueThreshold
         *        Only events with a value greater than or equal to this threshold are used for training a model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventValueThreshold(String eventValueThreshold);

        /**
         * <p>
         * Describes the properties for hyperparameter optimization (HPO).
         * </p>
         * 
         * @param hpoConfig
         *        Describes the properties for hyperparameter optimization (HPO).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hpoConfig(HPOConfig hpoConfig);

        /**
         * <p>
         * Describes the properties for hyperparameter optimization (HPO).
         * </p>
         * This is a convenience that creates an instance of the {@link HPOConfig.Builder} avoiding the need to create
         * one manually via {@link HPOConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link HPOConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #hpoConfig(HPOConfig)}.
         * 
         * @param hpoConfig
         *        a consumer that will call methods on {@link HPOConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hpoConfig(HPOConfig)
         */
        default Builder hpoConfig(Consumer<HPOConfig.Builder> hpoConfig) {
            return hpoConfig(HPOConfig.builder().applyMutation(hpoConfig).build());
        }

        /**
         * <p>
         * Lists the hyperparameter names and ranges.
         * </p>
         * 
         * @param algorithmHyperParameters
         *        Lists the hyperparameter names and ranges.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmHyperParameters(Map<String, String> algorithmHyperParameters);

        /**
         * <p>
         * Lists the feature transformation parameters.
         * </p>
         * 
         * @param featureTransformationParameters
         *        Lists the feature transformation parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTransformationParameters(Map<String, String> featureTransformationParameters);

        /**
         * <p>
         * The <a>AutoMLConfig</a> object containing a list of recipes to search when AutoML is performed.
         * </p>
         * 
         * @param autoMLConfig
         *        The <a>AutoMLConfig</a> object containing a list of recipes to search when AutoML is performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMLConfig(AutoMLConfig autoMLConfig);

        /**
         * <p>
         * The <a>AutoMLConfig</a> object containing a list of recipes to search when AutoML is performed.
         * </p>
         * This is a convenience that creates an instance of the {@link AutoMLConfig.Builder} avoiding the need to
         * create one manually via {@link AutoMLConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoMLConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #autoMLConfig(AutoMLConfig)}.
         * 
         * @param autoMLConfig
         *        a consumer that will call methods on {@link AutoMLConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoMLConfig(AutoMLConfig)
         */
        default Builder autoMLConfig(Consumer<AutoMLConfig.Builder> autoMLConfig) {
            return autoMLConfig(AutoMLConfig.builder().applyMutation(autoMLConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String eventValueThreshold;

        private HPOConfig hpoConfig;

        private Map<String, String> algorithmHyperParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> featureTransformationParameters = DefaultSdkAutoConstructMap.getInstance();

        private AutoMLConfig autoMLConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(SolutionConfig model) {
            eventValueThreshold(model.eventValueThreshold);
            hpoConfig(model.hpoConfig);
            algorithmHyperParameters(model.algorithmHyperParameters);
            featureTransformationParameters(model.featureTransformationParameters);
            autoMLConfig(model.autoMLConfig);
        }

        public final String getEventValueThreshold() {
            return eventValueThreshold;
        }

        @Override
        public final Builder eventValueThreshold(String eventValueThreshold) {
            this.eventValueThreshold = eventValueThreshold;
            return this;
        }

        public final void setEventValueThreshold(String eventValueThreshold) {
            this.eventValueThreshold = eventValueThreshold;
        }

        public final HPOConfig.Builder getHpoConfig() {
            return hpoConfig != null ? hpoConfig.toBuilder() : null;
        }

        @Override
        public final Builder hpoConfig(HPOConfig hpoConfig) {
            this.hpoConfig = hpoConfig;
            return this;
        }

        public final void setHpoConfig(HPOConfig.BuilderImpl hpoConfig) {
            this.hpoConfig = hpoConfig != null ? hpoConfig.build() : null;
        }

        public final Map<String, String> getAlgorithmHyperParameters() {
            return algorithmHyperParameters;
        }

        @Override
        public final Builder algorithmHyperParameters(Map<String, String> algorithmHyperParameters) {
            this.algorithmHyperParameters = HyperParametersCopier.copy(algorithmHyperParameters);
            return this;
        }

        public final void setAlgorithmHyperParameters(Map<String, String> algorithmHyperParameters) {
            this.algorithmHyperParameters = HyperParametersCopier.copy(algorithmHyperParameters);
        }

        public final Map<String, String> getFeatureTransformationParameters() {
            return featureTransformationParameters;
        }

        @Override
        public final Builder featureTransformationParameters(Map<String, String> featureTransformationParameters) {
            this.featureTransformationParameters = FeatureTransformationParametersCopier.copy(featureTransformationParameters);
            return this;
        }

        public final void setFeatureTransformationParameters(Map<String, String> featureTransformationParameters) {
            this.featureTransformationParameters = FeatureTransformationParametersCopier.copy(featureTransformationParameters);
        }

        public final AutoMLConfig.Builder getAutoMLConfig() {
            return autoMLConfig != null ? autoMLConfig.toBuilder() : null;
        }

        @Override
        public final Builder autoMLConfig(AutoMLConfig autoMLConfig) {
            this.autoMLConfig = autoMLConfig;
            return this;
        }

        public final void setAutoMLConfig(AutoMLConfig.BuilderImpl autoMLConfig) {
            this.autoMLConfig = autoMLConfig != null ? autoMLConfig.build() : null;
        }

        @Override
        public SolutionConfig build() {
            return new SolutionConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
