/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53domains;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53domains.model.CheckDomainAvailabilityRequest;
import software.amazon.awssdk.services.route53domains.model.CheckDomainAvailabilityResponse;
import software.amazon.awssdk.services.route53domains.model.CheckDomainTransferabilityRequest;
import software.amazon.awssdk.services.route53domains.model.CheckDomainTransferabilityResponse;
import software.amazon.awssdk.services.route53domains.model.DeleteTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.DeleteTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.DisableDomainAutoRenewRequest;
import software.amazon.awssdk.services.route53domains.model.DisableDomainAutoRenewResponse;
import software.amazon.awssdk.services.route53domains.model.DisableDomainTransferLockRequest;
import software.amazon.awssdk.services.route53domains.model.DisableDomainTransferLockResponse;
import software.amazon.awssdk.services.route53domains.model.DomainLimitExceededException;
import software.amazon.awssdk.services.route53domains.model.DuplicateRequestException;
import software.amazon.awssdk.services.route53domains.model.EnableDomainAutoRenewRequest;
import software.amazon.awssdk.services.route53domains.model.EnableDomainAutoRenewResponse;
import software.amazon.awssdk.services.route53domains.model.EnableDomainTransferLockRequest;
import software.amazon.awssdk.services.route53domains.model.EnableDomainTransferLockResponse;
import software.amazon.awssdk.services.route53domains.model.GetContactReachabilityStatusRequest;
import software.amazon.awssdk.services.route53domains.model.GetContactReachabilityStatusResponse;
import software.amazon.awssdk.services.route53domains.model.GetDomainDetailRequest;
import software.amazon.awssdk.services.route53domains.model.GetDomainDetailResponse;
import software.amazon.awssdk.services.route53domains.model.GetDomainSuggestionsRequest;
import software.amazon.awssdk.services.route53domains.model.GetDomainSuggestionsResponse;
import software.amazon.awssdk.services.route53domains.model.GetOperationDetailRequest;
import software.amazon.awssdk.services.route53domains.model.GetOperationDetailResponse;
import software.amazon.awssdk.services.route53domains.model.InvalidInputException;
import software.amazon.awssdk.services.route53domains.model.ListDomainsRequest;
import software.amazon.awssdk.services.route53domains.model.ListDomainsResponse;
import software.amazon.awssdk.services.route53domains.model.ListOperationsRequest;
import software.amazon.awssdk.services.route53domains.model.ListOperationsResponse;
import software.amazon.awssdk.services.route53domains.model.ListTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.ListTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.OperationLimitExceededException;
import software.amazon.awssdk.services.route53domains.model.RegisterDomainRequest;
import software.amazon.awssdk.services.route53domains.model.RegisterDomainResponse;
import software.amazon.awssdk.services.route53domains.model.RenewDomainRequest;
import software.amazon.awssdk.services.route53domains.model.RenewDomainResponse;
import software.amazon.awssdk.services.route53domains.model.ResendContactReachabilityEmailRequest;
import software.amazon.awssdk.services.route53domains.model.ResendContactReachabilityEmailResponse;
import software.amazon.awssdk.services.route53domains.model.RetrieveDomainAuthCodeRequest;
import software.amazon.awssdk.services.route53domains.model.RetrieveDomainAuthCodeResponse;
import software.amazon.awssdk.services.route53domains.model.Route53DomainsException;
import software.amazon.awssdk.services.route53domains.model.Route53DomainsRequest;
import software.amazon.awssdk.services.route53domains.model.TldRulesViolationException;
import software.amazon.awssdk.services.route53domains.model.TransferDomainRequest;
import software.amazon.awssdk.services.route53domains.model.TransferDomainResponse;
import software.amazon.awssdk.services.route53domains.model.UnsupportedTldException;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactPrivacyRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactPrivacyResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainNameserversRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainNameserversResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.ViewBillingRequest;
import software.amazon.awssdk.services.route53domains.model.ViewBillingResponse;
import software.amazon.awssdk.services.route53domains.paginators.ListDomainsPublisher;
import software.amazon.awssdk.services.route53domains.paginators.ListOperationsPublisher;
import software.amazon.awssdk.services.route53domains.transform.CheckDomainAvailabilityRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.CheckDomainTransferabilityRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DeleteTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DisableDomainAutoRenewRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DisableDomainTransferLockRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.EnableDomainAutoRenewRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.EnableDomainTransferLockRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetContactReachabilityStatusRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetDomainDetailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetDomainSuggestionsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetOperationDetailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RegisterDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RenewDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ResendContactReachabilityEmailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RetrieveDomainAuthCodeRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.TransferDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainContactPrivacyRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainContactRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainNameserversRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ViewBillingRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Route53DomainsAsyncClient}.
 *
 * @see Route53DomainsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53DomainsAsyncClient implements Route53DomainsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRoute53DomainsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53DomainsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * This operation checks the availability of one domain name. Note that if the availability status of a domain is
     * pending, you must submit another request to determine the availability of the domain name.
     * </p>
     *
     * @param checkDomainAvailabilityRequest
     *        The CheckDomainAvailability request contains the following elements.
     * @return A Java Future containing the result of the CheckDomainAvailability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.CheckDomainAvailability
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/CheckDomainAvailability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckDomainAvailabilityResponse> checkDomainAvailability(
            CheckDomainAvailabilityRequest checkDomainAvailabilityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckDomainAvailabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckDomainAvailabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckDomainAvailabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckDomainAvailabilityRequest, CheckDomainAvailabilityResponse>()
                            .withOperationName("CheckDomainAvailability")
                            .withMarshaller(new CheckDomainAvailabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(checkDomainAvailabilityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks whether a domain name can be transferred to Amazon Route 53.
     * </p>
     *
     * @param checkDomainTransferabilityRequest
     *        The CheckDomainTransferability request contains the following elements.
     * @return A Java Future containing the result of the CheckDomainTransferability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.CheckDomainTransferability
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/CheckDomainTransferability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckDomainTransferabilityResponse> checkDomainTransferability(
            CheckDomainTransferabilityRequest checkDomainTransferabilityRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckDomainTransferabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckDomainTransferabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckDomainTransferabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckDomainTransferabilityRequest, CheckDomainTransferabilityResponse>()
                            .withOperationName("CheckDomainTransferability")
                            .withMarshaller(new CheckDomainTransferabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(checkDomainTransferabilityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation deletes the specified tags for a domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param deleteTagsForDomainRequest
     *        The DeleteTagsForDomainRequest includes the following elements.
     * @return A Java Future containing the result of the DeleteTagsForDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.DeleteTagsForDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DeleteTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTagsForDomainResponse> deleteTagsForDomain(
            DeleteTagsForDomainRequest deleteTagsForDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTagsForDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTagsForDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsForDomainRequest, DeleteTagsForDomainResponse>()
                            .withOperationName("DeleteTagsForDomain")
                            .withMarshaller(new DeleteTagsForDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTagsForDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation disables automatic renewal of domain registration for the specified domain.
     * </p>
     *
     * @param disableDomainAutoRenewRequest
     * @return A Java Future containing the result of the DisableDomainAutoRenew operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.DisableDomainAutoRenew
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DisableDomainAutoRenew"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableDomainAutoRenewResponse> disableDomainAutoRenew(
            DisableDomainAutoRenewRequest disableDomainAutoRenewRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableDomainAutoRenewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableDomainAutoRenewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableDomainAutoRenewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableDomainAutoRenewRequest, DisableDomainAutoRenewResponse>()
                            .withOperationName("DisableDomainAutoRenew")
                            .withMarshaller(new DisableDomainAutoRenewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disableDomainAutoRenewRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation removes the transfer lock on the domain (specifically the <code>clientTransferProhibited</code>
     * status) to allow domain transfers. We recommend you refrain from performing this action unless you intend to
     * transfer the domain to a different registrar. Successful submission returns an operation ID that you can use to
     * track the progress and completion of the action. If the request is not completed successfully, the domain
     * registrant will be notified by email.
     * </p>
     *
     * @param disableDomainTransferLockRequest
     *        The DisableDomainTransferLock request includes the following element.
     * @return A Java Future containing the result of the DisableDomainTransferLock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>DuplicateRequestException The request is already in progress for the domain.</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.DisableDomainTransferLock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DisableDomainTransferLock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableDomainTransferLockResponse> disableDomainTransferLock(
            DisableDomainTransferLockRequest disableDomainTransferLockRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableDomainTransferLockResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableDomainTransferLockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableDomainTransferLockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableDomainTransferLockRequest, DisableDomainTransferLockResponse>()
                            .withOperationName("DisableDomainTransferLock")
                            .withMarshaller(new DisableDomainTransferLockRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disableDomainTransferLockRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation configures Amazon Route 53 to automatically renew the specified domain before the domain
     * registration expires. The cost of renewing your domain registration is billed to your AWS account.
     * </p>
     * <p>
     * The period during which you can renew a domain name varies by TLD. For a list of TLDs and their renewal policies,
     * see <a href="http://wiki.gandi.net/en/domains/renew#renewal_restoration_and_deletion_times">
     * "Renewal, restoration, and deletion times"</a> on the website for our registrar associate, Gandi. Amazon Route 53
     * requires that you renew before the end of the renewal period that is listed on the Gandi website so we can
     * complete processing before the deadline.
     * </p>
     *
     * @param enableDomainAutoRenewRequest
     * @return A Java Future containing the result of the EnableDomainAutoRenew operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.EnableDomainAutoRenew
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/EnableDomainAutoRenew"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableDomainAutoRenewResponse> enableDomainAutoRenew(
            EnableDomainAutoRenewRequest enableDomainAutoRenewRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableDomainAutoRenewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableDomainAutoRenewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableDomainAutoRenewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableDomainAutoRenewRequest, EnableDomainAutoRenewResponse>()
                            .withOperationName("EnableDomainAutoRenew")
                            .withMarshaller(new EnableDomainAutoRenewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(enableDomainAutoRenewRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation sets the transfer lock on the domain (specifically the <code>clientTransferProhibited</code>
     * status) to prevent domain transfers. Successful submission returns an operation ID that you can use to track the
     * progress and completion of the action. If the request is not completed successfully, the domain registrant will
     * be notified by email.
     * </p>
     *
     * @param enableDomainTransferLockRequest
     *        A request to set the transfer lock for the specified domain.
     * @return A Java Future containing the result of the EnableDomainTransferLock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>DuplicateRequestException The request is already in progress for the domain.</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.EnableDomainTransferLock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/EnableDomainTransferLock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableDomainTransferLockResponse> enableDomainTransferLock(
            EnableDomainTransferLockRequest enableDomainTransferLockRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableDomainTransferLockResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableDomainTransferLockResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableDomainTransferLockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableDomainTransferLockRequest, EnableDomainTransferLockResponse>()
                            .withOperationName("EnableDomainTransferLock")
                            .withMarshaller(new EnableDomainTransferLockRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(enableDomainTransferLockRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For operations that require confirmation that the email address for the registrant contact is valid, such as
     * registering a new domain, this operation returns information about whether the registrant contact has responded.
     * </p>
     * <p>
     * If you want us to resend the email, use the <code>ResendContactReachabilityEmail</code> operation.
     * </p>
     *
     * @param getContactReachabilityStatusRequest
     * @return A Java Future containing the result of the GetContactReachabilityStatus operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.GetContactReachabilityStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetContactReachabilityStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetContactReachabilityStatusResponse> getContactReachabilityStatus(
            GetContactReachabilityStatusRequest getContactReachabilityStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContactReachabilityStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetContactReachabilityStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContactReachabilityStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContactReachabilityStatusRequest, GetContactReachabilityStatusResponse>()
                            .withOperationName("GetContactReachabilityStatus")
                            .withMarshaller(new GetContactReachabilityStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getContactReachabilityStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns detailed information about a specified domain that is associated with the current AWS
     * account. Contact information for the domain is also returned as part of the output.
     * </p>
     *
     * @param getDomainDetailRequest
     *        The GetDomainDetail request includes the following element.
     * @return A Java Future containing the result of the GetDomainDetail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.GetDomainDetail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetDomainDetail" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainDetailResponse> getDomainDetail(GetDomainDetailRequest getDomainDetailRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainDetailResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainDetailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainDetailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainDetailRequest, GetDomainDetailResponse>()
                            .withOperationName("GetDomainDetail")
                            .withMarshaller(new GetDomainDetailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDomainDetailRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The GetDomainSuggestions operation returns a list of suggested domain names given a string, which can either be a
     * domain name or simply a word or phrase (without spaces).
     * </p>
     *
     * @param getDomainSuggestionsRequest
     * @return A Java Future containing the result of the GetDomainSuggestions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.GetDomainSuggestions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetDomainSuggestions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainSuggestionsResponse> getDomainSuggestions(
            GetDomainSuggestionsRequest getDomainSuggestionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainSuggestionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainSuggestionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainSuggestionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainSuggestionsRequest, GetDomainSuggestionsResponse>()
                            .withOperationName("GetDomainSuggestions")
                            .withMarshaller(new GetDomainSuggestionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDomainSuggestionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns the current status of an operation that is not completed.
     * </p>
     *
     * @param getOperationDetailRequest
     *        The <a>GetOperationDetail</a> request includes the following element.
     * @return A Java Future containing the result of the GetOperationDetail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.GetOperationDetail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetOperationDetail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOperationDetailResponse> getOperationDetail(GetOperationDetailRequest getOperationDetailRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOperationDetailResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOperationDetailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOperationDetailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOperationDetailRequest, GetOperationDetailResponse>()
                            .withOperationName("GetOperationDetail")
                            .withMarshaller(new GetOperationDetailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getOperationDetailRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns all the domain names registered with Amazon Route 53 for the current AWS account.
     * </p>
     *
     * @param listDomainsRequest
     *        The ListDomains request includes the following elements.
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDomainsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns all the domain names registered with Amazon Route 53 for the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDomains(software.amazon.awssdk.services.route53domains.model.ListDomainsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53domains.model.ListDomainsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53domains.model.ListDomainsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.route53domains.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     *        The ListDomains request includes the following elements.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDomainsPublisher listDomainsPaginator(ListDomainsRequest listDomainsRequest) {
        return new ListDomainsPublisher(this, applyPaginatorUserAgent(listDomainsRequest));
    }

    /**
     * <p>
     * This operation returns the operation IDs of operations that are not yet complete.
     * </p>
     *
     * @param listOperationsRequest
     *        The ListOperations request includes the following elements.
     * @return A Java Future containing the result of the ListOperations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOperationsResponse> listOperations(ListOperationsRequest listOperationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                            .withOperationName("ListOperations")
                            .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listOperationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns the operation IDs of operations that are not yet complete.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOperations(software.amazon.awssdk.services.route53domains.model.ListOperationsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListOperationsPublisher publisher = client.listOperationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListOperationsPublisher publisher = client.listOperationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53domains.model.ListOperationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53domains.model.ListOperationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOperations(software.amazon.awssdk.services.route53domains.model.ListOperationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOperationsRequest
     *        The ListOperations request includes the following elements.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListOperations" target="_top">AWS
     *      API Documentation</a>
     */
    public ListOperationsPublisher listOperationsPaginator(ListOperationsRequest listOperationsRequest) {
        return new ListOperationsPublisher(this, applyPaginatorUserAgent(listOperationsRequest));
    }

    /**
     * <p>
     * This operation returns all of the tags that are associated with the specified domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param listTagsForDomainRequest
     *        The ListTagsForDomainRequest includes the following elements.
     * @return A Java Future containing the result of the ListTagsForDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.ListTagsForDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForDomainResponse> listTagsForDomain(ListTagsForDomainRequest listTagsForDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForDomainRequest, ListTagsForDomainResponse>()
                            .withOperationName("ListTagsForDomain")
                            .withMarshaller(new ListTagsForDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation registers a domain. Domains are registered either by Amazon Registrar (for .com, .net, and .org
     * domains) or by our registrar associate, Gandi (for all other domains). For some top-level domains (TLDs), this
     * operation requires extra parameters.
     * </p>
     * <p>
     * When you register a domain, Amazon Route 53 does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a Amazon Route 53 hosted zone that has the same name as the domain. Amazon Route 53 assigns four name
     * servers to your hosted zone and automatically updates your domain registration with the names of these name
     * servers.
     * </p>
     * </li>
     * <li>
     * <p>
     * Enables autorenew, so your domain registration will renew automatically each year. We'll notify you in advance of
     * the renewal date so you can choose whether to renew the registration.
     * </p>
     * </li>
     * <li>
     * <p>
     * Optionally enables privacy protection, so WHOIS queries return contact information either for Amazon Registrar
     * (for .com, .net, and .org domains) or for our registrar associate, Gandi (for all other TLDs). If you don't
     * enable privacy protection, WHOIS queries return the information that you entered for the registrant, admin, and
     * tech contacts.
     * </p>
     * </li>
     * <li>
     * <p>
     * If registration is successful, returns an operation ID that you can use to track the progress and completion of
     * the action. If the request is not completed successfully, the domain registrant is notified by email.
     * </p>
     * </li>
     * <li>
     * <p>
     * Charges your AWS account an amount based on the top-level domain. For more information, see <a
     * href="http://aws.amazon.com/route53/pricing/">Amazon Route 53 Pricing</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param registerDomainRequest
     *        The RegisterDomain request includes the following elements.
     * @return A Java Future containing the result of the RegisterDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>DuplicateRequestException The request is already in progress for the domain.</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>DomainLimitExceededException The number of domains has exceeded the allowed threshold for the
     *         account.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.RegisterDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RegisterDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterDomainResponse> registerDomain(RegisterDomainRequest registerDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterDomainRequest, RegisterDomainResponse>()
                            .withOperationName("RegisterDomain")
                            .withMarshaller(new RegisterDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(registerDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation renews a domain for the specified number of years. The cost of renewing your domain is billed to
     * your AWS account.
     * </p>
     * <p>
     * We recommend that you renew your domain several weeks before the expiration date. Some TLD registries delete
     * domains before the expiration date if you haven't renewed far enough in advance. For more information about
     * renewing domain registration, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-renew.html">Renewing Registration for a
     * Domain</a> in the Amazon Route 53 Developer Guide.
     * </p>
     *
     * @param renewDomainRequest
     *        A <code>RenewDomain</code> request includes the number of years that you want to renew for and the current
     *        expiration year.
     * @return A Java Future containing the result of the RenewDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>DuplicateRequestException The request is already in progress for the domain.</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.RenewDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RenewDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RenewDomainResponse> renewDomain(RenewDomainRequest renewDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RenewDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RenewDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RenewDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RenewDomainRequest, RenewDomainResponse>()
                            .withOperationName("RenewDomain").withMarshaller(new RenewDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(renewDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For operations that require confirmation that the email address for the registrant contact is valid, such as
     * registering a new domain, this operation resends the confirmation email to the current email address for the
     * registrant contact.
     * </p>
     *
     * @param resendContactReachabilityEmailRequest
     * @return A Java Future containing the result of the ResendContactReachabilityEmail operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.ResendContactReachabilityEmail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ResendContactReachabilityEmail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ResendContactReachabilityEmailResponse> resendContactReachabilityEmail(
            ResendContactReachabilityEmailRequest resendContactReachabilityEmailRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResendContactReachabilityEmailResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResendContactReachabilityEmailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResendContactReachabilityEmailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResendContactReachabilityEmailRequest, ResendContactReachabilityEmailResponse>()
                            .withOperationName("ResendContactReachabilityEmail")
                            .withMarshaller(new ResendContactReachabilityEmailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(resendContactReachabilityEmailRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns the AuthCode for the domain. To transfer a domain to another registrar, you provide this
     * value to the new registrar.
     * </p>
     *
     * @param retrieveDomainAuthCodeRequest
     *        A request for the authorization code for the specified domain. To transfer a domain to another registrar,
     *        you provide this value to the new registrar.
     * @return A Java Future containing the result of the RetrieveDomainAuthCode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.RetrieveDomainAuthCode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RetrieveDomainAuthCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RetrieveDomainAuthCodeResponse> retrieveDomainAuthCode(
            RetrieveDomainAuthCodeRequest retrieveDomainAuthCodeRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetrieveDomainAuthCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RetrieveDomainAuthCodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RetrieveDomainAuthCodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetrieveDomainAuthCodeRequest, RetrieveDomainAuthCodeResponse>()
                            .withOperationName("RetrieveDomainAuthCode")
                            .withMarshaller(new RetrieveDomainAuthCodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(retrieveDomainAuthCodeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation transfers a domain from another registrar to Amazon Route 53. When the transfer is complete, the
     * domain is registered either with Amazon Registrar (for .com, .net, and .org domains) or with our registrar
     * associate, Gandi (for all other TLDs).
     * </p>
     * <p>
     * For transfer requirements, a detailed procedure, and information about viewing the status of a domain transfer,
     * see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-transfer-to-route-53.html">Transferring
     * Registration for a Domain to Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the registrar for your domain is also the DNS service provider for the domain, we highly recommend that you
     * consider transferring your DNS service to Amazon Route 53 or to another DNS service provider before you transfer
     * your registration. Some registrars provide free DNS service when you purchase a domain registration. When you
     * transfer the registration, the previous registrar will not renew your domain registration and could end your DNS
     * service at any time.
     * </p>
     * <important>
     * <p>
     * If the registrar for your domain is also the DNS service provider for the domain and you don't transfer DNS
     * service to another provider, your website, email, and the web applications associated with the domain might
     * become unavailable.
     * </p>
     * </important>
     * <p>
     * If the transfer is successful, this method returns an operation ID that you can use to track the progress and
     * completion of the action. If the transfer doesn't complete successfully, the domain registrant will be notified
     * by email.
     * </p>
     *
     * @param transferDomainRequest
     *        The TransferDomain request includes the following elements.
     * @return A Java Future containing the result of the TransferDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>DuplicateRequestException The request is already in progress for the domain.</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>DomainLimitExceededException The number of domains has exceeded the allowed threshold for the
     *         account.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.TransferDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/TransferDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TransferDomainResponse> transferDomain(TransferDomainRequest transferDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TransferDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TransferDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TransferDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TransferDomainRequest, TransferDomainResponse>()
                            .withOperationName("TransferDomain")
                            .withMarshaller(new TransferDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(transferDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation updates the contact information for a particular domain. You must specify information for at least
     * one contact: registrant, administrator, or technical.
     * </p>
     * <p>
     * If the update is successful, this method returns an operation ID that you can use to track the progress and
     * completion of the action. If the request is not completed successfully, the domain registrant will be notified by
     * email.
     * </p>
     *
     * @param updateDomainContactRequest
     *        The UpdateDomainContact request includes the following elements.
     * @return A Java Future containing the result of the UpdateDomainContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>DuplicateRequestException The request is already in progress for the domain.</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.UpdateDomainContact
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainContactResponse> updateDomainContact(
            UpdateDomainContactRequest updateDomainContactRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainContactRequest, UpdateDomainContactResponse>()
                            .withOperationName("UpdateDomainContact")
                            .withMarshaller(new UpdateDomainContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDomainContactRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation updates the specified domain contact's privacy setting. When privacy protection is enabled,
     * contact information such as email address is replaced either with contact information for Amazon Registrar (for
     * .com, .net, and .org domains) or with contact information for our registrar associate, Gandi.
     * </p>
     * <p>
     * This operation affects only the contact information for the specified contact type (registrant, administrator, or
     * tech). If the request succeeds, Amazon Route 53 returns an operation ID that you can use with
     * <a>GetOperationDetail</a> to track the progress and completion of the action. If the request doesn't complete
     * successfully, the domain registrant will be notified by email.
     * </p>
     *
     * @param updateDomainContactPrivacyRequest
     *        The UpdateDomainContactPrivacy request includes the following elements.
     * @return A Java Future containing the result of the UpdateDomainContactPrivacy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>DuplicateRequestException The request is already in progress for the domain.</li>
     *         <li>TldRulesViolationException The top-level domain does not support this operation.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.UpdateDomainContactPrivacy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainContactPrivacy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainContactPrivacyResponse> updateDomainContactPrivacy(
            UpdateDomainContactPrivacyRequest updateDomainContactPrivacyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainContactPrivacyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainContactPrivacyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainContactPrivacyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainContactPrivacyRequest, UpdateDomainContactPrivacyResponse>()
                            .withOperationName("UpdateDomainContactPrivacy")
                            .withMarshaller(new UpdateDomainContactPrivacyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDomainContactPrivacyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation replaces the current set of name servers for the domain with the specified set of name servers. If
     * you use Amazon Route 53 as your DNS service, specify the four name servers in the delegation set for the hosted
     * zone for the domain.
     * </p>
     * <p>
     * If successful, this operation returns an operation ID that you can use to track the progress and completion of
     * the action. If the request is not completed successfully, the domain registrant will be notified by email.
     * </p>
     *
     * @param updateDomainNameserversRequest
     *        Replaces the current set of name servers for the domain with the specified set of name servers. If you use
     *        Amazon Route 53 as your DNS service, specify the four name servers in the delegation set for the hosted
     *        zone for the domain.</p>
     *        <p>
     *        If successful, this operation returns an operation ID that you can use to track the progress and
     *        completion of the action. If the request is not completed successfully, the domain registrant will be
     *        notified by email.
     * @return A Java Future containing the result of the UpdateDomainNameservers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li> <li>DuplicateRequestException The request is already
     *         in progress for the domain.</li> <li>TldRulesViolationException The top-level domain does not support
     *         this operation.</li> <li>OperationLimitExceededException The number of operations or jobs running
     *         exceeded the allowed threshold for the account.</li> <li>UnsupportedTldException Amazon Route 53 does not
     *         support this top-level domain (TLD).</li> <li>SdkException Base class for all exceptions that can be
     *         thrown by the SDK (both service and client). Can be used for catch all scenarios.</li> <li>
     *         SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li> <li>Route53DomainsException Base class for all service exceptions. Unknown
     *         exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.UpdateDomainNameservers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainNameservers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainNameserversResponse> updateDomainNameservers(
            UpdateDomainNameserversRequest updateDomainNameserversRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainNameserversResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainNameserversResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainNameserversResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainNameserversRequest, UpdateDomainNameserversResponse>()
                            .withOperationName("UpdateDomainNameservers")
                            .withMarshaller(new UpdateDomainNameserversRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDomainNameserversRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation adds or updates tags for a specified domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param updateTagsForDomainRequest
     *        The UpdateTagsForDomainRequest includes the following elements.
     * @return A Java Future containing the result of the UpdateTagsForDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>OperationLimitExceededException The number of operations or jobs running exceeded the allowed
     *         threshold for the account.</li>
     *         <li>UnsupportedTldException Amazon Route 53 does not support this top-level domain (TLD).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.UpdateTagsForDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTagsForDomainResponse> updateTagsForDomain(
            UpdateTagsForDomainRequest updateTagsForDomainRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTagsForDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTagsForDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTagsForDomainRequest, UpdateTagsForDomainResponse>()
                            .withOperationName("UpdateTagsForDomain")
                            .withMarshaller(new UpdateTagsForDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateTagsForDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all the domain-related billing records for the current AWS account for a specified period
     * </p>
     *
     * @param viewBillingRequest
     *        The ViewBilling request includes the following elements.
     * @return A Java Future containing the result of the ViewBilling operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The requested item is not acceptable. For example, for an OperationId it might
     *         refer to the ID of an operation that is already completed. For a domain name, it might not be a valid
     *         domain name or belong to the requester account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53DomainsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53DomainsAsyncClient.ViewBilling
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ViewBilling" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ViewBillingResponse> viewBilling(ViewBillingRequest viewBillingRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ViewBillingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ViewBillingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ViewBillingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ViewBillingRequest, ViewBillingResponse>()
                            .withOperationName("ViewBilling").withMarshaller(new ViewBillingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(viewBillingRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53DomainsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedTLD")
                                .exceptionBuilderSupplier(UnsupportedTldException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInput")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainLimitExceeded")
                                .exceptionBuilderSupplier(DomainLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TLDRulesViolation")
                                .exceptionBuilderSupplier(TldRulesViolationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationLimitExceeded")
                                .exceptionBuilderSupplier(OperationLimitExceededException::builder).build());
    }

    private <T extends Route53DomainsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
