/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53domains;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53domains.model.CheckDomainAvailabilityRequest;
import software.amazon.awssdk.services.route53domains.model.CheckDomainAvailabilityResponse;
import software.amazon.awssdk.services.route53domains.model.CheckDomainTransferabilityRequest;
import software.amazon.awssdk.services.route53domains.model.CheckDomainTransferabilityResponse;
import software.amazon.awssdk.services.route53domains.model.DeleteTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.DeleteTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.DisableDomainAutoRenewRequest;
import software.amazon.awssdk.services.route53domains.model.DisableDomainAutoRenewResponse;
import software.amazon.awssdk.services.route53domains.model.DisableDomainTransferLockRequest;
import software.amazon.awssdk.services.route53domains.model.DisableDomainTransferLockResponse;
import software.amazon.awssdk.services.route53domains.model.DomainLimitExceededException;
import software.amazon.awssdk.services.route53domains.model.DuplicateRequestException;
import software.amazon.awssdk.services.route53domains.model.EnableDomainAutoRenewRequest;
import software.amazon.awssdk.services.route53domains.model.EnableDomainAutoRenewResponse;
import software.amazon.awssdk.services.route53domains.model.EnableDomainTransferLockRequest;
import software.amazon.awssdk.services.route53domains.model.EnableDomainTransferLockResponse;
import software.amazon.awssdk.services.route53domains.model.GetContactReachabilityStatusRequest;
import software.amazon.awssdk.services.route53domains.model.GetContactReachabilityStatusResponse;
import software.amazon.awssdk.services.route53domains.model.GetDomainDetailRequest;
import software.amazon.awssdk.services.route53domains.model.GetDomainDetailResponse;
import software.amazon.awssdk.services.route53domains.model.GetDomainSuggestionsRequest;
import software.amazon.awssdk.services.route53domains.model.GetDomainSuggestionsResponse;
import software.amazon.awssdk.services.route53domains.model.GetOperationDetailRequest;
import software.amazon.awssdk.services.route53domains.model.GetOperationDetailResponse;
import software.amazon.awssdk.services.route53domains.model.InvalidInputException;
import software.amazon.awssdk.services.route53domains.model.ListDomainsRequest;
import software.amazon.awssdk.services.route53domains.model.ListDomainsResponse;
import software.amazon.awssdk.services.route53domains.model.ListOperationsRequest;
import software.amazon.awssdk.services.route53domains.model.ListOperationsResponse;
import software.amazon.awssdk.services.route53domains.model.ListTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.ListTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.OperationLimitExceededException;
import software.amazon.awssdk.services.route53domains.model.RegisterDomainRequest;
import software.amazon.awssdk.services.route53domains.model.RegisterDomainResponse;
import software.amazon.awssdk.services.route53domains.model.RenewDomainRequest;
import software.amazon.awssdk.services.route53domains.model.RenewDomainResponse;
import software.amazon.awssdk.services.route53domains.model.ResendContactReachabilityEmailRequest;
import software.amazon.awssdk.services.route53domains.model.ResendContactReachabilityEmailResponse;
import software.amazon.awssdk.services.route53domains.model.RetrieveDomainAuthCodeRequest;
import software.amazon.awssdk.services.route53domains.model.RetrieveDomainAuthCodeResponse;
import software.amazon.awssdk.services.route53domains.model.Route53DomainsException;
import software.amazon.awssdk.services.route53domains.model.Route53DomainsRequest;
import software.amazon.awssdk.services.route53domains.model.TldRulesViolationException;
import software.amazon.awssdk.services.route53domains.model.TransferDomainRequest;
import software.amazon.awssdk.services.route53domains.model.TransferDomainResponse;
import software.amazon.awssdk.services.route53domains.model.UnsupportedTldException;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactPrivacyRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactPrivacyResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainContactResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainNameserversRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateDomainNameserversResponse;
import software.amazon.awssdk.services.route53domains.model.UpdateTagsForDomainRequest;
import software.amazon.awssdk.services.route53domains.model.UpdateTagsForDomainResponse;
import software.amazon.awssdk.services.route53domains.model.ViewBillingRequest;
import software.amazon.awssdk.services.route53domains.model.ViewBillingResponse;
import software.amazon.awssdk.services.route53domains.paginators.ListDomainsIterable;
import software.amazon.awssdk.services.route53domains.paginators.ListOperationsIterable;
import software.amazon.awssdk.services.route53domains.transform.CheckDomainAvailabilityRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.CheckDomainTransferabilityRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DeleteTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DisableDomainAutoRenewRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.DisableDomainTransferLockRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.EnableDomainAutoRenewRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.EnableDomainTransferLockRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetContactReachabilityStatusRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetDomainDetailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetDomainSuggestionsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.GetOperationDetailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ListTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RegisterDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RenewDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ResendContactReachabilityEmailRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.RetrieveDomainAuthCodeRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.TransferDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainContactPrivacyRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainContactRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateDomainNameserversRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.UpdateTagsForDomainRequestMarshaller;
import software.amazon.awssdk.services.route53domains.transform.ViewBillingRequestMarshaller;

/**
 * Internal implementation of {@link Route53DomainsClient}.
 *
 * @see Route53DomainsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53DomainsClient implements Route53DomainsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53DomainsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * This operation checks the availability of one domain name. Note that if the availability status of a domain is
     * pending, you must submit another request to determine the availability of the domain name.
     * </p>
     *
     * @param checkDomainAvailabilityRequest
     *        The CheckDomainAvailability request contains the following elements.
     * @return Result of the CheckDomainAvailability operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.CheckDomainAvailability
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/CheckDomainAvailability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CheckDomainAvailabilityResponse checkDomainAvailability(CheckDomainAvailabilityRequest checkDomainAvailabilityRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CheckDomainAvailabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CheckDomainAvailabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CheckDomainAvailabilityRequest, CheckDomainAvailabilityResponse>()
                .withOperationName("CheckDomainAvailability").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(checkDomainAvailabilityRequest)
                .withMarshaller(new CheckDomainAvailabilityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Checks whether a domain name can be transferred to Amazon Route 53.
     * </p>
     *
     * @param checkDomainTransferabilityRequest
     *        The CheckDomainTransferability request contains the following elements.
     * @return Result of the CheckDomainTransferability operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.CheckDomainTransferability
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/CheckDomainTransferability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CheckDomainTransferabilityResponse checkDomainTransferability(
            CheckDomainTransferabilityRequest checkDomainTransferabilityRequest) throws InvalidInputException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CheckDomainTransferabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CheckDomainTransferabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CheckDomainTransferabilityRequest, CheckDomainTransferabilityResponse>()
                        .withOperationName("CheckDomainTransferability").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(checkDomainTransferabilityRequest)
                        .withMarshaller(new CheckDomainTransferabilityRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation deletes the specified tags for a domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param deleteTagsForDomainRequest
     *        The DeleteTagsForDomainRequest includes the following elements.
     * @return Result of the DeleteTagsForDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DeleteTagsForDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DeleteTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTagsForDomainResponse deleteTagsForDomain(DeleteTagsForDomainRequest deleteTagsForDomainRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteTagsForDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteTagsForDomainRequest, DeleteTagsForDomainResponse>()
                .withOperationName("DeleteTagsForDomain").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteTagsForDomainRequest)
                .withMarshaller(new DeleteTagsForDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation disables automatic renewal of domain registration for the specified domain.
     * </p>
     *
     * @param disableDomainAutoRenewRequest
     * @return Result of the DisableDomainAutoRenew operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DisableDomainAutoRenew
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DisableDomainAutoRenew"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableDomainAutoRenewResponse disableDomainAutoRenew(DisableDomainAutoRenewRequest disableDomainAutoRenewRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableDomainAutoRenewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableDomainAutoRenewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisableDomainAutoRenewRequest, DisableDomainAutoRenewResponse>()
                .withOperationName("DisableDomainAutoRenew").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disableDomainAutoRenewRequest)
                .withMarshaller(new DisableDomainAutoRenewRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation removes the transfer lock on the domain (specifically the <code>clientTransferProhibited</code>
     * status) to allow domain transfers. We recommend you refrain from performing this action unless you intend to
     * transfer the domain to a different registrar. Successful submission returns an operation ID that you can use to
     * track the progress and completion of the action. If the request is not completed successfully, the domain
     * registrant will be notified by email.
     * </p>
     *
     * @param disableDomainTransferLockRequest
     *        The DisableDomainTransferLock request includes the following element.
     * @return Result of the DisableDomainTransferLock operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.DisableDomainTransferLock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/DisableDomainTransferLock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableDomainTransferLockResponse disableDomainTransferLock(
            DisableDomainTransferLockRequest disableDomainTransferLockRequest) throws InvalidInputException,
            DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException, UnsupportedTldException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableDomainTransferLockResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableDomainTransferLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisableDomainTransferLockRequest, DisableDomainTransferLockResponse>()
                        .withOperationName("DisableDomainTransferLock").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(disableDomainTransferLockRequest)
                        .withMarshaller(new DisableDomainTransferLockRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation configures Amazon Route 53 to automatically renew the specified domain before the domain
     * registration expires. The cost of renewing your domain registration is billed to your AWS account.
     * </p>
     * <p>
     * The period during which you can renew a domain name varies by TLD. For a list of TLDs and their renewal policies,
     * see <a href="http://wiki.gandi.net/en/domains/renew#renewal_restoration_and_deletion_times">
     * "Renewal, restoration, and deletion times"</a> on the website for our registrar associate, Gandi. Amazon Route 53
     * requires that you renew before the end of the renewal period that is listed on the Gandi website so we can
     * complete processing before the deadline.
     * </p>
     *
     * @param enableDomainAutoRenewRequest
     * @return Result of the EnableDomainAutoRenew operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.EnableDomainAutoRenew
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/EnableDomainAutoRenew"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableDomainAutoRenewResponse enableDomainAutoRenew(EnableDomainAutoRenewRequest enableDomainAutoRenewRequest)
            throws InvalidInputException, UnsupportedTldException, TldRulesViolationException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableDomainAutoRenewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableDomainAutoRenewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<EnableDomainAutoRenewRequest, EnableDomainAutoRenewResponse>()
                .withOperationName("EnableDomainAutoRenew").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(enableDomainAutoRenewRequest)
                .withMarshaller(new EnableDomainAutoRenewRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation sets the transfer lock on the domain (specifically the <code>clientTransferProhibited</code>
     * status) to prevent domain transfers. Successful submission returns an operation ID that you can use to track the
     * progress and completion of the action. If the request is not completed successfully, the domain registrant will
     * be notified by email.
     * </p>
     *
     * @param enableDomainTransferLockRequest
     *        A request to set the transfer lock for the specified domain.
     * @return Result of the EnableDomainTransferLock operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.EnableDomainTransferLock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/EnableDomainTransferLock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableDomainTransferLockResponse enableDomainTransferLock(
            EnableDomainTransferLockRequest enableDomainTransferLockRequest) throws InvalidInputException,
            DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException, UnsupportedTldException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableDomainTransferLockResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableDomainTransferLockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<EnableDomainTransferLockRequest, EnableDomainTransferLockResponse>()
                        .withOperationName("EnableDomainTransferLock").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(enableDomainTransferLockRequest)
                        .withMarshaller(new EnableDomainTransferLockRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For operations that require confirmation that the email address for the registrant contact is valid, such as
     * registering a new domain, this operation returns information about whether the registrant contact has responded.
     * </p>
     * <p>
     * If you want us to resend the email, use the <code>ResendContactReachabilityEmail</code> operation.
     * </p>
     *
     * @param getContactReachabilityStatusRequest
     * @return Result of the GetContactReachabilityStatus operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetContactReachabilityStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetContactReachabilityStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetContactReachabilityStatusResponse getContactReachabilityStatus(
            GetContactReachabilityStatusRequest getContactReachabilityStatusRequest) throws InvalidInputException,
            OperationLimitExceededException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactReachabilityStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetContactReachabilityStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetContactReachabilityStatusRequest, GetContactReachabilityStatusResponse>()
                        .withOperationName("GetContactReachabilityStatus").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getContactReachabilityStatusRequest)
                        .withMarshaller(new GetContactReachabilityStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation returns detailed information about a specified domain that is associated with the current AWS
     * account. Contact information for the domain is also returned as part of the output.
     * </p>
     *
     * @param getDomainDetailRequest
     *        The GetDomainDetail request includes the following element.
     * @return Result of the GetDomainDetail operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetDomainDetail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetDomainDetail" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDomainDetailResponse getDomainDetail(GetDomainDetailRequest getDomainDetailRequest) throws InvalidInputException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDomainDetailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDomainDetailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDomainDetailRequest, GetDomainDetailResponse>()
                .withOperationName("GetDomainDetail").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDomainDetailRequest)
                .withMarshaller(new GetDomainDetailRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The GetDomainSuggestions operation returns a list of suggested domain names given a string, which can either be a
     * domain name or simply a word or phrase (without spaces).
     * </p>
     *
     * @param getDomainSuggestionsRequest
     * @return Result of the GetDomainSuggestions operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetDomainSuggestions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetDomainSuggestions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDomainSuggestionsResponse getDomainSuggestions(GetDomainSuggestionsRequest getDomainSuggestionsRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDomainSuggestionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDomainSuggestionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDomainSuggestionsRequest, GetDomainSuggestionsResponse>()
                .withOperationName("GetDomainSuggestions").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDomainSuggestionsRequest)
                .withMarshaller(new GetDomainSuggestionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation returns the current status of an operation that is not completed.
     * </p>
     *
     * @param getOperationDetailRequest
     *        The <a>GetOperationDetail</a> request includes the following element.
     * @return Result of the GetOperationDetail operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.GetOperationDetail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/GetOperationDetail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOperationDetailResponse getOperationDetail(GetOperationDetailRequest getOperationDetailRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOperationDetailResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetOperationDetailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetOperationDetailRequest, GetOperationDetailResponse>()
                .withOperationName("GetOperationDetail").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getOperationDetailRequest)
                .withMarshaller(new GetOperationDetailRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation returns all the domain names registered with Amazon Route 53 for the current AWS account.
     * </p>
     *
     * @param listDomainsRequest
     *        The ListDomains request includes the following elements.
     * @return Result of the ListDomains operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDomainsResponse listDomains(ListDomainsRequest listDomainsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDomainsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                .withOperationName("ListDomains").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDomainsRequest)
                .withMarshaller(new ListDomainsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation returns all the domain names registered with Amazon Route 53 for the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDomains(software.amazon.awssdk.services.route53domains.model.ListDomainsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53domains.paginators.ListDomainsIterable responses = client
     *             .listDomainsPaginator(request);
     *     for (software.amazon.awssdk.services.route53domains.model.ListDomainsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.route53domains.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     *        The ListDomains request includes the following elements.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDomainsIterable listDomainsPaginator(ListDomainsRequest listDomainsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        return new ListDomainsIterable(this, applyPaginatorUserAgent(listDomainsRequest));
    }

    /**
     * <p>
     * This operation returns the operation IDs of operations that are not yet complete.
     * </p>
     *
     * @param listOperationsRequest
     *        The ListOperations request includes the following elements.
     * @return Result of the ListOperations operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListOperationsResponse listOperations(ListOperationsRequest listOperationsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                .withOperationName("ListOperations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listOperationsRequest)
                .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation returns the operation IDs of operations that are not yet complete.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOperations(software.amazon.awssdk.services.route53domains.model.ListOperationsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListOperationsIterable responses = client.listOperationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53domains.paginators.ListOperationsIterable responses = client
     *             .listOperationsPaginator(request);
     *     for (software.amazon.awssdk.services.route53domains.model.ListOperationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53domains.paginators.ListOperationsIterable responses = client.listOperationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOperations(software.amazon.awssdk.services.route53domains.model.ListOperationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOperationsRequest
     *        The ListOperations request includes the following elements.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListOperationsIterable listOperationsPaginator(ListOperationsRequest listOperationsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, Route53DomainsException {
        return new ListOperationsIterable(this, applyPaginatorUserAgent(listOperationsRequest));
    }

    /**
     * <p>
     * This operation returns all of the tags that are associated with the specified domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param listTagsForDomainRequest
     *        The ListTagsForDomainRequest includes the following elements.
     * @return Result of the ListTagsForDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ListTagsForDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ListTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForDomainResponse listTagsForDomain(ListTagsForDomainRequest listTagsForDomainRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsForDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForDomainRequest, ListTagsForDomainResponse>()
                .withOperationName("ListTagsForDomain").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForDomainRequest)
                .withMarshaller(new ListTagsForDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation registers a domain. Domains are registered either by Amazon Registrar (for .com, .net, and .org
     * domains) or by our registrar associate, Gandi (for all other domains). For some top-level domains (TLDs), this
     * operation requires extra parameters.
     * </p>
     * <p>
     * When you register a domain, Amazon Route 53 does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a Amazon Route 53 hosted zone that has the same name as the domain. Amazon Route 53 assigns four name
     * servers to your hosted zone and automatically updates your domain registration with the names of these name
     * servers.
     * </p>
     * </li>
     * <li>
     * <p>
     * Enables autorenew, so your domain registration will renew automatically each year. We'll notify you in advance of
     * the renewal date so you can choose whether to renew the registration.
     * </p>
     * </li>
     * <li>
     * <p>
     * Optionally enables privacy protection, so WHOIS queries return contact information either for Amazon Registrar
     * (for .com, .net, and .org domains) or for our registrar associate, Gandi (for all other TLDs). If you don't
     * enable privacy protection, WHOIS queries return the information that you entered for the registrant, admin, and
     * tech contacts.
     * </p>
     * </li>
     * <li>
     * <p>
     * If registration is successful, returns an operation ID that you can use to track the progress and completion of
     * the action. If the request is not completed successfully, the domain registrant is notified by email.
     * </p>
     * </li>
     * <li>
     * <p>
     * Charges your AWS account an amount based on the top-level domain. For more information, see <a
     * href="http://aws.amazon.com/route53/pricing/">Amazon Route 53 Pricing</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param registerDomainRequest
     *        The RegisterDomain request includes the following elements.
     * @return Result of the RegisterDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws DomainLimitExceededException
     *         The number of domains has exceeded the allowed threshold for the account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.RegisterDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RegisterDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterDomainResponse registerDomain(RegisterDomainRequest registerDomainRequest) throws InvalidInputException,
            UnsupportedTldException, DuplicateRequestException, TldRulesViolationException, DomainLimitExceededException,
            OperationLimitExceededException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RegisterDomainRequest, RegisterDomainResponse>()
                .withOperationName("RegisterDomain").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(registerDomainRequest)
                .withMarshaller(new RegisterDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation renews a domain for the specified number of years. The cost of renewing your domain is billed to
     * your AWS account.
     * </p>
     * <p>
     * We recommend that you renew your domain several weeks before the expiration date. Some TLD registries delete
     * domains before the expiration date if you haven't renewed far enough in advance. For more information about
     * renewing domain registration, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-renew.html">Renewing Registration for a
     * Domain</a> in the Amazon Route 53 Developer Guide.
     * </p>
     *
     * @param renewDomainRequest
     *        A <code>RenewDomain</code> request includes the number of years that you want to renew for and the current
     *        expiration year.
     * @return Result of the RenewDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.RenewDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RenewDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RenewDomainResponse renewDomain(RenewDomainRequest renewDomainRequest) throws InvalidInputException,
            UnsupportedTldException, DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RenewDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RenewDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RenewDomainRequest, RenewDomainResponse>()
                .withOperationName("RenewDomain").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(renewDomainRequest)
                .withMarshaller(new RenewDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For operations that require confirmation that the email address for the registrant contact is valid, such as
     * registering a new domain, this operation resends the confirmation email to the current email address for the
     * registrant contact.
     * </p>
     *
     * @param resendContactReachabilityEmailRequest
     * @return Result of the ResendContactReachabilityEmail operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ResendContactReachabilityEmail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ResendContactReachabilityEmail"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResendContactReachabilityEmailResponse resendContactReachabilityEmail(
            ResendContactReachabilityEmailRequest resendContactReachabilityEmailRequest) throws InvalidInputException,
            OperationLimitExceededException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResendContactReachabilityEmailResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResendContactReachabilityEmailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ResendContactReachabilityEmailRequest, ResendContactReachabilityEmailResponse>()
                        .withOperationName("ResendContactReachabilityEmail").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(resendContactReachabilityEmailRequest)
                        .withMarshaller(new ResendContactReachabilityEmailRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation returns the AuthCode for the domain. To transfer a domain to another registrar, you provide this
     * value to the new registrar.
     * </p>
     *
     * @param retrieveDomainAuthCodeRequest
     *        A request for the authorization code for the specified domain. To transfer a domain to another registrar,
     *        you provide this value to the new registrar.
     * @return Result of the RetrieveDomainAuthCode operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.RetrieveDomainAuthCode
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/RetrieveDomainAuthCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RetrieveDomainAuthCodeResponse retrieveDomainAuthCode(RetrieveDomainAuthCodeRequest retrieveDomainAuthCodeRequest)
            throws InvalidInputException, UnsupportedTldException, AwsServiceException, SdkClientException,
            Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RetrieveDomainAuthCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RetrieveDomainAuthCodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RetrieveDomainAuthCodeRequest, RetrieveDomainAuthCodeResponse>()
                .withOperationName("RetrieveDomainAuthCode").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(retrieveDomainAuthCodeRequest)
                .withMarshaller(new RetrieveDomainAuthCodeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation transfers a domain from another registrar to Amazon Route 53. When the transfer is complete, the
     * domain is registered either with Amazon Registrar (for .com, .net, and .org domains) or with our registrar
     * associate, Gandi (for all other TLDs).
     * </p>
     * <p>
     * For transfer requirements, a detailed procedure, and information about viewing the status of a domain transfer,
     * see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/domain-transfer-to-route-53.html">Transferring
     * Registration for a Domain to Amazon Route 53</a> in the <i>Amazon Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the registrar for your domain is also the DNS service provider for the domain, we highly recommend that you
     * consider transferring your DNS service to Amazon Route 53 or to another DNS service provider before you transfer
     * your registration. Some registrars provide free DNS service when you purchase a domain registration. When you
     * transfer the registration, the previous registrar will not renew your domain registration and could end your DNS
     * service at any time.
     * </p>
     * <important>
     * <p>
     * If the registrar for your domain is also the DNS service provider for the domain and you don't transfer DNS
     * service to another provider, your website, email, and the web applications associated with the domain might
     * become unavailable.
     * </p>
     * </important>
     * <p>
     * If the transfer is successful, this method returns an operation ID that you can use to track the progress and
     * completion of the action. If the transfer doesn't complete successfully, the domain registrant will be notified
     * by email.
     * </p>
     *
     * @param transferDomainRequest
     *        The TransferDomain request includes the following elements.
     * @return Result of the TransferDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws DomainLimitExceededException
     *         The number of domains has exceeded the allowed threshold for the account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.TransferDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/TransferDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TransferDomainResponse transferDomain(TransferDomainRequest transferDomainRequest) throws InvalidInputException,
            UnsupportedTldException, DuplicateRequestException, TldRulesViolationException, DomainLimitExceededException,
            OperationLimitExceededException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TransferDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TransferDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TransferDomainRequest, TransferDomainResponse>()
                .withOperationName("TransferDomain").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(transferDomainRequest)
                .withMarshaller(new TransferDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation updates the contact information for a particular domain. You must specify information for at least
     * one contact: registrant, administrator, or technical.
     * </p>
     * <p>
     * If the update is successful, this method returns an operation ID that you can use to track the progress and
     * completion of the action. If the request is not completed successfully, the domain registrant will be notified by
     * email.
     * </p>
     *
     * @param updateDomainContactRequest
     *        The UpdateDomainContact request includes the following elements.
     * @return Result of the UpdateDomainContact operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateDomainContact
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainContactResponse updateDomainContact(UpdateDomainContactRequest updateDomainContactRequest)
            throws InvalidInputException, DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateDomainContactRequest, UpdateDomainContactResponse>()
                .withOperationName("UpdateDomainContact").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainContactRequest)
                .withMarshaller(new UpdateDomainContactRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation updates the specified domain contact's privacy setting. When privacy protection is enabled,
     * contact information such as email address is replaced either with contact information for Amazon Registrar (for
     * .com, .net, and .org domains) or with contact information for our registrar associate, Gandi.
     * </p>
     * <p>
     * This operation affects only the contact information for the specified contact type (registrant, administrator, or
     * tech). If the request succeeds, Amazon Route 53 returns an operation ID that you can use with
     * <a>GetOperationDetail</a> to track the progress and completion of the action. If the request doesn't complete
     * successfully, the domain registrant will be notified by email.
     * </p>
     *
     * @param updateDomainContactPrivacyRequest
     *        The UpdateDomainContactPrivacy request includes the following elements.
     * @return Result of the UpdateDomainContactPrivacy operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateDomainContactPrivacy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainContactPrivacy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainContactPrivacyResponse updateDomainContactPrivacy(
            UpdateDomainContactPrivacyRequest updateDomainContactPrivacyRequest) throws InvalidInputException,
            DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException, UnsupportedTldException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainContactPrivacyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainContactPrivacyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateDomainContactPrivacyRequest, UpdateDomainContactPrivacyResponse>()
                        .withOperationName("UpdateDomainContactPrivacy").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainContactPrivacyRequest)
                        .withMarshaller(new UpdateDomainContactPrivacyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation replaces the current set of name servers for the domain with the specified set of name servers. If
     * you use Amazon Route 53 as your DNS service, specify the four name servers in the delegation set for the hosted
     * zone for the domain.
     * </p>
     * <p>
     * If successful, this operation returns an operation ID that you can use to track the progress and completion of
     * the action. If the request is not completed successfully, the domain registrant will be notified by email.
     * </p>
     *
     * @param updateDomainNameserversRequest
     *        Replaces the current set of name servers for the domain with the specified set of name servers. If you use
     *        Amazon Route 53 as your DNS service, specify the four name servers in the delegation set for the hosted
     *        zone for the domain.</p>
     *        <p>
     *        If successful, this operation returns an operation ID that you can use to track the progress and
     *        completion of the action. If the request is not completed successfully, the domain registrant will be
     *        notified by email.
     * @return Result of the UpdateDomainNameservers operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws DuplicateRequestException
     *         The request is already in progress for the domain.
     * @throws TldRulesViolationException
     *         The top-level domain does not support this operation.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateDomainNameservers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateDomainNameservers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainNameserversResponse updateDomainNameservers(UpdateDomainNameserversRequest updateDomainNameserversRequest)
            throws InvalidInputException, DuplicateRequestException, TldRulesViolationException, OperationLimitExceededException,
            UnsupportedTldException, AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainNameserversResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainNameserversResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateDomainNameserversRequest, UpdateDomainNameserversResponse>()
                .withOperationName("UpdateDomainNameservers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainNameserversRequest)
                .withMarshaller(new UpdateDomainNameserversRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * This operation adds or updates tags for a specified domain.
     * </p>
     * <p>
     * All tag operations are eventually consistent; subsequent operations might not immediately represent all issued
     * operations.
     * </p>
     *
     * @param updateTagsForDomainRequest
     *        The UpdateTagsForDomainRequest includes the following elements.
     * @return Result of the UpdateTagsForDomain operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws OperationLimitExceededException
     *         The number of operations or jobs running exceeded the allowed threshold for the account.
     * @throws UnsupportedTldException
     *         Amazon Route 53 does not support this top-level domain (TLD).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.UpdateTagsForDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/UpdateTagsForDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateTagsForDomainResponse updateTagsForDomain(UpdateTagsForDomainRequest updateTagsForDomainRequest)
            throws InvalidInputException, OperationLimitExceededException, UnsupportedTldException, AwsServiceException,
            SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTagsForDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateTagsForDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateTagsForDomainRequest, UpdateTagsForDomainResponse>()
                .withOperationName("UpdateTagsForDomain").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateTagsForDomainRequest)
                .withMarshaller(new UpdateTagsForDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns all the domain-related billing records for the current AWS account for a specified period
     * </p>
     *
     * @param viewBillingRequest
     *        The ViewBilling request includes the following elements.
     * @return Result of the ViewBilling operation returned by the service.
     * @throws InvalidInputException
     *         The requested item is not acceptable. For example, for an OperationId it might refer to the ID of an
     *         operation that is already completed. For a domain name, it might not be a valid domain name or belong to
     *         the requester account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53DomainsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53DomainsClient.ViewBilling
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53domains-2014-05-15/ViewBilling" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ViewBillingResponse viewBilling(ViewBillingRequest viewBillingRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, Route53DomainsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ViewBillingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ViewBillingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ViewBillingRequest, ViewBillingResponse>()
                .withOperationName("ViewBilling").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(viewBillingRequest)
                .withMarshaller(new ViewBillingRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53DomainsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedTLD")
                                .exceptionBuilderSupplier(UnsupportedTldException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInput")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainLimitExceeded")
                                .exceptionBuilderSupplier(DomainLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TLDRulesViolation")
                                .exceptionBuilderSupplier(TldRulesViolationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationLimitExceeded")
                                .exceptionBuilderSupplier(OperationLimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends Route53DomainsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
