/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53domains.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Replaces the current set of name servers for the domain with the specified set of name servers. If you use Amazon
 * Route 53 as your DNS service, specify the four name servers in the delegation set for the hosted zone for the domain.
 * </p>
 * <p>
 * If successful, this operation returns an operation ID that you can use to track the progress and completion of the
 * action. If the request is not completed successfully, the domain registrant will be notified by email.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateDomainNameserversRequest extends Route53DomainsRequest implements
        ToCopyableBuilder<UpdateDomainNameserversRequest.Builder, UpdateDomainNameserversRequest> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateDomainNameserversRequest::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<String> FI_AUTH_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateDomainNameserversRequest::fiAuthKey)).setter(setter(Builder::fiAuthKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FIAuthKey").build()).build();

    private static final SdkField<List<Nameserver>> NAMESERVERS_FIELD = SdkField
            .<List<Nameserver>> builder(MarshallingType.LIST)
            .getter(getter(UpdateDomainNameserversRequest::nameservers))
            .setter(setter(Builder::nameservers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Nameservers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Nameserver> builder(MarshallingType.SDK_POJO)
                                            .constructor(Nameserver::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            FI_AUTH_KEY_FIELD, NAMESERVERS_FIELD));

    private final String domainName;

    private final String fiAuthKey;

    private final List<Nameserver> nameservers;

    private UpdateDomainNameserversRequest(BuilderImpl builder) {
        super(builder);
        this.domainName = builder.domainName;
        this.fiAuthKey = builder.fiAuthKey;
        this.nameservers = builder.nameservers;
    }

    /**
     * <p>
     * The name of the domain that you want to change name servers for.
     * </p>
     * 
     * @return The name of the domain that you want to change name servers for.
     */
    public String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The authorization key for .fi domains
     * </p>
     * 
     * @return The authorization key for .fi domains
     */
    public String fiAuthKey() {
        return fiAuthKey;
    }

    /**
     * Returns true if the Nameservers property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasNameservers() {
        return nameservers != null && !(nameservers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of new name servers for the domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNameservers()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of new name servers for the domain.
     */
    public List<Nameserver> nameservers() {
        return nameservers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(fiAuthKey());
        hashCode = 31 * hashCode + Objects.hashCode(nameservers());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateDomainNameserversRequest)) {
            return false;
        }
        UpdateDomainNameserversRequest other = (UpdateDomainNameserversRequest) obj;
        return Objects.equals(domainName(), other.domainName()) && Objects.equals(fiAuthKey(), other.fiAuthKey())
                && Objects.equals(nameservers(), other.nameservers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UpdateDomainNameserversRequest").add("DomainName", domainName()).add("FIAuthKey", fiAuthKey())
                .add("Nameservers", nameservers()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "FIAuthKey":
            return Optional.ofNullable(clazz.cast(fiAuthKey()));
        case "Nameservers":
            return Optional.ofNullable(clazz.cast(nameservers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateDomainNameserversRequest, T> g) {
        return obj -> g.apply((UpdateDomainNameserversRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Route53DomainsRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, UpdateDomainNameserversRequest> {
        /**
         * <p>
         * The name of the domain that you want to change name servers for.
         * </p>
         * 
         * @param domainName
         *        The name of the domain that you want to change name servers for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The authorization key for .fi domains
         * </p>
         * 
         * @param fiAuthKey
         *        The authorization key for .fi domains
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fiAuthKey(String fiAuthKey);

        /**
         * <p>
         * A list of new name servers for the domain.
         * </p>
         * 
         * @param nameservers
         *        A list of new name servers for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameservers(Collection<Nameserver> nameservers);

        /**
         * <p>
         * A list of new name servers for the domain.
         * </p>
         * 
         * @param nameservers
         *        A list of new name servers for the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameservers(Nameserver... nameservers);

        /**
         * <p>
         * A list of new name servers for the domain.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Nameserver>.Builder} avoiding the need to
         * create one manually via {@link List<Nameserver>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Nameserver>.Builder#build()} is called immediately and its
         * result is passed to {@link #nameservers(List<Nameserver>)}.
         * 
         * @param nameservers
         *        a consumer that will call methods on {@link List<Nameserver>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nameservers(List<Nameserver>)
         */
        Builder nameservers(Consumer<Nameserver.Builder>... nameservers);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Route53DomainsRequest.BuilderImpl implements Builder {
        private String domainName;

        private String fiAuthKey;

        private List<Nameserver> nameservers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateDomainNameserversRequest model) {
            super(model);
            domainName(model.domainName);
            fiAuthKey(model.fiAuthKey);
            nameservers(model.nameservers);
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getFiAuthKey() {
            return fiAuthKey;
        }

        @Override
        public final Builder fiAuthKey(String fiAuthKey) {
            this.fiAuthKey = fiAuthKey;
            return this;
        }

        public final void setFiAuthKey(String fiAuthKey) {
            this.fiAuthKey = fiAuthKey;
        }

        public final Collection<Nameserver.Builder> getNameservers() {
            return nameservers != null ? nameservers.stream().map(Nameserver::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder nameservers(Collection<Nameserver> nameservers) {
            this.nameservers = NameserverListCopier.copy(nameservers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nameservers(Nameserver... nameservers) {
            nameservers(Arrays.asList(nameservers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nameservers(Consumer<Nameserver.Builder>... nameservers) {
            nameservers(Stream.of(nameservers).map(c -> Nameserver.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNameservers(Collection<Nameserver.BuilderImpl> nameservers) {
            this.nameservers = NameserverListCopier.copyFromBuilder(nameservers);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateDomainNameserversRequest build() {
            return new UpdateDomainNameserversRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
