/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.sqs.batchmanager.SqsAsyncBatchManager;
import software.amazon.awssdk.services.sqs.endpoints.SqsClientContextParams;
import software.amazon.awssdk.services.sqs.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.sqs.internal.SqsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.sqs.model.AddPermissionRequest;
import software.amazon.awssdk.services.sqs.model.AddPermissionResponse;
import software.amazon.awssdk.services.sqs.model.BatchEntryIdsNotDistinctException;
import software.amazon.awssdk.services.sqs.model.BatchRequestTooLongException;
import software.amazon.awssdk.services.sqs.model.CancelMessageMoveTaskRequest;
import software.amazon.awssdk.services.sqs.model.CancelMessageMoveTaskResponse;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityBatchRequest;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityBatchResponse;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityRequest;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityResponse;
import software.amazon.awssdk.services.sqs.model.CreateQueueRequest;
import software.amazon.awssdk.services.sqs.model.CreateQueueResponse;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.DeleteMessageRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageResponse;
import software.amazon.awssdk.services.sqs.model.DeleteQueueRequest;
import software.amazon.awssdk.services.sqs.model.DeleteQueueResponse;
import software.amazon.awssdk.services.sqs.model.EmptyBatchRequestException;
import software.amazon.awssdk.services.sqs.model.GetQueueAttributesRequest;
import software.amazon.awssdk.services.sqs.model.GetQueueAttributesResponse;
import software.amazon.awssdk.services.sqs.model.GetQueueUrlRequest;
import software.amazon.awssdk.services.sqs.model.GetQueueUrlResponse;
import software.amazon.awssdk.services.sqs.model.InvalidAddressException;
import software.amazon.awssdk.services.sqs.model.InvalidAttributeNameException;
import software.amazon.awssdk.services.sqs.model.InvalidAttributeValueException;
import software.amazon.awssdk.services.sqs.model.InvalidBatchEntryIdException;
import software.amazon.awssdk.services.sqs.model.InvalidIdFormatException;
import software.amazon.awssdk.services.sqs.model.InvalidMessageContentsException;
import software.amazon.awssdk.services.sqs.model.InvalidSecurityException;
import software.amazon.awssdk.services.sqs.model.KmsAccessDeniedException;
import software.amazon.awssdk.services.sqs.model.KmsDisabledException;
import software.amazon.awssdk.services.sqs.model.KmsInvalidKeyUsageException;
import software.amazon.awssdk.services.sqs.model.KmsInvalidStateException;
import software.amazon.awssdk.services.sqs.model.KmsNotFoundException;
import software.amazon.awssdk.services.sqs.model.KmsOptInRequiredException;
import software.amazon.awssdk.services.sqs.model.KmsThrottledException;
import software.amazon.awssdk.services.sqs.model.ListDeadLetterSourceQueuesRequest;
import software.amazon.awssdk.services.sqs.model.ListDeadLetterSourceQueuesResponse;
import software.amazon.awssdk.services.sqs.model.ListMessageMoveTasksRequest;
import software.amazon.awssdk.services.sqs.model.ListMessageMoveTasksResponse;
import software.amazon.awssdk.services.sqs.model.ListQueueTagsRequest;
import software.amazon.awssdk.services.sqs.model.ListQueueTagsResponse;
import software.amazon.awssdk.services.sqs.model.ListQueuesRequest;
import software.amazon.awssdk.services.sqs.model.ListQueuesResponse;
import software.amazon.awssdk.services.sqs.model.MessageNotInflightException;
import software.amazon.awssdk.services.sqs.model.OverLimitException;
import software.amazon.awssdk.services.sqs.model.PurgeQueueInProgressException;
import software.amazon.awssdk.services.sqs.model.PurgeQueueRequest;
import software.amazon.awssdk.services.sqs.model.PurgeQueueResponse;
import software.amazon.awssdk.services.sqs.model.QueueDeletedRecentlyException;
import software.amazon.awssdk.services.sqs.model.QueueDoesNotExistException;
import software.amazon.awssdk.services.sqs.model.QueueNameExistsException;
import software.amazon.awssdk.services.sqs.model.ReceiptHandleIsInvalidException;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageRequest;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageResponse;
import software.amazon.awssdk.services.sqs.model.RemovePermissionRequest;
import software.amazon.awssdk.services.sqs.model.RemovePermissionResponse;
import software.amazon.awssdk.services.sqs.model.RequestThrottledException;
import software.amazon.awssdk.services.sqs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.SendMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.SendMessageRequest;
import software.amazon.awssdk.services.sqs.model.SendMessageResponse;
import software.amazon.awssdk.services.sqs.model.SetQueueAttributesRequest;
import software.amazon.awssdk.services.sqs.model.SetQueueAttributesResponse;
import software.amazon.awssdk.services.sqs.model.SqsException;
import software.amazon.awssdk.services.sqs.model.StartMessageMoveTaskRequest;
import software.amazon.awssdk.services.sqs.model.StartMessageMoveTaskResponse;
import software.amazon.awssdk.services.sqs.model.TagQueueRequest;
import software.amazon.awssdk.services.sqs.model.TagQueueResponse;
import software.amazon.awssdk.services.sqs.model.TooManyEntriesInBatchRequestException;
import software.amazon.awssdk.services.sqs.model.UnsupportedOperationException;
import software.amazon.awssdk.services.sqs.model.UntagQueueRequest;
import software.amazon.awssdk.services.sqs.model.UntagQueueResponse;
import software.amazon.awssdk.services.sqs.transform.AddPermissionRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.CancelMessageMoveTaskRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ChangeMessageVisibilityBatchRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ChangeMessageVisibilityRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.CreateQueueRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteMessageBatchRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteMessageRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.DeleteQueueRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.GetQueueAttributesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.GetQueueUrlRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ListDeadLetterSourceQueuesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ListMessageMoveTasksRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ListQueueTagsRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ListQueuesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.PurgeQueueRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.ReceiveMessageRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.SendMessageBatchRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.SendMessageRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.SetQueueAttributesRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.StartMessageMoveTaskRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.TagQueueRequestMarshaller;
import software.amazon.awssdk.services.sqs.transform.UntagQueueRequestMarshaller;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

/**
 * Internal implementation of {@link SqsAsyncClient}.
 *
 * @see SqsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSqsAsyncClient implements SqsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSqsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultSqsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "SQS" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Adds a permission to a queue for a specific <a
     * href="https://docs.aws.amazon.com/general/latest/gr/glos-chap.html#P">principal</a>. This allows sharing access
     * to the queue.
     * </p>
     * <p>
     * When you create a queue, you have full control access rights for the queue. Only you, the owner of the queue, can
     * grant or deny permissions to the queue. For more information about these permissions, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-writing-an-sqs-policy.html#write-messages-to-shared-queue"
     * >Allow Developers to Write Messages to a Shared Queue</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * <code>AddPermission</code> generates a policy for you. You can use <code> <a>SetQueueAttributes</a> </code> to
     * upload your policy. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-creating-custom-policies.html"
     * >Using Custom Policies with the Amazon SQS Access Policy Language</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * An Amazon SQS policy can have a maximum of seven actions per statement.
     * </p>
     * </li>
     * <li>
     * <p>
     * To remove the ability to change queue permissions, you must deny permission to the <code>AddPermission</code>,
     * <code>RemovePermission</code>, and <code>SetQueueAttributes</code> actions in your IAM policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SQS <code>AddPermission</code> does not support adding a non-account principal.
     * </p>
     * </li>
     * </ul>
     * </note> <note>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param addPermissionRequest
     * @return A Java Future containing the result of the AddPermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>OverLimitException The specified action violates a limit. For example, <code>ReceiveMessage</code>
     *         returns this error if the maximum number of in flight messages is reached and <code>AddPermission</code>
     *         returns this error if the maximum number of permissions for the queue is reached.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.AddPermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/AddPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddPermissionResponse> addPermission(AddPermissionRequest addPermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addPermissionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddPermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddPermissionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AddPermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddPermissionRequest, AddPermissionResponse>()
                            .withOperationName("AddPermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AddPermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(addPermissionRequest));
            CompletableFuture<AddPermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a specified message movement task. A message movement can only be cancelled when the current status is
     * RUNNING. Cancelling a message movement task does not revert the messages that have already been moved. It can
     * only stop the messages that have not been moved yet.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This action is currently limited to supporting message redrive from <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     * >dead-letter queues (DLQs)</a> only. In this context, the source queue is the dead-letter queue (DLQ), while the
     * destination queue can be the original source queue (from which the messages were driven to the
     * dead-letter-queue), or a custom destination queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Only one active message movement task is supported per queue at any given time.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param cancelMessageMoveTaskRequest
     * @return A Java Future containing the result of the CancelMessageMoveTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException One or more specified resources don't exist.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.CancelMessageMoveTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/CancelMessageMoveTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelMessageMoveTaskResponse> cancelMessageMoveTask(
            CancelMessageMoveTaskRequest cancelMessageMoveTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelMessageMoveTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelMessageMoveTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelMessageMoveTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelMessageMoveTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelMessageMoveTaskResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CancelMessageMoveTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelMessageMoveTaskRequest, CancelMessageMoveTaskResponse>()
                            .withOperationName("CancelMessageMoveTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelMessageMoveTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelMessageMoveTaskRequest));
            CompletableFuture<CancelMessageMoveTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the visibility timeout of a specified message in a queue to a new value. The default visibility timeout
     * for a message is 30 seconds. The minimum is 0 seconds. The maximum is 12 hours. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-visibility-timeout.html">
     * Visibility Timeout</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * For example, if the default timeout for a queue is 60 seconds, 15 seconds have elapsed since you received the
     * message, and you send a ChangeMessageVisibility call with <code>VisibilityTimeout</code> set to 10 seconds, the
     * 10 seconds begin to count from the time that you make the <code>ChangeMessageVisibility</code> call. Thus, any
     * attempt to change the visibility timeout or to delete that message 10 seconds after you initially change the
     * visibility timeout (a total of 25 seconds) might result in an error.
     * </p>
     * <p>
     * An Amazon SQS message has three basic states:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Sent to a queue by a producer.
     * </p>
     * </li>
     * <li>
     * <p>
     * Received from the queue by a consumer.
     * </p>
     * </li>
     * <li>
     * <p>
     * Deleted from the queue.
     * </p>
     * </li>
     * </ol>
     * <p>
     * A message is considered to be <i>stored</i> after it is sent to a queue by a producer, but not yet received from
     * the queue by a consumer (that is, between states 1 and 2). There is no limit to the number of stored messages. A
     * message is considered to be <i>in flight</i> after it is received from a queue by a consumer, but not yet deleted
     * from the queue (that is, between states 2 and 3). There is a limit to the number of in flight messages.
     * </p>
     * <p>
     * Limits that apply to in flight messages are unrelated to the <i>unlimited</i> number of stored messages.
     * </p>
     * <p>
     * For most standard queues (depending on queue traffic and message backlog), there can be a maximum of
     * approximately 120,000 in flight messages (received from a queue by a consumer, but not yet deleted from the
     * queue). If you reach this limit, Amazon SQS returns the <code>OverLimit</code> error message. To avoid reaching
     * the limit, you should delete messages from the queue after they're processed. You can also increase the number of
     * queues you use to process your messages. To request a limit increase, <a href=
     * "https://console.aws.amazon.com/support/home#/case/create?issueType=service-limit-increase&amp;limitType=service-code-sqs"
     * >file a support request</a>.
     * </p>
     * <p>
     * For FIFO queues, there can be a maximum of 120,000 in flight messages (received from a queue by a consumer, but
     * not yet deleted from the queue). If you reach this limit, Amazon SQS returns no error messages.
     * </p>
     * <important>
     * <p>
     * If you attempt to set the <code>VisibilityTimeout</code> to a value greater than the maximum time left, Amazon
     * SQS returns an error. Amazon SQS doesn't automatically recalculate and increase the timeout to the maximum
     * remaining time.
     * </p>
     * <p>
     * Unlike with a queue, when you change the visibility timeout for a specific message the timeout value is applied
     * immediately but isn't saved in memory for that message. If you don't delete a message after it is received, the
     * visibility timeout for the message reverts to the original timeout value (not to the value you set using the
     * <code>ChangeMessageVisibility</code> action) the next time the message is received.
     * </p>
     * </important>
     *
     * @param changeMessageVisibilityRequest
     * @return A Java Future containing the result of the ChangeMessageVisibility operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>MessageNotInflightException The specified message isn't in flight.</li>
     *         <li>ReceiptHandleIsInvalidException The specified receipt handle isn't valid.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.ChangeMessageVisibility
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ChangeMessageVisibility" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ChangeMessageVisibilityResponse> changeMessageVisibility(
            ChangeMessageVisibilityRequest changeMessageVisibilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(changeMessageVisibilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, changeMessageVisibilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ChangeMessageVisibility");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ChangeMessageVisibilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ChangeMessageVisibilityResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ChangeMessageVisibilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ChangeMessageVisibilityRequest, ChangeMessageVisibilityResponse>()
                            .withOperationName("ChangeMessageVisibility").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ChangeMessageVisibilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(changeMessageVisibilityRequest));
            CompletableFuture<ChangeMessageVisibilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the visibility timeout of multiple messages. This is a batch version of
     * <code> <a>ChangeMessageVisibility</a>.</code> The result of the action on each message is reported individually
     * in the response. You can send up to 10 <code> <a>ChangeMessageVisibility</a> </code> requests with each
     * <code>ChangeMessageVisibilityBatch</code> action.
     * </p>
     * <important>
     * <p>
     * Because the batch request can result in a combination of successful and unsuccessful actions, you should check
     * for batch errors even when the call returns an HTTP status code of <code>200</code>.
     * </p>
     * </important>
     *
     * @param changeMessageVisibilityBatchRequest
     * @return A Java Future containing the result of the ChangeMessageVisibilityBatch operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyEntriesInBatchRequestException The batch request contains more entries than permissible. For
     *         Amazon SQS, the maximum number of entries you can include in a single <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessageBatch.html"
     *         >SendMessageBatch</a>, <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessageBatch.html"
     *         >DeleteMessageBatch</a>, or <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibilityBatch.html"
     *         >ChangeMessageVisibilityBatch</a> request is 10.</li>
     *         <li>EmptyBatchRequestException The batch request doesn't contain any entries.</li>
     *         <li>BatchEntryIdsNotDistinctException Two or more batch entries in the request have the same
     *         <code>Id</code>.</li>
     *         <li>InvalidBatchEntryIdException The <code>Id</code> of a batch entry in a batch request doesn't abide by
     *         the specification.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.ChangeMessageVisibilityBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ChangeMessageVisibilityBatch"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ChangeMessageVisibilityBatchResponse> changeMessageVisibilityBatch(
            ChangeMessageVisibilityBatchRequest changeMessageVisibilityBatchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(changeMessageVisibilityBatchRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, changeMessageVisibilityBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ChangeMessageVisibilityBatch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ChangeMessageVisibilityBatchResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ChangeMessageVisibilityBatchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ChangeMessageVisibilityBatchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ChangeMessageVisibilityBatchRequest, ChangeMessageVisibilityBatchResponse>()
                            .withOperationName("ChangeMessageVisibilityBatch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ChangeMessageVisibilityBatchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(changeMessageVisibilityBatchRequest));
            CompletableFuture<ChangeMessageVisibilityBatchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new standard or FIFO queue. You can pass one or more attributes in the request. Keep the following in
     * mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you don't specify the <code>FifoQueue</code> attribute, Amazon SQS creates a standard queue.
     * </p>
     * <note>
     * <p>
     * You can't change the queue type after you create it and you can't convert an existing standard queue into a FIFO
     * queue. You must either create a new FIFO queue for your application or delete your existing standard queue and
     * recreate it as a FIFO queue. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html#FIFO-queues-moving"
     * >Moving From a standard queue to a FIFO queue</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * If you don't provide a value for an attribute, the queue is created with the default value for the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you delete a queue, you must wait at least 60 seconds before creating a queue with the same name.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To successfully create a new queue, you must provide a queue name that adheres to the <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/limits-queues.html">limits
     * related to queues</a> and is unique within the scope of your queues.
     * </p>
     * <note>
     * <p>
     * After you create a queue, you must wait at least one second after the queue is created to be able to use the
     * queue.
     * </p>
     * </note>
     * <p>
     * To retrieve the URL of a queue, use the <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_GetQueueUrl.html">
     * <code>GetQueueUrl</code> </a> action. This action only requires the <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_CreateQueue.html#API_CreateQueue_RequestSyntax"
     * > <code>QueueName</code> </a> parameter.
     * </p>
     * <p>
     * When creating queues, keep the following points in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify the name of an existing queue and provide the exact same names and values for all its attributes,
     * the <a href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_CreateQueue.html">
     * <code>CreateQueue</code> </a> action will return the URL of the existing queue instead of creating a new one.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you attempt to create a queue with a name that already exists but with different attribute names or values,
     * the <code>CreateQueue</code> action will return an error. This ensures that existing queues are not inadvertently
     * altered.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param createQueueRequest
     * @return A Java Future containing the result of the CreateQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>QueueDeletedRecentlyException You must wait 60 seconds after deleting a queue before you can create
     *         another queue with the same name.</li>
     *         <li>QueueNameExistsException A queue with this name already exists. Amazon SQS returns this error only if
     *         the request includes attributes whose values differ from those of the existing queue.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidAttributeNameException The specified attribute doesn't exist.</li>
     *         <li>InvalidAttributeValueException A queue attribute value is invalid.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.CreateQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/CreateQueue" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_CreateQueue_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<CreateQueueResponse> createQueue(CreateQueueRequest createQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateQueueResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateQueueRequest, CreateQueueResponse>()
                            .withOperationName("CreateQueue").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createQueueRequest));
            CompletableFuture<CreateQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified message from the specified queue. To select the message to delete, use the
     * <code>ReceiptHandle</code> of the message (<i>not</i> the <code>MessageId</code> which you receive when you send
     * the message). Amazon SQS can delete a message from a queue even if a visibility timeout setting causes the
     * message to be locked by another consumer. Amazon SQS automatically deletes messages left in a queue longer than
     * the retention period configured for the queue.
     * </p>
     * <note>
     * <p>
     * Each time you receive a message, meaning when a consumer retrieves a message from the queue, it comes with a
     * unique <code>ReceiptHandle</code>. If you receive the same message more than once, you will get a different
     * <code>ReceiptHandle</code> each time. When you want to delete a message using the <code>DeleteMessage</code>
     * action, you must use the <code>ReceiptHandle</code> from the most recent time you received the message. If you
     * use an old <code>ReceiptHandle</code>, the request will succeed, but the message might not be deleted.
     * </p>
     * <p>
     * For standard queues, it is possible to receive a message even after you delete it. This might happen on rare
     * occasions if one of the servers which stores a copy of the message is unavailable when you send the request to
     * delete the message. The copy remains on the server and might be returned to you during a subsequent receive
     * request. You should ensure that your application is idempotent, so that receiving a message more than once does
     * not cause issues.
     * </p>
     * </note>
     *
     * @param deleteMessageRequest
     * @return A Java Future containing the result of the DeleteMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidIdFormatException The specified receipt handle isn't valid for the current version.</li>
     *         <li>ReceiptHandleIsInvalidException The specified receipt handle isn't valid.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.DeleteMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/DeleteMessage" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_DeleteMessage_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<DeleteMessageResponse> deleteMessage(DeleteMessageRequest deleteMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMessageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMessageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMessageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMessageRequest, DeleteMessageResponse>()
                            .withOperationName("DeleteMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMessageRequest));
            CompletableFuture<DeleteMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes up to ten messages from the specified queue. This is a batch version of
     * <code> <a>DeleteMessage</a>.</code> The result of the action on each message is reported individually in the
     * response.
     * </p>
     * <important>
     * <p>
     * Because the batch request can result in a combination of successful and unsuccessful actions, you should check
     * for batch errors even when the call returns an HTTP status code of <code>200</code>.
     * </p>
     * </important>
     *
     * @param deleteMessageBatchRequest
     * @return A Java Future containing the result of the DeleteMessageBatch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyEntriesInBatchRequestException The batch request contains more entries than permissible. For
     *         Amazon SQS, the maximum number of entries you can include in a single <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessageBatch.html"
     *         >SendMessageBatch</a>, <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessageBatch.html"
     *         >DeleteMessageBatch</a>, or <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibilityBatch.html"
     *         >ChangeMessageVisibilityBatch</a> request is 10.</li>
     *         <li>EmptyBatchRequestException The batch request doesn't contain any entries.</li>
     *         <li>BatchEntryIdsNotDistinctException Two or more batch entries in the request have the same
     *         <code>Id</code>.</li>
     *         <li>InvalidBatchEntryIdException The <code>Id</code> of a batch entry in a batch request doesn't abide by
     *         the specification.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.DeleteMessageBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/DeleteMessageBatch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMessageBatchResponse> deleteMessageBatch(DeleteMessageBatchRequest deleteMessageBatchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMessageBatchRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMessageBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMessageBatch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMessageBatchResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMessageBatchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteMessageBatchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMessageBatchRequest, DeleteMessageBatchResponse>()
                            .withOperationName("DeleteMessageBatch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMessageBatchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMessageBatchRequest));
            CompletableFuture<DeleteMessageBatchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the queue specified by the <code>QueueUrl</code>, regardless of the queue's contents.
     * </p>
     * <important>
     * <p>
     * Be careful with the <code>DeleteQueue</code> action: When you delete a queue, any messages in the queue are no
     * longer available.
     * </p>
     * </important>
     * <p>
     * When you delete a queue, the deletion process takes up to 60 seconds. Requests you send involving that queue
     * during the 60 seconds might succeed. For example, a <code> <a>SendMessage</a> </code> request might succeed, but
     * after 60 seconds the queue and the message you sent no longer exist.
     * </p>
     * <p>
     * When you delete a queue, you must wait at least 60 seconds before creating a queue with the same name.
     * </p>
     * <note>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * The delete operation uses the HTTP <code>GET</code> verb.
     * </p>
     * </note>
     *
     * @param deleteQueueRequest
     * @return A Java Future containing the result of the DeleteQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.DeleteQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/DeleteQueue" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_DeleteQueue_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<DeleteQueueResponse> deleteQueue(DeleteQueueRequest deleteQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteQueueResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteQueueRequest, DeleteQueueResponse>()
                            .withOperationName("DeleteQueue").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteQueueRequest));
            CompletableFuture<DeleteQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets attributes for the specified queue.
     * </p>
     * <note>
     * <p>
     * To determine whether a queue is <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues.html">FIFO</a>, you
     * can check whether <code>QueueName</code> ends with the <code>.fifo</code> suffix.
     * </p>
     * </note>
     *
     * @param getQueueAttributesRequest
     * @return A Java Future containing the result of the GetQueueAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidAttributeNameException The specified attribute doesn't exist.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.GetQueueAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/GetQueueAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueueAttributesResponse> getQueueAttributes(GetQueueAttributesRequest getQueueAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueueAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueueAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueueAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueueAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQueueAttributesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueueAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueueAttributesRequest, GetQueueAttributesResponse>()
                            .withOperationName("GetQueueAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQueueAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueueAttributesRequest));
            CompletableFuture<GetQueueAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetQueueUrl</code> API returns the URL of an existing Amazon SQS queue. This is useful when you know
     * the queue's name but need to retrieve its URL for further operations.
     * </p>
     * <p>
     * To access a queue owned by another Amazon Web Services account, use the <code>QueueOwnerAWSAccountId</code>
     * parameter to specify the account ID of the queue's owner. Note that the queue owner must grant you the necessary
     * permissions to access the queue. For more information about accessing shared queues, see the
     * <code> <a>AddPermission</a> </code> API or <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-writing-an-sqs-policy.html#write-messages-to-shared-queue"
     * >Allow developers to write messages to a shared queue</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     *
     * @param getQueueUrlRequest
     *        Retrieves the URL of an existing queue based on its name and, optionally, the Amazon Web Services account
     *        ID.
     * @return A Java Future containing the result of the GetQueueUrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.GetQueueUrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/GetQueueUrl" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_GetQueueUrl_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<GetQueueUrlResponse> getQueueUrl(GetQueueUrlRequest getQueueUrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueueUrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueueUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueueUrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueueUrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetQueueUrlResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQueueUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueueUrlRequest, GetQueueUrlResponse>()
                            .withOperationName("GetQueueUrl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQueueUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQueueUrlRequest));
            CompletableFuture<GetQueueUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of your queues that have the <code>RedrivePolicy</code> queue attribute configured with a
     * dead-letter queue.
     * </p>
     * <p>
     * The <code>ListDeadLetterSourceQueues</code> methods supports pagination. Set parameter <code>MaxResults</code> in
     * the request to specify the maximum number of results to be returned in the response. If you do not set
     * <code>MaxResults</code>, the response includes a maximum of 1,000 results. If you set <code>MaxResults</code> and
     * there are additional results to display, the response includes a value for <code>NextToken</code>. Use
     * <code>NextToken</code> as a parameter in your next request to <code>ListDeadLetterSourceQueues</code> to receive
     * the next page of results.
     * </p>
     * <p>
     * For more information about using dead-letter queues, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     * >Using Amazon SQS Dead-Letter Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     *
     * @param listDeadLetterSourceQueuesRequest
     * @return A Java Future containing the result of the ListDeadLetterSourceQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.ListDeadLetterSourceQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ListDeadLetterSourceQueues"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeadLetterSourceQueuesResponse> listDeadLetterSourceQueues(
            ListDeadLetterSourceQueuesRequest listDeadLetterSourceQueuesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeadLetterSourceQueuesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeadLetterSourceQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeadLetterSourceQueues");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeadLetterSourceQueuesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeadLetterSourceQueuesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListDeadLetterSourceQueuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeadLetterSourceQueuesRequest, ListDeadLetterSourceQueuesResponse>()
                            .withOperationName("ListDeadLetterSourceQueues").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDeadLetterSourceQueuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDeadLetterSourceQueuesRequest));
            CompletableFuture<ListDeadLetterSourceQueuesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the most recent message movement tasks (up to 10) under a specific source queue.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This action is currently limited to supporting message redrive from <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     * >dead-letter queues (DLQs)</a> only. In this context, the source queue is the dead-letter queue (DLQ), while the
     * destination queue can be the original source queue (from which the messages were driven to the
     * dead-letter-queue), or a custom destination queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Only one active message movement task is supported per queue at any given time.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param listMessageMoveTasksRequest
     * @return A Java Future containing the result of the ListMessageMoveTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException One or more specified resources don't exist.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.ListMessageMoveTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ListMessageMoveTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMessageMoveTasksResponse> listMessageMoveTasks(
            ListMessageMoveTasksRequest listMessageMoveTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMessageMoveTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMessageMoveTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMessageMoveTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMessageMoveTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMessageMoveTasksResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListMessageMoveTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMessageMoveTasksRequest, ListMessageMoveTasksResponse>()
                            .withOperationName("ListMessageMoveTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMessageMoveTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMessageMoveTasksRequest));
            CompletableFuture<ListMessageMoveTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all cost allocation tags added to the specified Amazon SQS queue. For an overview, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-tags.html">Tagging
     * Your Amazon SQS Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param listQueueTagsRequest
     * @return A Java Future containing the result of the ListQueueTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.ListQueueTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ListQueueTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListQueueTagsResponse> listQueueTags(ListQueueTagsRequest listQueueTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQueueTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueueTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueueTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQueueTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListQueueTagsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListQueueTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQueueTagsRequest, ListQueueTagsResponse>()
                            .withOperationName("ListQueueTags").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQueueTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQueueTagsRequest));
            CompletableFuture<ListQueueTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of your queues in the current region. The response includes a maximum of 1,000 results. If you
     * specify a value for the optional <code>QueueNamePrefix</code> parameter, only queues with a name that begins with
     * the specified value are returned.
     * </p>
     * <p>
     * The <code>listQueues</code> methods supports pagination. Set parameter <code>MaxResults</code> in the request to
     * specify the maximum number of results to be returned in the response. If you do not set <code>MaxResults</code>,
     * the response includes a maximum of 1,000 results. If you set <code>MaxResults</code> and there are additional
     * results to display, the response includes a value for <code>NextToken</code>. Use <code>NextToken</code> as a
     * parameter in your next request to <code>listQueues</code> to receive the next page of results.
     * </p>
     * <note>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param listQueuesRequest
     * @return A Java Future containing the result of the ListQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.ListQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ListQueues" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_ListQueues_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<ListQueuesResponse> listQueues(ListQueuesRequest listQueuesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQueuesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueues");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQueuesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListQueuesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListQueuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQueuesRequest, ListQueuesResponse>().withOperationName("ListQueues")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQueuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQueuesRequest));
            CompletableFuture<ListQueuesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes available messages in a queue (including in-flight messages) specified by the <code>QueueURL</code>
     * parameter.
     * </p>
     * <important>
     * <p>
     * When you use the <code>PurgeQueue</code> action, you can't retrieve any messages deleted from a queue.
     * </p>
     * <p>
     * The message deletion process takes up to 60 seconds. We recommend waiting for 60 seconds regardless of your
     * queue's size.
     * </p>
     * </important>
     * <p>
     * Messages sent to the queue <i>before</i> you call <code>PurgeQueue</code> might be received but are deleted
     * within the next minute.
     * </p>
     * <p>
     * Messages sent to the queue <i>after</i> you call <code>PurgeQueue</code> might be deleted while the queue is
     * being purged.
     * </p>
     *
     * @param purgeQueueRequest
     * @return A Java Future containing the result of the PurgeQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>PurgeQueueInProgressException Indicates that the specified queue previously received a
     *         <code>PurgeQueue</code> request within the last 60 seconds (the time it can take to delete the messages
     *         in the queue).</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.PurgeQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/PurgeQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PurgeQueueResponse> purgeQueue(PurgeQueueRequest purgeQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(purgeQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, purgeQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PurgeQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PurgeQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PurgeQueueResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PurgeQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PurgeQueueRequest, PurgeQueueResponse>().withOperationName("PurgeQueue")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PurgeQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(purgeQueueRequest));
            CompletableFuture<PurgeQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves one or more messages (up to 10), from the specified queue. Using the <code>WaitTimeSeconds</code>
     * parameter enables long-poll support. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-long-polling.html">Amazon
     * SQS Long Polling</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Short poll is the default behavior where a weighted random set of machines is sampled on a
     * <code>ReceiveMessage</code> call. Therefore, only the messages on the sampled machines are returned. If the
     * number of messages in the queue is small (fewer than 1,000), you most likely get fewer messages than you
     * requested per <code>ReceiveMessage</code> call. If the number of messages in the queue is extremely small, you
     * might not receive any messages in a particular <code>ReceiveMessage</code> response. If this happens, repeat the
     * request.
     * </p>
     * <p>
     * For each message returned, the response includes the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The message body.
     * </p>
     * </li>
     * <li>
     * <p>
     * An MD5 digest of the message body. For information about MD5, see <a
     * href="https://www.ietf.org/rfc/rfc1321.txt">RFC1321</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>MessageId</code> you received when you sent the message to the queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * The receipt handle.
     * </p>
     * </li>
     * <li>
     * <p>
     * The message attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * An MD5 digest of the message attributes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The receipt handle is the identifier you must provide when deleting the message. For more information, see <a
     * href
     * ="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-message-identifiers.html"
     * >Queue and Message Identifiers</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * You can provide the <code>VisibilityTimeout</code> parameter in your request. The parameter is applied to the
     * messages that Amazon SQS returns in the response. If you don't include the parameter, the overall visibility
     * timeout for the queue is used for the returned messages. The default visibility timeout for a queue is 30
     * seconds.
     * </p>
     * <note>
     * <p>
     * In the future, new attributes might be added. If you write code that calls this action, we recommend that you
     * structure your code so that it can handle new attributes gracefully.
     * </p>
     * </note>
     *
     * @param receiveMessageRequest
     *        Retrieves one or more messages from a specified queue.
     * @return A Java Future containing the result of the ReceiveMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>OverLimitException The specified action violates a limit. For example, <code>ReceiveMessage</code>
     *         returns this error if the maximum number of in flight messages is reached and <code>AddPermission</code>
     *         returns this error if the maximum number of permissions for the queue is reached.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>KmsDisabledException The request was denied due to request throttling.</li>
     *         <li>KmsInvalidStateException The request was rejected because the state of the specified resource is not
     *         valid for this request.</li>
     *         <li>KmsNotFoundException The request was rejected because the specified entity or resource could not be
     *         found.</li>
     *         <li>KmsOptInRequiredException The request was rejected because the specified key policy isn't
     *         syntactically or semantically correct.</li>
     *         <li>KmsThrottledException Amazon Web Services KMS throttles requests for the following conditions.</li>
     *         <li>KmsAccessDeniedException The caller doesn't have the required KMS access.</li>
     *         <li>KmsInvalidKeyUsageException The request was rejected for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The KeyUsage value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (KeySpec).
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.ReceiveMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/ReceiveMessage" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_ReceiveMessage_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<ReceiveMessageResponse> receiveMessage(ReceiveMessageRequest receiveMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(receiveMessageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, receiveMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReceiveMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReceiveMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReceiveMessageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ReceiveMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReceiveMessageRequest, ReceiveMessageResponse>()
                            .withOperationName("ReceiveMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ReceiveMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(receiveMessageRequest));
            CompletableFuture<ReceiveMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes any permissions in the queue policy that matches the specified <code>Label</code> parameter.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * Only the owner of a queue can remove permissions from it.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To remove the ability to change queue permissions, you must deny permission to the <code>AddPermission</code>,
     * <code>RemovePermission</code>, and <code>SetQueueAttributes</code> actions in your IAM policy.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param removePermissionRequest
     * @return A Java Future containing the result of the RemovePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.RemovePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemovePermissionResponse> removePermission(RemovePermissionRequest removePermissionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemovePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemovePermissionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RemovePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                            .withOperationName("RemovePermission").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(removePermissionRequest));
            CompletableFuture<RemovePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delivers a message to the specified queue.
     * </p>
     * <important>
     * <p>
     * A message can include only XML, JSON, and unformatted text. The following Unicode characters are allowed. For
     * more information, see the <a href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
     * </p>
     * <p>
     * <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
     * <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
     * </p>
     * <p>
     * If a message contains characters outside the allowed set, Amazon SQS rejects the message and returns an
     * InvalidMessageContents error. Ensure that your message body includes only valid characters to avoid this
     * exception.
     * </p>
     * </important>
     *
     * @param sendMessageRequest
     * @return A Java Future containing the result of the SendMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidMessageContentsException The message contains characters outside the allowed set.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>KmsDisabledException The request was denied due to request throttling.</li>
     *         <li>KmsInvalidStateException The request was rejected because the state of the specified resource is not
     *         valid for this request.</li>
     *         <li>KmsNotFoundException The request was rejected because the specified entity or resource could not be
     *         found.</li>
     *         <li>KmsOptInRequiredException The request was rejected because the specified key policy isn't
     *         syntactically or semantically correct.</li>
     *         <li>KmsThrottledException Amazon Web Services KMS throttles requests for the following conditions.</li>
     *         <li>KmsAccessDeniedException The caller doesn't have the required KMS access.</li>
     *         <li>KmsInvalidKeyUsageException The request was rejected for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The KeyUsage value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (KeySpec).
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.SendMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/SendMessage" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_SendMessage_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<SendMessageResponse> sendMessage(SendMessageRequest sendMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendMessageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendMessageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendMessageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SendMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendMessageRequest, SendMessageResponse>()
                            .withOperationName("SendMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendMessageRequest));
            CompletableFuture<SendMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use <code>SendMessageBatch</code> to send up to 10 messages to the specified queue by assigning either
     * identical or different values to each message (or by not assigning values at all). This is a batch version of
     * <code> <a>SendMessage</a>.</code> For a FIFO queue, multiple messages within a single batch are enqueued in the
     * order they are sent.
     * </p>
     * <p>
     * The result of sending each message is reported individually in the response. Because the batch request can result
     * in a combination of successful and unsuccessful actions, you should check for batch errors even when the call
     * returns an HTTP status code of <code>200</code>.
     * </p>
     * <p>
     * The maximum allowed individual message size and the maximum total payload size (the sum of the individual lengths
     * of all of the batched messages) are both 1 MiB 1,048,576 bytes.
     * </p>
     * <important>
     * <p>
     * A message can include only XML, JSON, and unformatted text. The following Unicode characters are allowed. For
     * more information, see the <a href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
     * </p>
     * <p>
     * <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
     * <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
     * </p>
     * <p>
     * If a message contains characters outside the allowed set, Amazon SQS rejects the message and returns an
     * InvalidMessageContents error. Ensure that your message body includes only valid characters to avoid this
     * exception.
     * </p>
     * </important>
     * <p>
     * If you don't specify the <code>DelaySeconds</code> parameter for an entry, Amazon SQS uses the default value for
     * the queue.
     * </p>
     *
     * @param sendMessageBatchRequest
     * @return A Java Future containing the result of the SendMessageBatch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyEntriesInBatchRequestException The batch request contains more entries than permissible. For
     *         Amazon SQS, the maximum number of entries you can include in a single <a
     *         href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessageBatch.html"
     *         >SendMessageBatch</a>, <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_DeleteMessageBatch.html"
     *         >DeleteMessageBatch</a>, or <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_ChangeMessageVisibilityBatch.html"
     *         >ChangeMessageVisibilityBatch</a> request is 10.</li>
     *         <li>EmptyBatchRequestException The batch request doesn't contain any entries.</li>
     *         <li>BatchEntryIdsNotDistinctException Two or more batch entries in the request have the same
     *         <code>Id</code>.</li>
     *         <li>BatchRequestTooLongException The length of all the messages put together is more than the limit.</li>
     *         <li>InvalidBatchEntryIdException The <code>Id</code> of a batch entry in a batch request doesn't abide by
     *         the specification.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>KmsDisabledException The request was denied due to request throttling.</li>
     *         <li>KmsInvalidStateException The request was rejected because the state of the specified resource is not
     *         valid for this request.</li>
     *         <li>KmsNotFoundException The request was rejected because the specified entity or resource could not be
     *         found.</li>
     *         <li>KmsOptInRequiredException The request was rejected because the specified key policy isn't
     *         syntactically or semantically correct.</li>
     *         <li>KmsThrottledException Amazon Web Services KMS throttles requests for the following conditions.</li>
     *         <li>KmsAccessDeniedException The caller doesn't have the required KMS access.</li>
     *         <li>KmsInvalidKeyUsageException The request was rejected for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The KeyUsage value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (KeySpec).
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.SendMessageBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/SendMessageBatch" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_SendMessageBatch_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<SendMessageBatchResponse> sendMessageBatch(SendMessageBatchRequest sendMessageBatchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendMessageBatchRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendMessageBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendMessageBatch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendMessageBatchResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendMessageBatchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SendMessageBatchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendMessageBatchRequest, SendMessageBatchResponse>()
                            .withOperationName("SendMessageBatch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendMessageBatchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendMessageBatchRequest));
            CompletableFuture<SendMessageBatchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the value of one or more queue attributes, like a policy. When you change a queue's attributes, the change
     * can take up to 60 seconds for most of the attributes to propagate throughout the Amazon SQS system. Changes made
     * to the <code>MessageRetentionPeriod</code> attribute can take up to 15 minutes and will impact existing messages
     * in the queue potentially causing them to be expired and deleted if the <code>MessageRetentionPeriod</code> is
     * reduced below the age of existing messages.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * In the future, new attributes might be added. If you write code that calls this action, we recommend that you
     * structure your code so that it can handle new attributes gracefully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To remove the ability to change queue permissions, you must deny permission to the <code>AddPermission</code>,
     * <code>RemovePermission</code>, and <code>SetQueueAttributes</code> actions in your IAM policy.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param setQueueAttributesRequest
     * @return A Java Future containing the result of the SetQueueAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidAttributeNameException The specified attribute doesn't exist.</li>
     *         <li>InvalidAttributeValueException A queue attribute value is invalid.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>OverLimitException The specified action violates a limit. For example, <code>ReceiveMessage</code>
     *         returns this error if the maximum number of in flight messages is reached and <code>AddPermission</code>
     *         returns this error if the maximum number of permissions for the queue is reached.</li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.SetQueueAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/SetQueueAttributes" target="_top">AWS API
     *      Documentation</a>
     * @see <a href="https://docs.aws.amazon.com/code-library/latest/ug/sqs_example_sqs_SetQueueAttributes_section.html"
     *      target="_top">Code Example</a>
     */
    @Override
    public CompletableFuture<SetQueueAttributesResponse> setQueueAttributes(SetQueueAttributesRequest setQueueAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setQueueAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setQueueAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetQueueAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetQueueAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SetQueueAttributesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SetQueueAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetQueueAttributesRequest, SetQueueAttributesResponse>()
                            .withOperationName("SetQueueAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetQueueAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setQueueAttributesRequest));
            CompletableFuture<SetQueueAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an asynchronous task to move messages from a specified source queue to a specified destination queue.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This action is currently limited to supporting message redrive from queues that are configured as <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-dead-letter-queues.html"
     * >dead-letter queues (DLQs)</a> of other Amazon SQS queues only. Non-SQS queue sources of dead-letter queues, such
     * as Lambda or Amazon SNS topics, are currently not supported.
     * </p>
     * </li>
     * <li>
     * <p>
     * In dead-letter queues redrive context, the <code>StartMessageMoveTask</code> the source queue is the DLQ, while
     * the destination queue can be the original source queue (from which the messages were driven to the
     * dead-letter-queue), or a custom destination queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Only one active message movement task is supported per queue at any given time.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param startMessageMoveTaskRequest
     * @return A Java Future containing the result of the StartMessageMoveTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException One or more specified resources don't exist.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.StartMessageMoveTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/StartMessageMoveTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartMessageMoveTaskResponse> startMessageMoveTask(
            StartMessageMoveTaskRequest startMessageMoveTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMessageMoveTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMessageMoveTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMessageMoveTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMessageMoveTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMessageMoveTaskResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartMessageMoveTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMessageMoveTaskRequest, StartMessageMoveTaskResponse>()
                            .withOperationName("StartMessageMoveTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMessageMoveTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMessageMoveTaskRequest));
            CompletableFuture<StartMessageMoveTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add cost allocation tags to the specified Amazon SQS queue. For an overview, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-tags.html">Tagging
     * Your Amazon SQS Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * When you use queue tags, keep the following guidelines in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Adding more than 50 tags to a queue isn't recommended.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags don't have any semantic meaning. Amazon SQS interprets tags as character strings.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * A new tag with a key identical to that of an existing tag overwrites the existing tag.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a full list of tag restrictions, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-limits.html#limits-queues"
     * >Quotas related to queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param tagQueueRequest
     * @return A Java Future containing the result of the TagQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.TagQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/TagQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagQueueResponse> tagQueue(TagQueueRequest tagQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagQueueResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagQueueRequest, TagQueueResponse>().withOperationName("TagQueue")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagQueueRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagQueueRequest));
            CompletableFuture<TagQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove cost allocation tags from the specified Amazon SQS queue. For an overview, see <a
     * href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-queue-tags.html">Tagging
     * Your Amazon SQS Queues</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Cross-account permissions don't apply to this action. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-customer-managed-policy-examples.html#grant-cross-account-permissions-to-role-and-user-name"
     * >Grant cross-account permissions to a role and a username</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param untagQueueRequest
     * @return A Java Future containing the result of the UntagQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidAddressException The specified ID is invalid.</li>
     *         <li>RequestThrottledException The request was denied due to request throttling.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Exceeds the permitted request rate for the queue or for the recipient of the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Ensure that the request rate is within the Amazon SQS limits for sending messages. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-quotas.html#quotas-requests"
     *         >Amazon SQS quotas</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>QueueDoesNotExistException Ensure that the <code>QueueUrl</code> is correct and that the queue has
     *         not been deleted.</li>
     *         <li>InvalidSecurityException The request was not made over HTTPS or did not use SigV4 for signing.</li>
     *         <li>UnsupportedOperationException Error code 400. Unsupported operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SqsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample SqsAsyncClient.UntagQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/UntagQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagQueueResponse> untagQueue(UntagQueueRequest untagQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SQS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagQueueResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "PurgeQueueInProgress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("PurgeQueueInProgress").httpStatusCode(400)
                            .exceptionBuilderSupplier(PurgeQueueInProgressException::builder).build());
                case "KmsInvalidKeyUsage":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsage").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "InvalidAttributeName":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeName").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeNameException::builder).build());
                case "RequestThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(RequestThrottledException::builder).build());
                case "BatchEntryIdsNotDistinct":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchEntryIdsNotDistinct").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchEntryIdsNotDistinctException::builder).build());
                case "TooManyEntriesInBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyEntriesInBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyEntriesInBatchRequestException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BatchRequestTooLong":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BatchRequestTooLong").httpStatusCode(400)
                            .exceptionBuilderSupplier(BatchRequestTooLongException::builder).build());
                case "UnsupportedOperation":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "KmsInvalidState":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidState").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "InvalidIdFormat":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidIdFormat").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidIdFormatException::builder).build());
                case "KmsNotFound":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFound").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "KmsThrottled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsThrottled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsThrottledException::builder).build());
                case "KmsOptInRequired":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsOptInRequired").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsOptInRequiredException::builder).build());
                case "KmsAccessDenied":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsAccessDenied").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsAccessDeniedException::builder).build());
                case "OverLimit":
                    return Optional.of(ExceptionMetadata.builder().errorCode("OverLimit").httpStatusCode(400)
                            .exceptionBuilderSupplier(OverLimitException::builder).build());
                case "QueueDoesNotExist":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDoesNotExist").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDoesNotExistException::builder).build());
                case "KmsDisabled":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabled").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "QueueNameExists":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueNameExists").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueNameExistsException::builder).build());
                case "MessageNotInflight":
                    return Optional.of(ExceptionMetadata.builder().errorCode("MessageNotInflight").httpStatusCode(400)
                            .exceptionBuilderSupplier(MessageNotInflightException::builder).build());
                case "ReceiptHandleIsInvalid":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ReceiptHandleIsInvalid").httpStatusCode(400)
                            .exceptionBuilderSupplier(ReceiptHandleIsInvalidException::builder).build());
                case "InvalidAttributeValue":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAttributeValue").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAttributeValueException::builder).build());
                case "InvalidMessageContents":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidMessageContents").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidMessageContentsException::builder).build());
                case "QueueDeletedRecently":
                    return Optional.of(ExceptionMetadata.builder().errorCode("QueueDeletedRecently").httpStatusCode(400)
                            .exceptionBuilderSupplier(QueueDeletedRecentlyException::builder).build());
                case "EmptyBatchRequest":
                    return Optional.of(ExceptionMetadata.builder().errorCode("EmptyBatchRequest").httpStatusCode(400)
                            .exceptionBuilderSupplier(EmptyBatchRequestException::builder).build());
                case "InvalidSecurity":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSecurity").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidSecurityException::builder).build());
                case "InvalidAddress":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAddress").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidAddressException::builder).build());
                case "InvalidBatchEntryId":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidBatchEntryId").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidBatchEntryIdException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagQueueRequest, UntagQueueResponse>().withOperationName("UntagQueue")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagQueueRequest));
            CompletableFuture<UntagQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public SqsAsyncBatchManager batchManager() {
        return SqsAsyncBatchManager.builder().client(this).scheduledExecutor(executorService).build();
    }

    @Override
    public final SqsServiceClientConfiguration serviceClientConfiguration() {
        return new SqsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(SqsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0").hasAwsQueryCompatible(true);
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        SqsServiceClientConfigurationBuilder serviceConfigBuilder = new SqsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        AttributeMap newContextParams = configuration.option(SdkClientOption.CLIENT_CONTEXT_PARAMS);
        AttributeMap originalContextParams = clientConfiguration.option(SdkClientOption.CLIENT_CONTEXT_PARAMS);
        newContextParams = (newContextParams != null) ? newContextParams : AttributeMap.empty();
        originalContextParams = originalContextParams != null ? originalContextParams : AttributeMap.empty();
        Validate.validState(
                Objects.equals(originalContextParams.get(SqsClientContextParams.CHECKSUM_VALIDATION_ENABLED),
                        newContextParams.get(SqsClientContextParams.CHECKSUM_VALIDATION_ENABLED)),
                "CHECKSUM_VALIDATION_ENABLED cannot be modified by request level plugins");
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
