package com.spun.util.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.spun.util.ObjectUtils;
import com.spun.util.StringUtils;
import com.spun.util.logger.SimpleLogger;

public class XMLUtils
{
  public static String locateFile(String fileLocation, String[] backupPaths)
  {
    String[] newArray = new String[backupPaths.length + 1];
    System.arraycopy(backupPaths, 0, newArray, 1, backupPaths.length);
    newArray[0] = ".";
    backupPaths = newArray;
    for (int i = 0; i < backupPaths.length; i++)
    {
      String tfileLocation = backupPaths[i] + File.separator + fileLocation;
      File file = new File(tfileLocation);
      if (file.exists())
      { return FileUtils.getResolvedPath(file); }
    }
    throw new Error(String.format("Couldn't find '%s' from locations %s with current directory '%s'", fileLocation,
        Arrays.asList(backupPaths), FileUtils.getResolvedPath(new File("."))));
  }

  public static HashMap<String, Object> parseProperties(String absoluteFileLocation, XMLNodeExtractor extractor)
  {
    try
    {
      FileInputStream stream = new FileInputStream(absoluteFileLocation);
      Document document = parseXML(stream);
      return extractProperties(document, extractor);
    }
    catch (Exception e)
    {
      SimpleLogger.variable("Property File ", absoluteFileLocation);
      throw ObjectUtils.throwAsError(e);
    }
  }

  private static HashMap<String, Object> extractProperties(Document document, XMLNodeExtractor extractor)
  {
    HashMap<String, Object> properties = new HashMap<String, Object>();
    NodeList list = document.getDocumentElement().getChildNodes();
    for (int i = 0; i < list.getLength(); i++)
    {
      extractor.extractProperty(list.item(i), properties);
    }
    return properties;
  }

  public static Document parseXML(String xml)
  {
    try
    {
      DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
      return builder.parse(new InputSource(new StringReader(xml)));
    }
    catch (Exception e)
    {
      throw ObjectUtils.throwAsError(e);
    }
  }

  public static Document parseXML(File xml)
  {
    return ObjectUtils.throwAsError(
        () -> DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new FileInputStream(xml)));
  }

  public static Document parseXML(InputStream stream)
  {
    return ObjectUtils.throwAsError(() -> DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(stream));
  }

  public static String extractStringValue(Node node)
  {
    NodeList childNodes = node.getChildNodes();
    if ((childNodes.getLength() == 1) && (childNodes.item(0).getChildNodes().getLength() == 0))
    {
      return StringUtils.loadNullableString(childNodes.item(0).getNodeValue());
    }
    else if (childNodes.getLength() > 1)
    { throw new Error("Should not be multiple children for node '" + node.getNodeName() + "'"); }
    return null;
  }

  public static String prettyPrint(String input, int indent)
  {
    return XMLUtils.prettyPrint(input, indent, false);
  }

  public static String prettyPrint(String input, int indent, boolean reorder)
  {
    try
    {
      Source xmlInput = new StreamSource(new StringReader(input));
      if (reorder)
      {
        xmlInput = sortXml(xmlInput);
      }
      StringWriter stringWriter = new StringWriter();
      StreamResult xmlOutput = new StreamResult(stringWriter);
      TransformerFactory transformerFactory = TransformerFactory.newInstance();
      transformerFactory.setAttribute("indent-number", indent);
      Transformer transformer = transformerFactory.newTransformer();
      transformer.setOutputProperty(OutputKeys.INDENT, "yes");
      transformer.transform(xmlInput, xmlOutput);
      try (Writer writer = xmlOutput.getWriter())
      {
        return writer.toString();
      }
    }
    catch (TransformerException | IOException e)
    {
      return input;
    }
  }

  private static Source sortXml(Source xmlInput)
  {
    try
    {
      Document doc = DocumentBuilderFactory.newInstance().newDocumentBuilder()
          .parse(new InputSource(((StreamSource) xmlInput).getReader()));
      sortNode(doc.getDocumentElement());
      return new DOMSource(doc);
    }
    catch (Exception e)
    {
      throw ObjectUtils.throwAsError(e);
    }
  }

  private static void sortNode(Node node)
  {
    sortAttributes(node);
    NodeList children = node.getChildNodes();
    java.util.List<Node> elementNodes = new java.util.ArrayList<Node>();
    for (int i = 0; i < children.getLength(); i++)
    {
      Node child = children.item(i);
      if (child.getNodeType() == Node.ELEMENT_NODE)
      {
        sortNode(child);
        elementNodes.add(child);
      }
    }
    elementNodes.sort((a, b) -> a.getNodeName().compareTo(b.getNodeName()));
    for (Node child : elementNodes)
    {
      node.removeChild(child);
      node.appendChild(child);
    }
  }

  private static void sortAttributes(Node node)
  {
    if (node.hasAttributes())
    {
      org.w3c.dom.NamedNodeMap attrs = node.getAttributes();
      java.util.TreeMap<String, Node> sortedAttrs = new java.util.TreeMap<String, Node>();
      for (int i = 0; i < attrs.getLength(); i++)
      {
        Node attr = attrs.item(i);
        sortedAttrs.put(attr.getNodeName(), attr);
      }
      for (int i = attrs.getLength() - 1; i >= 0; i--)
      {
        attrs.removeNamedItem(attrs.item(i).getNodeName());
      }
      for (Node attr : sortedAttrs.values())
      {
        attrs.setNamedItem(attr);
      }
    }
  }
}
