package com.atlassian.mail.msgraph.service;

import com.atlassian.mail.msgraph.settings.providers.MailConnectionSettingsProvider;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.microsoft.graph.authentication.IAuthenticationProvider;
import com.microsoft.graph.http.GraphServiceException;
import com.microsoft.graph.options.QueryOption;
import com.microsoft.graph.requests.GraphServiceClient;
import com.microsoft.graph.requests.MailFolderCollectionPage;
import com.microsoft.graph.requests.MessageCollectionPage;
import com.microsoft.graph.requests.UserRequestBuilder;
import io.atlassian.fugue.Checked;
import io.atlassian.fugue.Either;
import io.atlassian.fugue.Option;
import io.atlassian.fugue.Unit;

import java.net.URL;
import java.util.Objects;
import okhttp3.Request;
import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static io.atlassian.fugue.Either.left;
import static io.atlassian.fugue.Either.right;
import static io.atlassian.fugue.Option.none;

public class MicrosoftGraphMailClient {
    private static final Logger logger = LoggerFactory.getLogger(MicrosoftGraphMailClient.class);

    public static final Integer MS_LIST_MESSAGES_API_PAGE_SIZE = 10;
    public static final Integer HYBRID_MS_EXCHANGE_SERVER_ERROR = 405;

    private final GraphServiceClient<Request> graphClient;

    private Option<String> userPrincipalName = none();

    public MicrosoftGraphMailClient(MailConnectionSettingsProvider settingsProvider) {
        this.graphClient = getGraphClient(settingsProvider);
    }

    @VisibleForTesting
    protected MicrosoftGraphMailClient(GraphServiceClient<Request> graphClient) {
        this.graphClient = graphClient;
        this.userPrincipalName = none();
    }

    @Nonnull
    @VisibleForTesting
    GraphServiceClient<Request> getGraphClient(MailConnectionSettingsProvider settingsProvider) {
        return GraphServiceClient.builder()
                .authenticationProvider(
                        new SimpleAuthProvider(() -> settingsProvider.getAuthenticationToken().getOrThrow(
                                () -> new CouldNotRetrieveAccessTokenException(
                                        String.format("Could not retrieve access token for connection [%s]", settingsProvider.getMailSettings().getId())
                                )
                        ))
                )
                .buildClient();
    }

    @Nonnull
    public Either<Throwable, MessageCollectionPage> getMessages(DateTime pullFromDate, String folderId) {
        return executeGraphClientRequest(g -> userRequestBuilder(g)
                        .mailFolders(folderId)
                        .messages()
                        .buildRequest(getGraphClientQueryOptions(pullFromDate))
                        .top(MS_LIST_MESSAGES_API_PAGE_SIZE)
                        .select("id")
                        .get());
    }

    @Nonnull
    public Either<Throwable, MessageCollectionPage> getNextMessagesPage(MessageCollectionPage messagesPage) {
        return executeGraphClientRequest(g -> Objects.requireNonNull(messagesPage.getNextPage())
                .buildRequest()
                .get());
    }

    @Nonnull
    public List<Message> getMessageStubsFromMessagePage(MessageCollectionPage messagesPage) {
        return messagesPage
                .getCurrentPage()
                .stream()
                .flatMap(messageStub -> getMimeMessage(messageStub).toStream())
                .collect(Collectors.toList());
    }

    @Nonnull
    public Either<Throwable, List<Message>> getMessageStubsFromMessagePageNew(MessageCollectionPage messagesPage) {

        List<com.microsoft.graph.models.Message> messageStubs = messagesPage.getCurrentPage();
        if (messageStubs.size() == 0) {
            return right(new ArrayList<>());
        }

        List<Message> messages = new ArrayList<>();
        Either<Throwable, Message> mimeMessage = null;

        for (com.microsoft.graph.models.Message messageStub : messageStubs) {
            mimeMessage = getMimeMessage(messageStub);

            if (mimeMessage.isRight()) {
                messages.add(mimeMessage.right().get());
            }
        }

        if (messages.isEmpty() && mimeMessage.isLeft()) {
            try {
                GraphServiceException exception = (GraphServiceException) mimeMessage.left().get();
                if (exception.getResponseCode() == HYBRID_MS_EXCHANGE_SERVER_ERROR) {
                    return left(mimeMessage.left().get());
                }
            } catch (Exception e) {
                return right(messages);
            }
        }

        return right(messages);
    }

    @Nonnull
    public Either<Throwable, Message> getMimeMessage(com.microsoft.graph.models.Message messageStub) {
        String pathForMimeMessage = userRequestString() + "/messages/" + messageStub.id + "/$value";

        Properties props = new Properties();
        Session session = Session.getDefaultInstance(props, null);

        return executeGraphClientRequest(g -> g.customRequest(pathForMimeMessage, InputStream.class).buildRequest().get())
                .flatMap(stream -> {
                    try {
                        Message msg = new MicrosoftMimeMessage(messageStub.id, session, stream);
                        return right(msg);
                    } catch (MessagingException e) {
                        logger.error(String.format("Failed construct MimeMessage from message stream (id: %s)", messageStub.id), e);
                        return left(e);
                    }
                });
    }

    public Either<Throwable, MailFolderCollectionPage> getFolderIdByName(String name) {
        return executeGraphClientRequest(g -> userRequestBuilder(g)
                .mailFolders()
                .buildRequest()
                .filter("displayName eq '" + name + "'")
                .get());
    }

    private UserRequestBuilder userRequestBuilder(GraphServiceClient<Request> g) {
        return g.me();
    }

    private String userRequestString() {
        return userPrincipalName.fold(
                () -> "/me",
                name -> String.format("/users('%s')", name));
    }

    public Either<Throwable, Unit> markMessageRead(Message message) {
        if (message instanceof MicrosoftMimeMessage) {
            MicrosoftMimeMessage msMessage = (MicrosoftMimeMessage) message;
            com.microsoft.graph.models.Message markAsRead = new com.microsoft.graph.models.Message();
            markAsRead.isRead = true;

            markAsRead.id = msMessage.microsoftMessageId;
            logger.info(String.format("Marking message with microsoftMessageId (%s) as read", markAsRead.id));

            return executeGraphClientRequest(g ->
                    userRequestBuilder(g).messages(msMessage.getMicrosoftMessageId()).buildRequest().patch(markAsRead))
                    .map(m -> Unit.VALUE);
        } else {
            throw new IllegalArgumentException(getClass() + " asked to handle non-Microsoft source message");
        }
    }

    private <T> Either<Throwable, T> executeGraphClientRequest(Function<GraphServiceClient<Request>, T> func) {
        Either<Throwable, T> either = Checked.now(() -> func.apply(graphClient)).toEither().leftMap(l -> (Throwable) l);

        if (either.isLeft()) {
            handleGraphClientResponseException(either.left().get());
        } else if (either.isRight()) {
            Object response = either.right().get();
            if (response != null) handleGraphClientSuccess();
        }

        return either;
    }

    private void handleGraphClientResponseException(Throwable th) {
        if (th instanceof GraphServiceException) {
            GraphServiceException ex = (GraphServiceException) th;
            logger.error(String.valueOf(ex));
        } else {
            logger.error(String.valueOf(th));
        }
    }

    private void handleGraphClientSuccess() {
        logger.debug("Graph client request successful");
    }

    private List<QueryOption> getGraphClientQueryOptions(DateTime pullFromDate) {
        String pullFromDateISO = pullFromDate.toString(ISODateTimeFormat.dateTime());

        QueryOption unreadAndReceivedAfterOption = new QueryOption("$filter", "isRead eq false and receivedDateTime ge " + pullFromDateISO);
        QueryOption oldestFirstOption = new QueryOption("$orderBy", "receivedDateTime asc");

        return ImmutableList.of(unreadAndReceivedAfterOption, oldestFirstOption);
    }

    private static class SimpleAuthProvider implements IAuthenticationProvider {

        final Supplier<String> tokenSupplier;

        public SimpleAuthProvider(Supplier<String> tokenSupplier) {
            this.tokenSupplier = tokenSupplier;
        }

        @Nonnull
        @Override
        public CompletableFuture<String> getAuthorizationTokenAsync(@Nonnull URL requestUrl) {
            return CompletableFuture.completedFuture(tokenSupplier.get());
        }

    }

    @VisibleForTesting
    public static class MicrosoftMimeMessage extends MimeMessage {

        private final String microsoftMessageId;

        public MicrosoftMimeMessage(String messageStubId, Session session, InputStream is) throws MessagingException {
            super(session, is);
            this.microsoftMessageId = messageStubId;
        }

        public String getMicrosoftMessageId() {
            return microsoftMessageId;
        }
    }

}
