/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.soebes.maven.plugins.dmg;

import java.io.File;

import org.apache.maven.plugin.AbstractMojo;

/**
 * This class contains all configurable parameters for
 * the doxygen call.
 * These parameters can be defined in the configuration section
 * of the plugin in.
 * @author Karl Heinz Marbaise
 *
 */
public abstract class AbstractDoxygenConfigurationMojo 
	extends AbstractMojo
{

	/**
	 * This defines which file will be used as
	 * Doxygen configuration file instead of creating one.
	 * 
	 * @parameter expression="${doxygen.configurationFile}" 
	 */
	private File configurationFile;

	/**
	 * This tag specifies the encoding used for all characters in the config file 
	 * that follow. The default is UTF-8 which is also the encoding used for all 
	 * text before the first occurrence of this tag. Doxygen uses libiconv (or the 
	 * iconv built into libc) for the transcoding. See 
	 * http://www.gnu.org/software/libiconv for the list of possible encodings.
	 *
	 * @parameter expression="${doxygen.doxyfileEncoding}" default-value="UTF-8"
	 */
	private String doxyfileEncoding;

	/**
	 * The PROJECT_NAME tag is a single word (or a sequence of words surrounded 
	 * by quotes) that should identify the project.
	 *
	 * @parameter expression="${doxygen.projectName}" default-value="MIGRATION.PARENT"
	 */
	private String projectName;

	/**
	 * The PROJECT_NUMBER tag can be used to enter a project or revision number. 
	 * This could be handy for archiving the generated documentation or 
	 * if some version control system is used.
	 *
	 * @parameter expression="${doxygen.projectNumber}"
	 */
	private String projectNumber;

	/**
	 * If the CREATE_SUBDIRS tag is set to YES, then doxygen will create 
	 * 4096 sub-directories (in 2 levels) under the output directory of each output 
	 * format and will distribute the generated files over these directories. 
	 * Enabling this option can be useful when feeding doxygen a huge amount of 
	 * source files, where putting all generated files in the same directory would 
	 * otherwise cause performance problems for the file system.
	 *
	 * @parameter expression="${doxygen.createSubdirs}" default-value="false"
	 */
	private boolean createSubdirs;

	/**
	 * The OUTPUT_LANGUAGE tag is used to specify the language in which all 
	 * documentation generated by doxygen is written. Doxygen will use this 
	 * information to generate all constant output in the proper language. 
	 * The default language is English, other supported languages are: 
	 * Afrikaans, Arabic, Brazilian, Catalan, Chinese, Chinese-Traditional, 
	 * Croatian, Czech, Danish, Dutch, Farsi, Finnish, French, German, Greek, 
	 * Hungarian, Italian, Japanese, Japanese-en (Japanese with English messages), 
	 * Korean, Korean-en, Lithuanian, Norwegian, Macedonian, Persian, Polish, 
	 * Portuguese, Romanian, Russian, Serbian, Serbian-Cyrilic, Slovak, Slovene, 
	 * Spanish, Swedish, and Ukrainian.
	 *
	 * @parameter expression="${doxygen.outputLanguage}" default-value="English"
	 */
	private String outputLanguage;

	/**
	 * If the BRIEF_MEMBER_DESC tag is set to YES (the default) Doxygen will 
	 * include brief member descriptions after the members that are listed in 
	 * the file and class documentation (similar to JavaDoc). 
	 * Set to NO to disable this.
	 *
	 * @parameter expression="${doxygen.briefMemberDesc}" default-value="true"
	 */
	private boolean briefMemberDesc;

	/**
	 * If the REPEAT_BRIEF tag is set to YES (the default) Doxygen will prepend 
	 * the brief description of a member or function before the detailed description. 
	 * Note: if both HIDE_UNDOC_MEMBERS and BRIEF_MEMBER_DESC are set to NO, the 
	 * brief descriptions will be completely suppressed.
	 *
	 * @parameter expression="${doxygen.repeatBrief}" default-value="true"
	 */
	private boolean repeatBrief;

	/**
	 * This tag implements a quasi-intelligent brief description abbreviator 
	 * that is used to form the text in various listings. Each string 
	 * in this list, if found as the leading text of the brief description, will be 
	 * stripped from the text and the result after processing the whole list, is 
	 * used as the annotated text. Otherwise, the brief description is used as-is. 
	 * If left blank, the following values are used ("$name" is automatically 
	 * replaced with the name of the entity): "The $name class" "The $name widget" 
	 * "The $name file" "is" "provides" "specifies" "contains" 
	 * "represents" "a" "an" "the"
	 *
	 * @parameter expression="${doxygen.abbreviateBrief}"
	 */
	private String abbreviateBrief;

	/**
	 * If the ALWAYS_DETAILED_SEC and REPEAT_BRIEF tags are both set to YES then 
	 * Doxygen will generate a detailed section even if there is only a brief 
	 * description.
	 *
	 * @parameter expression="${doxygen.alwaysDetailedSec}" default-value="false"
	 */
	private boolean alwaysDetailedSec;

	/**
	 * If the INLINE_INHERITED_MEMB tag is set to YES, doxygen will show all 
	 * inherited members of a class in the documentation of that class as if those 
	 * members were ordinary class members. Constructors, destructors and assignment 
	 * operators of the base classes will not be shown.
	 *
	 * @parameter expression="${doxygen.inlineInheritedMemb}" default-value="false"
	 */
	private boolean inlineInheritedMemb;

	/**
	 * If the FULL_PATH_NAMES tag is set to YES then Doxygen will prepend the full 
	 * path before files name in the file list and in the header files. If set 
	 * to NO the shortest path that makes the file name unique will be used.
	 *
	 * @parameter expression="${doxygen.fullPathNames}" default-value="true"
	 */
	private boolean fullPathNames;

	/**
	 * If the FULL_PATH_NAMES tag is set to YES then the STRIP_FROM_PATH tag 
	 * can be used to strip a user-defined part of the path. Stripping is 
	 * only done if one of the specified strings matches the left-hand part of 
	 * the path. The tag can be used to show relative paths in the file list. 
	 * If left blank the directory from which doxygen is run is used as the 
	 * path to strip.
	 *
	 * @parameter expression="${doxygen.stripFromPath}"
	 */
	private String stripFromPath;

	/**
	 * The STRIP_FROM_INC_PATH tag can be used to strip a user-defined part of 
	 * the path mentioned in the documentation of a class, which tells 
	 * the reader which header file to include in order to use a class. 
	 * If left blank only the name of the header file containing the class 
	 * definition is used. Otherwise one should specify the include paths that 
	 * are normally passed to the compiler using the -I flag.
	 *
	 * @parameter expression="${doxygen.stripFromIncPath}"
	 */
	private String stripFromIncPath;

	/**
	 * If the SHORT_NAMES tag is set to YES, doxygen will generate much shorter 
	 * (but less readable) file names. This can be useful is your file systems 
	 * doesn't support long names like on DOS, Mac, or CD-ROM.
	 *
	 * @parameter expression="${doxygen.shortNames}" default-value="false"
	 */
	private boolean shortNames;

	/**
	 * If the JAVADOC_AUTOBRIEF tag is set to YES then Doxygen 
	 * will interpret the first line (until the first dot) of a JavaDoc-style 
	 * comment as the brief description. If set to NO, the JavaDoc 
	 * comments will behave just like regular Qt-style comments 
	 * (thus requiring an explicit @brief command for a brief description.)
	 *
	 * @parameter expression="${doxygen.javadocAutobrief}" default-value="false"
	 */
	private boolean javadocAutobrief;

	/**
	 * If the QT_AUTOBRIEF tag is set to YES then Doxygen will 
	 * interpret the first line (until the first dot) of a Qt-style 
	 * comment as the brief description. If set to NO, the comments 
	 * will behave just like regular Qt-style comments (thus requiring 
	 * an explicit \brief command for a brief description.)
	 *
	 * @parameter expression="${doxygen.qtAutobrief}" default-value="false"
	 */
	private boolean qtAutobrief;

	/**
	 * The MULTILINE_CPP_IS_BRIEF tag can be set to YES to make Doxygen 
	 * treat a multi-line C++ special comment block (i.e. a block of //! or /// 
	 * comments) as a brief description. This used to be the default behaviour. 
	 * The new default is to treat a multi-line C++ comment block as a detailed 
	 * description. Set this tag to YES if you prefer the old behaviour instead.
	 *
	 * @parameter expression="${doxygen.multilineCppIsBrief}" default-value="false"
	 */
	private boolean multilineCppIsBrief;

	/**
	 * If the INHERIT_DOCS tag is set to YES (the default) then an undocumented 
	 * member inherits the documentation from any documented member that it 
	 * re-implements.
	 *
	 * @parameter expression="${doxygen.inheritDocs}" default-value="true"
	 */
	private boolean inheritDocs;

	/**
	 * If the SEPARATE_MEMBER_PAGES tag is set to YES, then doxygen will produce 
	 * a new page for each member. If set to NO, the documentation of a member will 
	 * be part of the file/class/namespace that contains it.
	 *
	 * @parameter expression="${doxygen.separateMemberPages}" default-value="false"
	 */
	private boolean separateMemberPages;

	/**
	 * The TAB_SIZE tag can be used to set the number of spaces in a tab. 
	 * Doxygen uses this value to replace tabs by spaces in code fragments.
	 *
	 * @parameter expression="${doxygen.tabSize}" default-value="8"
	 */
	private Integer tabSize;

	/**
	 * This tag can be used to specify a number of aliases that acts 
	 * as commands in the documentation. An alias has the form "name=value". 
	 * For example adding "sideeffect=\par Side Effects:\n" will allow you to 
	 * put the command \sideeffect (or @sideeffect) in the documentation, which 
	 * will result in a user-defined paragraph with heading "Side Effects:". 
	 * You can put \n's in the value part of an alias to insert newlines.
	 *
	 * @parameter expression="${doxygen.aliases}"
	 */
	private String aliases;

	/**
	 * Set the OPTIMIZE_OUTPUT_FOR_C tag to YES if your project consists of C 
	 * sources only. Doxygen will then generate output that is more tailored for C. 
	 * For instance, some of the names that are used will be different. The list 
	 * of all members will be omitted, etc.
	 *
	 * @parameter expression="${doxygen.optimizeOutputForC}" default-value="false"
	 */
	private boolean optimizeOutputForC;

	/**
	 * Set the OPTIMIZE_OUTPUT_JAVA tag to YES if your project consists of Java 
	 * sources only. Doxygen will then generate output that is more tailored for 
	 * Java. For instance, namespaces will be presented as packages, qualified 
	 * scopes will look different, etc.
	 *
	 * @parameter expression="${doxygen.optimizeOutputJava}" default-value="false"
	 */
	private boolean optimizeOutputJava;

	/**
	 * Set the OPTIMIZE_FOR_FORTRAN tag to YES if your project consists of Fortran 
	 * sources only. Doxygen will then generate output that is more tailored for 
	 * Fortran.
	 *
	 * @parameter expression="${doxygen.optimizeForFortran}" default-value="false"
	 */
	private boolean optimizeForFortran;

	/**
	 * Set the OPTIMIZE_OUTPUT_VHDL tag to YES if your project consists of VHDL 
	 * sources. Doxygen will then generate output that is tailored for 
	 * VHDL.
	 *
	 * @parameter expression="${doxygen.optimizeOutputVhdl}" default-value="false"
	 */
	private boolean optimizeOutputVhdl;

	/**
	 * If you use STL classes (i.e. std::string, std::vector, etc.) but do not want 
	 * to include (a tag file for) the STL sources as input, then you should 
	 * set this tag to YES in order to let doxygen match functions declarations and 
	 * definitions whose arguments contain STL classes (e.g. func(std::string); v.s. 
	 * func(std::string) {}). This also make the inheritance and collaboration 
	 * diagrams that involve STL classes more complete and accurate.
	 *
	 * @parameter expression="${doxygen.builtinStlSupport}" default-value="true"
	 */
	private boolean builtinStlSupport;

	/**
	 * If you use Microsoft's C++/CLI language, you should set this option to YES to
	 * enable parsing support.
	 *
	 * @parameter expression="${doxygen.cppCliSupport}" default-value="false"
	 */
	private boolean cppCliSupport;

	/**
	 * Set the SIP_SUPPORT tag to YES if your project consists of sip sources only. 
	 * Doxygen will parse them like normal C++ but will assume all classes use public 
	 * instead of private inheritance when no explicit protection keyword is present.
	 *
	 * @parameter expression="${doxygen.sipSupport}" default-value="false"
	 */
	private boolean sipSupport;

	/**
	 * For Microsoft's IDL there are propget and propput attributes to indicate getter 
	 * and setter methods for a property. Setting this option to YES (the default) 
	 * will make doxygen to replace the get and set methods by a property in the 
	 * documentation. This will only work if the methods are indeed getting or 
	 * setting a simple type. If this is not the case, or you want to show the 
	 * methods anyway, you should set this option to NO.
	 *
	 * @parameter expression="${doxygen.idlPropertySupport}" default-value="true"
	 */
	private boolean idlPropertySupport;

	/**
	 * If member grouping is used in the documentation and the DISTRIBUTE_GROUP_DOC 
	 * tag is set to YES, then doxygen will reuse the documentation of the first 
	 * member in the group (if any) for the other members of the group. By default 
	 * all members of a group must be documented explicitly.
	 *
	 * @parameter expression="${doxygen.distributeGroupDoc}" default-value="false"
	 */
	private boolean distributeGroupDoc;

	/**
	 * Set the SUBGROUPING tag to YES (the default) to allow class member groups of 
	 * the same type (for instance a group of public functions) to be put as a 
	 * subgroup of that type (e.g. under the Public Functions section). Set it to 
	 * NO to prevent subgrouping. Alternatively, this can be done per class using 
	 * the \nosubgrouping command.
	 *
	 * @parameter expression="${doxygen.subgrouping}" default-value="true"
	 */
	private boolean subgrouping;

	/**
	 * When TYPEDEF_HIDES_STRUCT is enabled, a typedef of a struct, union, or enum 
	 * is documented as struct, union, or enum with the name of the typedef. So 
	 * typedef struct TypeS {} TypeT, will appear in the documentation as a struct 
	 * with name TypeT. When disabled the typedef will appear as a member of a file, 
	 * namespace, or class. And the struct will be named TypeS. This can typically 
	 * be useful for C code in case the coding convention dictates that all compound 
	 * types are typedef'ed and only the typedef is referenced, never the tag name.
	 *
	 * @parameter expression="${doxygen.typedefHidesStruct}" default-value="false"
	 */
	private boolean typedefHidesStruct;

	/**
	 * The SYMBOL_CACHE_SIZE determines the size of the internal cache use to 
	 * determine which symbols to keep in memory and which to flush to disk.
	 * When the cache is full, less often used symbols will be written to disk.
	 * For small to medium size projects (&lt;1000 input files) the default value is 
	 * probably good enough. For larger projects a too small cache size can cause 
	 * doxygen to be busy swapping symbols to and from disk most of the time 
	 * causing a significant performance penality. 
	 * If the system has enough physical memory increasing the cache will improve the 
	 * performance by keeping more symbols in memory. Note that the value works on 
	 * a logarithmic scale so increasing the size by one will rougly double the 
	 * memory usage. The cache size is given by this formula: 
	 * 2^(16+SYMBOL_CACHE_SIZE). The valid range is 0..9, the default is 0, 
	 * corresponding to a cache size of 2^16 = 65536 symbols
	 *
	 * @parameter expression="${doxygen.symbolCacheSize}" default-value="0"
	 */
	private Integer symbolCacheSize;

	/**
	 * If the EXTRACT_ALL tag is set to YES doxygen will assume all entities in 
	 * documentation are documented, even if no documentation was available. 
	 * Private class members and static file members will be hidden unless 
	 * the EXTRACT_PRIVATE and EXTRACT_STATIC tags are set to YES
	 *
	 * @parameter expression="${doxygen.extractAll}" default-value="false"
	 */
	private boolean extractAll;

	/**
	 * If the EXTRACT_PRIVATE tag is set to YES all private members of a class 
	 * will be included in the documentation.
	 *
	 * @parameter expression="${doxygen.extractPrivate}" default-value="false"
	 */
	private boolean extractPrivate;

	/**
	 * If the EXTRACT_STATIC tag is set to YES all static members of a file 
	 * will be included in the documentation.
	 *
	 * @parameter expression="${doxygen.extractStatic}" default-value="false"
	 */
	private boolean extractStatic;

	/**
	 * If the EXTRACT_LOCAL_CLASSES tag is set to YES classes (and structs) 
	 * defined locally in source files will be included in the documentation. 
	 * If set to NO only classes defined in header files are included.
	 *
	 * @parameter expression="${doxygen.extractLocalClasses}" default-value="true"
	 */
	private boolean extractLocalClasses;

	/**
	 * This flag is only useful for Objective-C code. When set to YES local 
	 * methods, which are defined in the implementation section but not in 
	 * the interface are included in the documentation. 
	 * If set to NO (the default) only methods in the interface are included.
	 *
	 * @parameter expression="${doxygen.extractLocalMethods}" default-value="false"
	 */
	private boolean extractLocalMethods;

	/**
	 * If this flag is set to YES, the members of anonymous namespaces will be 
	 * extracted and appear in the documentation as a namespace called 
	 * 'anonymous_namespace{file}', where file will be replaced with the base 
	 * name of the file that contains the anonymous namespace. By default 
	 * anonymous namespace are hidden.
	 *
	 * @parameter expression="${doxygen.extractAnonNspaces}" default-value="false"
	 */
	private boolean extractAnonNspaces;

	/**
	 * If the HIDE_UNDOC_MEMBERS tag is set to YES, Doxygen will hide all 
	 * undocumented members of documented classes, files or namespaces. 
	 * If set to NO (the default) these members will be included in the 
	 * various overviews, but no documentation section is generated. 
	 * This option has no effect if EXTRACT_ALL is enabled.
	 *
	 * @parameter expression="${doxygen.hideUndocMembers}" default-value="false"
	 */
	private boolean hideUndocMembers;

	/**
	 * If the HIDE_UNDOC_CLASSES tag is set to YES, Doxygen will hide all 
	 * undocumented classes that are normally visible in the class hierarchy. 
	 * If set to NO (the default) these classes will be included in the various 
	 * overviews. This option has no effect if EXTRACT_ALL is enabled.
	 *
	 * @parameter expression="${doxygen.hideUndocClasses}" default-value="false"
	 */
	private boolean hideUndocClasses;

	/**
	 * If the HIDE_FRIEND_COMPOUNDS tag is set to YES, Doxygen will hide all 
	 * friend (class|struct|union) declarations. 
	 * If set to NO (the default) these declarations will be included in the 
	 * documentation.
	 *
	 * @parameter expression="${doxygen.hideFriendCompounds}" default-value="false"
	 */
	private boolean hideFriendCompounds;

	/**
	 * If the HIDE_IN_BODY_DOCS tag is set to YES, Doxygen will hide any 
	 * documentation blocks found inside the body of a function. 
	 * If set to NO (the default) these blocks will be appended to the 
	 * function's detailed documentation block.
	 *
	 * @parameter expression="${doxygen.hideInBodyDocs}" default-value="false"
	 */
	private boolean hideInBodyDocs;

	/**
	 * The INTERNAL_DOCS tag determines if documentation 
	 * that is typed after a \internal command is included. If the tag is set 
	 * to NO (the default) then the documentation will be excluded. 
	 * Set it to YES to include the internal documentation.
	 *
	 * @parameter expression="${doxygen.internalDocs}" default-value="false"
	 */
	private boolean internalDocs;

	/**
	 * If the CASE_SENSE_NAMES tag is set to NO then Doxygen will only generate 
	 * file names in lower-case letters. If set to YES upper-case letters are also 
	 * allowed. This is useful if you have classes or files whose names only differ 
	 * in case and if your file system supports case sensitive file names. Windows 
	 * and Mac users are advised to set this option to NO.
	 *
	 * @parameter expression="${doxygen.caseSenseNames}" default-value="true"
	 */
	private boolean caseSenseNames;

	/**
	 * If the HIDE_SCOPE_NAMES tag is set to NO (the default) then Doxygen 
	 * will show members with their full class and namespace scopes in the 
	 * documentation. If set to YES the scope will be hidden.
	 *
	 * @parameter expression="${doxygen.hideScopeNames}" default-value="false"
	 */
	private boolean hideScopeNames;

	/**
	 * If the SHOW_INCLUDE_FILES tag is set to YES (the default) then Doxygen 
	 * will put a list of the files that are included by a file in the documentation 
	 * of that file.
	 *
	 * @parameter expression="${doxygen.showIncludeFiles}" default-value="true"
	 */
	private boolean showIncludeFiles;

	/**
	 * If the INLINE_INFO tag is set to YES (the default) then a tag [inline] 
	 * is inserted in the documentation for inline members.
	 *
	 * @parameter expression="${doxygen.inlineInfo}" default-value="true"
	 */
	private boolean inlineInfo;

	/**
	 * If the SORT_MEMBER_DOCS tag is set to YES (the default) then doxygen 
	 * will sort the (detailed) documentation of file and class members 
	 * alphabetically by member name. If set to NO the members will appear in 
	 * declaration order.
	 *
	 * @parameter expression="${doxygen.sortMemberDocs}" default-value="true"
	 */
	private boolean sortMemberDocs;

	/**
	 * If the SORT_BRIEF_DOCS tag is set to YES then doxygen will sort the 
	 * brief documentation of file, namespace and class members alphabetically 
	 * by member name. If set to NO (the default) the members will appear in 
	 * declaration order.
	 *
	 * @parameter expression="${doxygen.sortBriefDocs}" default-value="false"
	 */
	private boolean sortBriefDocs;

	/**
	 * If the SORT_GROUP_NAMES tag is set to YES then doxygen will sort the 
	 * hierarchy of group names into alphabetical order. If set to NO (the default) 
	 * the group names will appear in their defined order.
	 *
	 * @parameter expression="${doxygen.sortGroupNames}" default-value="false"
	 */
	private boolean sortGroupNames;

	/**
	 * If the SORT_BY_SCOPE_NAME tag is set to YES, the class list will be 
	 * sorted by fully-qualified names, including namespaces. If set to 
	 * NO (the default), the class list will be sorted only by class name, 
	 * not including the namespace part. 
	 * Note: This option is not very useful if HIDE_SCOPE_NAMES is set to YES.
	 * Note: This option applies only to the class list, not to the 
	 * alphabetical list.
	 *
	 * @parameter expression="${doxygen.sortByScopeName}" default-value="false"
	 */
	private boolean sortByScopeName;

	/**
	 * The GENERATE_TODOLIST tag can be used to enable (YES) or 
	 * disable (NO) the todo list. This list is created by putting \todo 
	 * commands in the documentation.
	 *
	 * @parameter expression="${doxygen.generateTodolist}" default-value="true"
	 */
	private boolean generateTodolist;

	/**
	 * The GENERATE_TESTLIST tag can be used to enable (YES) or 
	 * disable (NO) the test list. This list is created by putting \test 
	 * commands in the documentation.
	 *
	 * @parameter expression="${doxygen.generateTestlist}" default-value="true"
	 */
	private boolean generateTestlist;

	/**
	 * The GENERATE_BUGLIST tag can be used to enable (YES) or 
	 * disable (NO) the bug list. This list is created by putting \bug 
	 * commands in the documentation.
	 *
	 * @parameter expression="${doxygen.generateBuglist}" default-value="true"
	 */
	private boolean generateBuglist;

	/**
	 * The GENERATE_DEPRECATEDLIST tag can be used to enable (YES) or 
	 * disable (NO) the deprecated list. This list is created by putting 
	 * \deprecated commands in the documentation.
	 *
	 * @parameter expression="${doxygen.generateDeprecatedlist}" default-value="true"
	 */
	private boolean generateDeprecatedlist;

	/**
	 * The ENABLED_SECTIONS tag can be used to enable conditional 
	 * documentation sections, marked by \if sectionname ... \endif.
	 *
	 * @parameter expression="${doxygen.enabledSections}"
	 */
	private String enabledSections;

	/**
	 * The MAX_INITIALIZER_LINES tag determines the maximum number of lines 
	 * the initial value of a variable or define consists of for it to appear in 
	 * the documentation. If the initializer consists of more lines than specified 
	 * here it will be hidden. Use a value of 0 to hide initializers completely. 
	 * The appearance of the initializer of individual variables and defines in the 
	 * documentation can be controlled using \showinitializer or \hideinitializer 
	 * command in the documentation regardless of this setting.
	 *
	 * @parameter expression="${doxygen.maxInitializerLines}" default-value="30"
	 */
	private Integer maxInitializerLines;

	/**
	 * Set the SHOW_USED_FILES tag to NO to disable the list of files generated 
	 * at the bottom of the documentation of classes and structs. If set to YES the 
	 * list will mention the files that were used to generate the documentation.
	 *
	 * @parameter expression="${doxygen.showUsedFiles}" default-value="true"
	 */
	private boolean showUsedFiles;

	/**
	 * If the sources in your project are distributed over multiple directories 
	 * then setting the SHOW_DIRECTORIES tag to YES will show the directory hierarchy 
	 * in the documentation. The default is NO.
	 *
	 * @parameter expression="${doxygen.showDirectories}" default-value="true"
	 */
	private boolean showDirectories;

	/**
	 * Set the SHOW_FILES tag to NO to disable the generation of the Files page.
	 * This will remove the Files entry from the Quick Index and from the 
	 * Folder Tree View (if specified). The default is YES.
	 *
	 * @parameter expression="${doxygen.showFiles}" default-value="true"
	 */
	private boolean showFiles;

	/**
	 * Set the SHOW_NAMESPACES tag to NO to disable the generation of the 
	 * Namespaces page.  This will remove the Namespaces entry from the Quick Index
	 * and from the Folder Tree View (if specified). The default is YES.
	 *
	 * @parameter expression="${doxygen.showNamespaces}" default-value="true"
	 */
	private boolean showNamespaces;

	/**
	 * The FILE_VERSION_FILTER tag can be used to specify a program or script that 
	 * doxygen should invoke to get the current version for each file (typically from 
	 * the version control system). Doxygen will invoke the program by executing (via 
	 * popen()) the command &lt;command&gt; &lt;input-file&gt;, where &lt;command&gt; is the value of 
	 * the FILE_VERSION_FILTER tag, and &lt;input-file&gt; is the name of an input file 
	 * provided by doxygen. Whatever the program writes to standard output 
	 * is used as the file version. See the manual for examples.
	 *
	 * @parameter expression="${doxygen.fileVersionFilter}"
	 */
	private String fileVersionFilter;

	/**
	 * The LAYOUT_FILE tag can be used to specify a layout file which will be parsed by 
	 * doxygen. The layout file controls the global structure of the generated output files 
	 * in an output format independent way. The create the layout file that represents 
	 * doxygen's defaults, run doxygen with the -l option. You can optionally specify a 
	 * file name after the option, if omitted DoxygenLayout.xml will be used as the name 
	 * of the layout file.
	 *
	 * @parameter expression="${doxygen.layoutFile}"
	 */
	private String layoutFile;

	/**
	 * The QUIET tag can be used to turn on/off the messages that are generated 
	 * by doxygen. Possible values are YES and NO. If left blank NO is used.
	 *
	 * @parameter expression="${doxygen.quiet}" default-value="true"
	 */
	private boolean quiet;

	/**
	 * The WARNINGS tag can be used to turn on/off the warning messages that are 
	 * generated by doxygen. Possible values are YES and NO. If left blank 
	 * NO is used.
	 *
	 * @parameter expression="${doxygen.warnings}" default-value="true"
	 */
	private boolean warnings;

	/**
	 * If WARN_IF_UNDOCUMENTED is set to YES, then doxygen will generate warnings 
	 * for undocumented members. If EXTRACT_ALL is set to YES then this flag will 
	 * automatically be disabled.
	 *
	 * @parameter expression="${doxygen.warnIfUndocumented}" default-value="true"
	 */
	private boolean warnIfUndocumented;

	/**
	 * If WARN_IF_DOC_ERROR is set to YES, doxygen will generate warnings for 
	 * potential errors in the documentation, such as not documenting some 
	 * parameters in a documented function, or documenting parameters that 
	 * don't exist or using markup commands wrongly.
	 *
	 * @parameter expression="${doxygen.warnIfDocError}" default-value="true"
	 */
	private boolean warnIfDocError;

	/**
	 * This WARN_NO_PARAMDOC option can be abled to get warnings for 
	 * functions that are documented, but have no documentation for their parameters 
	 * or return value. If set to NO (the default) doxygen will only warn about 
	 * wrong or incomplete parameter documentation, but not about the absence of 
	 * documentation.
	 *
	 * @parameter expression="${doxygen.warnNoParamdoc}" default-value="true"
	 */
	private boolean warnNoParamdoc;

	/**
	 * The WARN_FORMAT tag determines the format of the warning messages that 
	 * doxygen can produce. The string should contain the $file, $line, and $text 
	 * tags, which will be replaced by the file and line number from which the 
	 * warning originated and the warning text. Optionally the format may contain 
	 * $version, which will be replaced by the version of the file (if it could 
	 * be obtained via FILE_VERSION_FILTER)
	 *
	 * @parameter expression="${doxygen.warnFormat}" default-value=""$file:$line: $text""
	 */
	private String warnFormat;

	/**
	 * The WARN_LOGFILE tag can be used to specify a file to which warning 
	 * and error messages should be written. If left blank the output is written 
	 * to stderr.
	 *
	 * @parameter expression="${doxygen.warnLogfile}"
	 */
	private String warnLogfile;

	/**
	 * The INPUT tag can be used to specify the files and/or directories that contain 
	 * documented source files. You may enter file names like "myfile.cpp" or 
	 * directories like "/usr/src/myproject". Separate the files or directories 
	 * with spaces.
	 *
	 * @parameter expression="${doxygen.input}"
	 */
	private String input;

	/**
	 * This tag can be used to specify the character encoding of the source files 
	 * that doxygen parses. Internally doxygen uses the UTF-8 encoding, which is 
	 * also the default input encoding. Doxygen uses libiconv (or the iconv built 
	 * into libc) for the transcoding. See http://www.gnu.org/software/libiconv for 
	 * the list of possible encodings.
	 *
	 * @parameter expression="${doxygen.inputEncoding}" default-value="UTF-8"
	 */
	private String inputEncoding;

	/**
	 * If the value of the INPUT tag contains directories, you can use the 
	 * FILE_PATTERNS tag to specify one or more wildcard pattern (like *.cpp 
	 * and *.h) to filter out the source-files in the directories. If left 
	 * blank the following patterns are tested: 
	 * *.c *.cc *.cxx *.cpp *.c++ *.java *.ii *.ixx *.ipp *.i++ *.inl *.h *.hh *.hxx 
	 * *.hpp *.h++ *.idl *.odl *.cs *.php *.php3 *.inc *.m *.mm *.py *.f90
	 *
	 * @parameter expression="${doxygen.filePatterns}" default-value="*.doxygen *.doxy *.cpp *.cxx *.hpp *.hxx *.ipp *.ixx *.py *.java"
	 */
	private String filePatterns;

	/**
	 * The RECURSIVE tag can be used to turn specify whether or not subdirectories 
	 * should be searched for input files as well. Possible values are YES and NO. 
	 * If left blank NO is used.
	 *
	 * @parameter expression="${doxygen.recursive}" default-value="false"
	 */
	private boolean recursive;

	/**
	 * The EXCLUDE tag can be used to specify files and/or directories that should 
	 * excluded from the INPUT source files. This way you can easily exclude a 
	 * subdirectory from a directory tree whose root is specified with the INPUT tag.
	 *
	 * @parameter expression="${doxygen.exclude}" default-value=".svn"
	 */
	private String exclude;

	/**
	 * The EXCLUDE_SYMLINKS tag can be used select whether or not files or 
	 * directories that are symbolic links (a Unix filesystem feature) are excluded 
	 * from the input.
	 *
	 * @parameter expression="${doxygen.excludeSymlinks}" default-value="false"
	 */
	private boolean excludeSymlinks;

	/**
	 * If the value of the INPUT tag contains directories, you can use the 
	 * EXCLUDE_PATTERNS tag to specify one or more wildcard patterns to exclude 
	 * certain files from those directories. Note that the wildcards are matched 
	 * against the file with absolute path, so to exclude all test directories 
	 * for example use the pattern &x2a/test/&x2a
	 *
	 * @parameter expression="${doxygen.excludePatterns}"
	 */
	private String excludePatterns;

	/**
	 * The EXCLUDE_SYMBOLS tag can be used to specify one or more symbol names 
	 * (namespaces, classes, functions, etc.) that should be excluded from the 
	 * output. The symbol name can be a fully qualified name, a word, or if the 
	 * wildcard * is used, a substring. Examples: ANamespace, AClass, 
	 * AClass::ANamespace, ANamespace::*Test
	 *
	 * @parameter expression="${doxygen.excludeSymbols}"
	 */
	private String excludeSymbols;

	/**
	 * The EXAMPLE_PATH tag can be used to specify one or more files or 
	 * directories that contain example code fragments that are included (see 
	 * the \include command).
	 *
	 * @parameter expression="${doxygen.examplePath}"
	 */
	private String examplePath;

	/**
	 * If the value of the EXAMPLE_PATH tag contains directories, you can use the 
	 * EXAMPLE_PATTERNS tag to specify one or more wildcard pattern (like *.cpp 
	 * and *.h) to filter out the source-files in the directories. If left 
	 * blank all files are included.
	 *
	 * @parameter expression="${doxygen.examplePatterns}"
	 */
	private String examplePatterns;

	/**
	 * If the EXAMPLE_RECURSIVE tag is set to YES then subdirectories will be 
	 * searched for input files to be used with the \include or \dontinclude 
	 * commands irrespective of the value of the RECURSIVE tag. 
	 * Possible values are YES and NO. If left blank NO is used.
	 *
	 * @parameter expression="${doxygen.exampleRecursive}" default-value="false"
	 */
	private boolean exampleRecursive;

	/**
	 * The IMAGE_PATH tag can be used to specify one or more files or 
	 * directories that contain image that are included in the documentation (see 
	 * the \image command).
	 *
	 * @parameter expression="${doxygen.imagePath}"
	 */
	private String imagePath;

	/**
	 * The INPUT_FILTER tag can be used to specify a program that doxygen should 
	 * invoke to filter for each input file. Doxygen will invoke the filter program 
	 * by executing (via popen()) the command &lt;filter&gt; &lt;input-file&gt;, where &lt;filter&gt; 
	 * is the value of the INPUT_FILTER tag, and &lt;input-file&gt; is the name of an 
	 * input file. Doxygen will then use the output that the filter program writes 
	 * to standard output.  If FILTER_PATTERNS is specified, this tag will be 
	 * ignored.
	 *
	 * @parameter expression="${doxygen.inputFilter}"
	 */
	private String inputFilter;

	/**
	 * The FILTER_PATTERNS tag can be used to specify filters on a per file pattern 
	 * basis.  Doxygen will compare the file name with each pattern and apply the 
	 * filter if there is a match.  The filters are a list of the form: 
	 * pattern=filter (like *.cpp=my_cpp_filter). See INPUT_FILTER for further 
	 * info on how filters are used. If FILTER_PATTERNS is empty, INPUT_FILTER 
	 * is applied to all files.
	 *
	 * @parameter expression="${doxygen.filterPatterns}"
	 */
	private String filterPatterns;

	/**
	 * If the FILTER_SOURCE_FILES tag is set to YES, the input filter (if set using 
	 * INPUT_FILTER) will be used to filter the input files when producing source 
	 * files to browse (i.e. when SOURCE_BROWSER is set to YES).
	 *
	 * @parameter expression="${doxygen.filterSourceFiles}" default-value="false"
	 */
	private boolean filterSourceFiles;

	/**
	 * If the SOURCE_BROWSER tag is set to YES then a list of source files will 
	 * be generated. Documented entities will be cross-referenced with these sources. 
	 * Note: To get rid of all source code in the generated output, make sure also 
	 * VERBATIM_HEADERS is set to NO.
	 *
	 * @parameter expression="${doxygen.sourceBrowser}" default-value="true"
	 */
	private boolean sourceBrowser;

	/**
	 * Setting the INLINE_SOURCES tag to YES will include the body 
	 * of functions and classes directly in the documentation.
	 *
	 * @parameter expression="${doxygen.inlineSources}" default-value="false"
	 */
	private boolean inlineSources;

	/**
	 * Setting the STRIP_CODE_COMMENTS tag to YES (the default) will instruct 
	 * doxygen to hide any special comment blocks from generated source code 
	 * fragments. Normal C and C++ comments will always remain visible.
	 *
	 * @parameter expression="${doxygen.stripCodeComments}" default-value="true"
	 */
	private boolean stripCodeComments;

	/**
	 * If the REFERENCED_BY_RELATION tag is set to YES 
	 * then for each documented function all documented 
	 * functions referencing it will be listed.
	 *
	 * @parameter expression="${doxygen.referencedByRelation}" default-value="false"
	 */
	private boolean referencedByRelation;

	/**
	 * If the REFERENCES_RELATION tag is set to YES 
	 * then for each documented function all documented entities 
	 * called/used by that function will be listed.
	 *
	 * @parameter expression="${doxygen.referencesRelation}" default-value="false"
	 */
	private boolean referencesRelation;

	/**
	 * If the REFERENCES_LINK_SOURCE tag is set to YES (the default)
	 * and SOURCE_BROWSER tag is set to YES, then the hyperlinks from
	 * functions in REFERENCES_RELATION and REFERENCED_BY_RELATION lists will
	 * link to the source code.  Otherwise they will link to the documentstion.
	 *
	 * @parameter expression="${doxygen.referencesLinkSource}" default-value="true"
	 */
	private boolean referencesLinkSource;

	/**
	 * If the USE_HTAGS tag is set to YES then the references to source code 
	 * will point to the HTML generated by the htags(1) tool instead of doxygen 
	 * built-in source browser. The htags tool is part of GNU's global source 
	 * tagging system (see http://www.gnu.org/software/global/global.html). You 
	 * will need version 4.8.6 or higher.
	 *
	 * @parameter expression="${doxygen.useHtags}" default-value="false"
	 */
	private boolean useHtags;

	/**
	 * If the VERBATIM_HEADERS tag is set to YES (the default) then Doxygen 
	 * will generate a verbatim copy of the header file for each class for 
	 * which an include is specified. Set to NO to disable this.
	 *
	 * @parameter expression="${doxygen.verbatimHeaders}" default-value="true"
	 */
	private boolean verbatimHeaders;

	/**
	 * If the ALPHABETICAL_INDEX tag is set to YES, an alphabetical index 
	 * of all compounds will be generated. Enable this if the project 
	 * contains a lot of classes, structs, unions or interfaces.
	 *
	 * @parameter expression="${doxygen.alphabeticalIndex}" default-value="true"
	 */
	private boolean alphabeticalIndex;

	/**
	 * If the alphabetical index is enabled (see ALPHABETICAL_INDEX) then 
	 * the COLS_IN_ALPHA_INDEX tag can be used to specify the number of columns 
	 * in which this list will be split (can be a number in the range [1..20])
	 *
	 * @parameter expression="${doxygen.colsInAlphaIndex}" default-value="5"
	 */
	private Integer colsInAlphaIndex;

	/**
	 * In case all classes in a project start with a common prefix, all 
	 * classes will be put under the same header in the alphabetical index. 
	 * The IGNORE_PREFIX tag can be used to specify one or more prefixes that 
	 * should be ignored while generating the index headers.
	 *
	 * @parameter expression="${doxygen.ignorePrefix}"
	 */
	private String ignorePrefix;

	/**
	 * If the GENERATE_HTML tag is set to YES (the default) Doxygen will 
	 * generate HTML output.
	 *
	 * @parameter expression="${doxygen.generateHtml}" default-value="true"
	 */
	private boolean generateHtml;

	/**
	 * The HTML_OUTPUT tag is used to specify where the HTML docs will be put. 
	 * If a relative path is entered the value of OUTPUT_DIRECTORY will be 
	 * put in front of it. If left blank `html' will be used as the default path.
	 *
	 * @parameter expression="${doxygen.htmlOutput}" default-value="doxygen"
	 */
	private String htmlOutput;

	/**
	 * The HTML_FILE_EXTENSION tag can be used to specify the file extension for 
	 * each generated HTML page (for example: .htm,.php,.asp). If it is left blank 
	 * doxygen will generate files with .html extension.
	 *
	 * @parameter expression="${doxygen.htmlFileExtension}" default-value=".html"
	 */
	private String htmlFileExtension;

	/**
	 * The HTML_HEADER tag can be used to specify a personal HTML header for 
	 * each generated HTML page. If it is left blank doxygen will generate a 
	 * standard header.
	 *
	 * @parameter expression="${doxygen.htmlHeader}"
	 */
	private String htmlHeader;

	/**
	 * The HTML_FOOTER tag can be used to specify a personal HTML footer for 
	 * each generated HTML page. If it is left blank doxygen will generate a 
	 * standard footer.
	 *
	 * @parameter expression="${doxygen.htmlFooter}"
	 */
	private String htmlFooter;

	/**
	 * The HTML_STYLESHEET tag can be used to specify a user-defined cascading 
	 * style sheet that is used by each HTML page. It can be used to 
	 * fine-tune the look of the HTML output. If the tag is left blank doxygen 
	 * will generate a default style sheet. Note that doxygen will try to copy 
	 * the style sheet file to the HTML output directory, so don't put your own 
	 * stylesheet in the HTML output directory as well, or it will be erased!
	 *
	 * @parameter expression="${doxygen.htmlStylesheet}"
	 */
	private String htmlStylesheet;

	/**
	 * If the HTML_ALIGN_MEMBERS tag is set to YES, the members of classes, 
	 * files or namespaces will be aligned in HTML using tables. If set to 
	 * NO a bullet list will be used.
	 *
	 * @parameter expression="${doxygen.htmlAlignMembers}" default-value="true"
	 */
	private boolean htmlAlignMembers;

	/**
	 * If the HTML_DYNAMIC_SECTIONS tag is set to YES then the generated HTML 
	 * documentation will contain sections that can be hidden and shown after the 
	 * page has loaded. For this to work a browser that supports 
	 * JavaScript and DHTML is required (for instance Mozilla 1.0+, Firefox 
	 * Netscape 6.0+, Internet explorer 5.0+, Konqueror, or Safari).
	 *
	 * @parameter expression="${doxygen.htmlDynamicSections}" default-value="true"
	 */
	private boolean htmlDynamicSections;

	/**
	 * If the GENERATE_DOCSET tag is set to YES, additional index files 
	 * will be generated that can be used as input for Apple's Xcode 3 
	 * integrated development environment, introduced with OSX 10.5 (Leopard). 
	 * To create a documentation set, doxygen will generate a Makefile in the 
	 * HTML output directory. Running make will produce the docset in that 
	 * directory and running "make install" will install the docset in 
	 * ~/Library/Developer/Shared/Documentation/DocSets so that Xcode will find 
	 * it at startup. 
	 * See http://developer.apple.com/tools/creatingdocsetswithdoxygen.html for more information.
	 *
	 * @parameter expression="${doxygen.generateDocset}" default-value="false"
	 */
	private boolean generateDocset;

	/**
	 * When GENERATE_DOCSET tag is set to YES, this tag determines the name of the 
	 * feed. A documentation feed provides an umbrella under which multiple 
	 * documentation sets from a single provider (such as a company or product suite) 
	 * can be grouped.
	 *
	 * @parameter expression="${doxygen.docsetFeedname}" default-value=""Doxygen generated docs""
	 */
	private String docsetFeedname;

	/**
	 * When GENERATE_DOCSET tag is set to YES, this tag specifies a string that 
	 * should uniquely identify the documentation set bundle. This should be a 
	 * reverse domain-name style string, e.g. com.mycompany.MyDocSet. Doxygen 
	 * will append .docset to the name.
	 *
	 * @parameter expression="${doxygen.docsetBundleId}" default-value="org.doxygen.Project"
	 */
	private String docsetBundleId;

	/**
	 * If the GENERATE_HTMLHELP tag is set to YES, additional index files 
	 * will be generated that can be used as input for tools like the 
	 * Microsoft HTML help workshop to generate a compiled HTML help file (.chm) 
	 * of the generated HTML documentation.
	 *
	 * @parameter expression="${doxygen.generateHtmlhelp}" default-value="false"
	 */
	private boolean generateHtmlhelp;

	/**
	 * If the GENERATE_HTMLHELP tag is set to YES, the CHM_FILE tag can 
	 * be used to specify the file name of the resulting .chm file. You 
	 * can add a path in front of the file if the result should not be 
	 * written to the html output directory.
	 *
	 * @parameter expression="${doxygen.chmFile}"
	 */
	private String chmFile;

	/**
	 * If the GENERATE_HTMLHELP tag is set to YES, the HHC_LOCATION tag can 
	 * be used to specify the location (absolute path including file name) of 
	 * the HTML help compiler (hhc.exe). If non-empty doxygen will try to run 
	 * the HTML help compiler on the generated index.hhp.
	 *
	 * @parameter expression="${doxygen.hhcLocation}"
	 */
	private String hhcLocation;

	/**
	 * If the GENERATE_HTMLHELP tag is set to YES, the GENERATE_CHI flag 
	 * controls if a separate .chi index file is generated (YES) or that 
	 * it should be included in the master .chm file (NO).
	 *
	 * @parameter expression="${doxygen.generateChi}" default-value="false"
	 */
	private boolean generateChi;

	/**
	 * If the GENERATE_HTMLHELP tag is set to YES, the CHM_INDEX_ENCODING
	 * is used to encode HtmlHelp index (hhk), content (hhc) and project file
	 * content.
	 *
	 * @parameter expression="${doxygen.chmIndexEncoding}"
	 */
	private String chmIndexEncoding;

	/**
	 * If the GENERATE_HTMLHELP tag is set to YES, the BINARY_TOC flag 
	 * controls whether a binary table of contents is generated (YES) or a 
	 * normal table of contents (NO) in the .chm file.
	 *
	 * @parameter expression="${doxygen.binaryToc}" default-value="false"
	 */
	private boolean binaryToc;

	/**
	 * The TOC_EXPAND flag can be set to YES to add extra items for group members 
	 * to the contents of the HTML help documentation and to the tree view.
	 *
	 * @parameter expression="${doxygen.tocExpand}" default-value="false"
	 */
	private boolean tocExpand;

	/**
	 * If the GENERATE_QHP tag is set to YES and both QHP_NAMESPACE and QHP_VIRTUAL_FOLDER 
	 * are set, an additional index file will be generated that can be used as input for 
	 * Qt's qhelpgenerator to generate a Qt Compressed Help (.qch) of the generated 
	 * HTML documentation.
	 *
	 * @parameter expression="${doxygen.generateQhp}" default-value="false"
	 */
	private boolean generateQhp;

	/**
	 * If the QHG_LOCATION tag is specified, the QCH_FILE tag can 
	 * be used to specify the file name of the resulting .qch file. 
	 * The path specified is relative to the HTML output folder.
	 *
	 * @parameter expression="${doxygen.qchFile}"
	 */
	private String qchFile;

	/**
	 * The QHP_NAMESPACE tag specifies the namespace to use when generating 
	 * Qt Help Project output. For more information please see 
	 * &lt;a href="http://doc.trolltech.com/qthelpproject.html#namespace"&gt;Qt Help Project / Namespace&lt;/a&gt;.
	 *
	 * @parameter expression="${doxygen.qhpNamespace}" default-value="org.doxygen.Project"
	 */
	private String qhpNamespace;

	/**
	 * The QHP_VIRTUAL_FOLDER tag specifies the namespace to use when generating 
	 * Qt Help Project output. For more information please see 
	 * &lt;a href="http://doc.trolltech.com/qthelpproject.html#virtual-folders"&gt;Qt Help Project / Virtual Folders&lt;/a&gt;.
	 *
	 * @parameter expression="${doxygen.qhpVirtualFolder}" default-value="doc"
	 */
	private String qhpVirtualFolder;

	/**
	 * If the GENERATE_QHP tag is set to YES, the QHG_LOCATION tag can 
	 * be used to specify the location of Qt's qhelpgenerator. 
	 * If non-empty doxygen will try to run qhelpgenerator on the generated 
	 * .qhp file .
	 *
	 * @parameter expression="${doxygen.qhgLocation}"
	 */
	private String qhgLocation;

	/**
	 * The DISABLE_INDEX tag can be used to turn on/off the condensed index at 
	 * top of each HTML page. The value NO (the default) enables the index and 
	 * the value YES disables it.
	 *
	 * @parameter expression="${doxygen.disableIndex}" default-value="false"
	 */
	private boolean disableIndex;

	/**
	 * This tag can be used to set the number of enum values (range [1..20]) 
	 * that doxygen will group on one line in the generated HTML documentation.
	 *
	 * @parameter expression="${doxygen.enumValuesPerLine}" default-value="4"
	 */
	private Integer enumValuesPerLine;

	/**
	 * The GENERATE_TREEVIEW tag is used to specify whether a tree-like index
	 * structure should be generated to display hierarchical information.
	 * If the tag value is set to FRAME, a side panel will be generated
	 * containing a tree-like index structure (just like the one that 
	 * is generated for HTML Help). For this to work a browser that supports 
	 * JavaScript, DHTML, CSS and frames is required (for instance Mozilla 1.0+, 
	 * Netscape 6.0+, Internet explorer 5.0+, or Konqueror). Windows users are 
	 * probably better off using the HTML help feature. Other possible values 
	 * for this tag are: HIERARCHIES, which will generate the Groups, Directories,
	 * and Class Hierarchy pages using a tree view instead of an ordered list;
	 * ALL, which combines the behavior of FRAME and HIERARCHIES; and NONE, which
	 * disables this behavior completely. For backwards compatibility with previous
	 * releases of Doxygen, the values YES and NO are equivalent to FRAME and NONE
	 * respectively.
	 *
	 * @parameter expression="${doxygen.generateTreeview}" default-value="ALL"
	 */
	private String generateTreeview;

	/**
	 * If the treeview is enabled (see GENERATE_TREEVIEW) then this tag can be 
	 * used to set the initial width (in pixels) of the frame in which the tree 
	 * is shown.
	 *
	 * @parameter expression="${doxygen.treeviewWidth}" default-value="250"
	 */
	private Integer treeviewWidth;

	/**
	 * Use this tag to change the font size of Latex formulas included 
	 * as images in the HTML documentation. The default is 10. Note that 
	 * when you change the font size after a successful doxygen run you need 
	 * to manually remove any form_*.png images from the HTML output directory 
	 * to force them to be regenerated.
	 *
	 * @parameter expression="${doxygen.formulaFontsize}" default-value="10"
	 */
	private Integer formulaFontsize;

	/**
	 * If the GENERATE_LATEX tag is set to YES (the default) Doxygen will 
	 * generate Latex output.
	 *
	 * @parameter expression="${doxygen.generateLatex}" default-value="false"
	 */
	private boolean generateLatex;

	/**
	 * The LATEX_OUTPUT tag is used to specify where the LaTeX docs will be put. 
	 * If a relative path is entered the value of OUTPUT_DIRECTORY will be 
	 * put in front of it. If left blank `latex' will be used as the default path.
	 *
	 * @parameter expression="${doxygen.latexOutput}" default-value="latex"
	 */
	private String latexOutput;

	/**
	 * The LATEX_CMD_NAME tag can be used to specify the LaTeX command name to be 
	 * invoked. If left blank `latex' will be used as the default command name.
	 *
	 * @parameter expression="${doxygen.latexCmdName}" default-value="latex"
	 */
	private String latexCmdName;

	/**
	 * The MAKEINDEX_CMD_NAME tag can be used to specify the command name to 
	 * generate index for LaTeX. If left blank `makeindex' will be used as the 
	 * default command name.
	 *
	 * @parameter expression="${doxygen.makeindexCmdName}" default-value="makeindex"
	 */
	private String makeindexCmdName;

	/**
	 * If the COMPACT_LATEX tag is set to YES Doxygen generates more compact 
	 * LaTeX documents. This may be useful for small projects and may help to 
	 * save some trees in general.
	 *
	 * @parameter expression="${doxygen.compactLatex}" default-value="false"
	 */
	private boolean compactLatex;

	/**
	 * The PAPER_TYPE tag can be used to set the paper type that is used 
	 * by the printer. Possible values are: a4, a4wide, letter, legal and 
	 * executive. If left blank a4wide will be used.
	 *
	 * @parameter expression="${doxygen.paperType}" default-value="a4wide"
	 */
	private String paperType;

	/**
	 * The EXTRA_PACKAGES tag can be to specify one or more names of LaTeX 
	 * packages that should be included in the LaTeX output.
	 *
	 * @parameter expression="${doxygen.extraPackages}"
	 */
	private String extraPackages;

	/**
	 * The LATEX_HEADER tag can be used to specify a personal LaTeX header for 
	 * the generated latex document. The header should contain everything until 
	 * the first chapter. If it is left blank doxygen will generate a 
	 * standard header. Notice: only use this tag if you know what you are doing!
	 *
	 * @parameter expression="${doxygen.latexHeader}"
	 */
	private String latexHeader;

	/**
	 * If the PDF_HYPERLINKS tag is set to YES, the LaTeX that is generated 
	 * is prepared for conversion to pdf (using ps2pdf). The pdf file will 
	 * contain links (just like the HTML output) instead of page references 
	 * This makes the output suitable for online browsing using a pdf viewer.
	 *
	 * @parameter expression="${doxygen.pdfHyperlinks}" default-value="true"
	 */
	private boolean pdfHyperlinks;

	/**
	 * If the USE_PDFLATEX tag is set to YES, pdflatex will be used instead of 
	 * plain latex in the generated Makefile. Set this option to YES to get a 
	 * higher quality PDF documentation.
	 *
	 * @parameter expression="${doxygen.usePdflatex}" default-value="true"
	 */
	private boolean usePdflatex;

	/**
	 * If the LATEX_BATCHMODE tag is set to YES, doxygen will add the \\batchmode. 
	 * command to the generated LaTeX files. This will instruct LaTeX to keep 
	 * running if errors occur, instead of asking the user for help. 
	 * This option is also used when generating formulas in HTML.
	 *
	 * @parameter expression="${doxygen.latexBatchmode}" default-value="false"
	 */
	private boolean latexBatchmode;

	/**
	 * If LATEX_HIDE_INDICES is set to YES then doxygen will not 
	 * include the index chapters (such as File Index, Compound Index, etc.) 
	 * in the output.
	 *
	 * @parameter expression="${doxygen.latexHideIndices}" default-value="false"
	 */
	private boolean latexHideIndices;

	/**
	 * If the GENERATE_RTF tag is set to YES Doxygen will generate RTF output 
	 * The RTF output is optimized for Word 97 and may not look very pretty with 
	 * other RTF readers or editors.
	 *
	 * @parameter expression="${doxygen.generateRtf}" default-value="false"
	 */
	private boolean generateRtf;

	/**
	 * The RTF_OUTPUT tag is used to specify where the RTF docs will be put. 
	 * If a relative path is entered the value of OUTPUT_DIRECTORY will be 
	 * put in front of it. If left blank `rtf' will be used as the default path.
	 *
	 * @parameter expression="${doxygen.rtfOutput}" default-value="rtf"
	 */
	private String rtfOutput;

	/**
	 * If the COMPACT_RTF tag is set to YES Doxygen generates more compact 
	 * RTF documents. This may be useful for small projects and may help to 
	 * save some trees in general.
	 *
	 * @parameter expression="${doxygen.compactRtf}" default-value="false"
	 */
	private boolean compactRtf;

	/**
	 * If the RTF_HYPERLINKS tag is set to YES, the RTF that is generated 
	 * will contain hyperlink fields. The RTF file will 
	 * contain links (just like the HTML output) instead of page references. 
	 * This makes the output suitable for online browsing using WORD or other 
	 * programs which support those fields. 
	 * Note: wordpad (write) and others do not support links.
	 *
	 * @parameter expression="${doxygen.rtfHyperlinks}" default-value="false"
	 */
	private boolean rtfHyperlinks;

	/**
	 * Load stylesheet definitions from file. Syntax is similar to doxygen's 
	 * config file, i.e. a series of assignments. You only have to provide 
	 * replacements, missing definitions are set to their default value.
	 *
	 * @parameter expression="${doxygen.rtfStylesheetFile}"
	 */
	private String rtfStylesheetFile;

	/**
	 * Set optional variables used in the generation of an rtf document. 
	 * Syntax is similar to doxygen's config file.
	 *
	 * @parameter expression="${doxygen.rtfExtensionsFile}"
	 */
	private String rtfExtensionsFile;

	/**
	 * If the GENERATE_MAN tag is set to YES (the default) Doxygen will 
	 * generate man pages
	 *
	 * @parameter expression="${doxygen.generateMan}" default-value="false"
	 */
	private boolean generateMan;

	/**
	 * The MAN_OUTPUT tag is used to specify where the man pages will be put. 
	 * If a relative path is entered the value of OUTPUT_DIRECTORY will be 
	 * put in front of it. If left blank `man' will be used as the default path.
	 *
	 * @parameter expression="${doxygen.manOutput}" default-value="man"
	 */
	private String manOutput;

	/**
	 * The MAN_EXTENSION tag determines the extension that is added to 
	 * the generated man pages (default is the subroutine's section .3)
	 *
	 * @parameter expression="${doxygen.manExtension}" default-value=".3"
	 */
	private String manExtension;

	/**
	 * If the MAN_LINKS tag is set to YES and Doxygen generates man output, 
	 * then it will generate one additional man file for each entity 
	 * documented in the real man page(s). These additional files 
	 * only source the real man page, but without them the man command 
	 * would be unable to find the correct page. The default is NO.
	 *
	 * @parameter expression="${doxygen.manLinks}" default-value="false"
	 */
	private boolean manLinks;

	/**
	 * If the GENERATE_XML tag is set to YES Doxygen will 
	 * generate an XML file that captures the structure of 
	 * the code including all documentation.
	 *
	 * @parameter expression="${doxygen.generateXml}" default-value="false"
	 */
	private boolean generateXml;

	/**
	 * The XML_OUTPUT tag is used to specify where the XML pages will be put. 
	 * If a relative path is entered the value of OUTPUT_DIRECTORY will be 
	 * put in front of it. If left blank `xml' will be used as the default path.
	 *
	 * @parameter expression="${doxygen.xmlOutput}" default-value="xml"
	 */
	private String xmlOutput;

	/**
	 * The XML_SCHEMA tag can be used to specify an XML schema, 
	 * which can be used by a validating XML parser to check the 
	 * syntax of the XML files.
	 *
	 * @parameter expression="${doxygen.xmlSchema}"
	 */
	private String xmlSchema;

	/**
	 * The XML_DTD tag can be used to specify an XML DTD, 
	 * which can be used by a validating XML parser to check the 
	 * syntax of the XML files.
	 *
	 * @parameter expression="${doxygen.xmlDtd}"
	 */
	private String xmlDtd;

	/**
	 * If the XML_PROGRAMLISTING tag is set to YES Doxygen will 
	 * dump the program listings (including syntax highlighting 
	 * and cross-referencing information) to the XML output. Note that 
	 * enabling this will significantly increase the size of the XML output.
	 *
	 * @parameter expression="${doxygen.xmlProgramlisting}" default-value="true"
	 */
	private boolean xmlProgramlisting;

	/**
	 * If the GENERATE_AUTOGEN_DEF tag is set to YES Doxygen will 
	 * generate an AutoGen Definitions (see autogen.sf.net) file 
	 * that captures the structure of the code including all 
	 * documentation. Note that this feature is still experimental 
	 * and incomplete at the moment.
	 *
	 * @parameter expression="${doxygen.generateAutogenDef}" default-value="false"
	 */
	private boolean generateAutogenDef;

	/**
	 * If the GENERATE_PERLMOD tag is set to YES Doxygen will 
	 * generate a Perl module file that captures the structure of 
	 * the code including all documentation. Note that this 
	 * feature is still experimental and incomplete at the 
	 * moment.
	 *
	 * @parameter expression="${doxygen.generatePerlmod}" default-value="false"
	 */
	private boolean generatePerlmod;

	/**
	 * If the PERLMOD_LATEX tag is set to YES Doxygen will generate 
	 * the necessary Makefile rules, Perl scripts and LaTeX code to be able 
	 * to generate PDF and DVI output from the Perl module output.
	 *
	 * @parameter expression="${doxygen.perlmodLatex}" default-value="false"
	 */
	private boolean perlmodLatex;

	/**
	 * If the PERLMOD_PRETTY tag is set to YES the Perl module output will be 
	 * nicely formatted so it can be parsed by a human reader.  This is useful 
	 * if you want to understand what is going on.  On the other hand, if this 
	 * tag is set to NO the size of the Perl module output will be much smaller 
	 * and Perl will parse it just the same.
	 *
	 * @parameter expression="${doxygen.perlmodPretty}" default-value="true"
	 */
	private boolean perlmodPretty;

	/**
	 * The names of the make variables in the generated doxyrules.make file 
	 * are prefixed with the string contained in PERLMOD_MAKEVAR_PREFIX. 
	 * This is useful so different doxyrules.make files included by the same 
	 * Makefile don't overwrite each other's variables.
	 *
	 * @parameter expression="${doxygen.perlmodMakevarPrefix}"
	 */
	private String perlmodMakevarPrefix;

	/**
	 * If the ENABLE_PREPROCESSING tag is set to YES (the default) Doxygen will 
	 * evaluate all C-preprocessor directives found in the sources and include 
	 * files.
	 *
	 * @parameter expression="${doxygen.enablePreprocessing}" default-value="true"
	 */
	private boolean enablePreprocessing;

	/**
	 * If the MACRO_EXPANSION tag is set to YES Doxygen will expand all macro 
	 * names in the source code. If set to NO (the default) only conditional 
	 * compilation will be performed. Macro expansion can be done in a controlled 
	 * way by setting EXPAND_ONLY_PREDEF to YES.
	 *
	 * @parameter expression="${doxygen.macroExpansion}" default-value="false"
	 */
	private boolean macroExpansion;

	/**
	 * If the EXPAND_ONLY_PREDEF and MACRO_EXPANSION tags are both set to YES 
	 * then the macro expansion is limited to the macros specified with the 
	 * PREDEFINED and EXPAND_AS_DEFINED tags.
	 *
	 * @parameter expression="${doxygen.expandOnlyPredef}" default-value="false"
	 */
	private boolean expandOnlyPredef;

	/**
	 * If the SEARCH_INCLUDES tag is set to YES (the default) the includes files 
	 * in the INCLUDE_PATH (see below) will be search if a #include is found.
	 *
	 * @parameter expression="${doxygen.searchIncludes}" default-value="true"
	 */
	private boolean searchIncludes;

	/**
	 * The INCLUDE_PATH tag can be used to specify one or more directories that 
	 * contain include files that are not input files but should be processed by 
	 * the preprocessor.
	 *
	 * @parameter expression="${doxygen.includePath}"
	 */
	private String includePath;

	/**
	 * You can use the INCLUDE_FILE_PATTERNS tag to specify one or more wildcard 
	 * patterns (like *.h and *.hpp) to filter out the header-files in the 
	 * directories. If left blank, the patterns specified with FILE_PATTERNS will 
	 * be used.
	 *
	 * @parameter expression="${doxygen.includeFilePatterns}"
	 */
	private String includeFilePatterns;

	/**
	 * The PREDEFINED tag can be used to specify one or more macro names that 
	 * are defined before the preprocessor is started (similar to the -D option of 
	 * gcc). The argument of the tag is a list of macros of the form: name 
	 * or name=definition (no spaces). If the definition and the = are 
	 * omitted =1 is assumed. To prevent a macro definition from being 
	 * undefined via #undef or recursively expanded use the := operator 
	 * instead of the = operator.
	 *
	 * @parameter expression="${doxygen.predefined}"
	 */
	private String predefined;

	/**
	 * If the MACRO_EXPANSION and EXPAND_ONLY_PREDEF tags are set to YES then 
	 * this tag can be used to specify a list of macro names that should be expanded. 
	 * The macro definition that is found in the sources will be used. 
	 * Use the PREDEFINED tag if you want to use a different macro definition.
	 *
	 * @parameter expression="${doxygen.expandAsDefined}"
	 */
	private String expandAsDefined;

	/**
	 * If the SKIP_FUNCTION_MACROS tag is set to YES (the default) then 
	 * doxygen's preprocessor will remove all function-like macros that are alone 
	 * on a line, have an all uppercase name, and do not end with a semicolon. Such 
	 * function macros are typically used for boiler-plate code, and will confuse 
	 * the parser if not removed.
	 *
	 * @parameter expression="${doxygen.skipFunctionMacros}" default-value="true"
	 */
	private boolean skipFunctionMacros;

	/**
	 * The TAGFILES option can be used to specify one or more tagfiles. 
	 * Optionally an initial location of the external documentation 
	 * can be added for each tagfile. The format of a tag file without 
	 * this location is as follows: 
	 * TAGFILES = file1 file2 ... 
	 * Adding location for the tag files is done as follows: 
	 * TAGFILES = file1=loc1 "file2 = loc2" ... 
	 * where "loc1" and "loc2" can be relative or absolute paths or 
	 * URLs. If a location is present for each tag, the installdox tool 
	 * does not have to be run to correct the links.
	 * Note that each tag file must have a unique name
	 * (where the name does NOT include the path)
	 * If a tag file is not located in the directory in which doxygen 
	 * is run, you must also specify the path to the tagfile here.
	 *
	 * @parameter expression="${doxygen.tagfiles}"
	 */
	private String tagfiles;

	/**
	 * When a file name is specified after GENERATE_TAGFILE, doxygen will create 
	 * a tag file that is based on the input files it reads.
	 *
	 * @parameter expression="${doxygen.generateTagfile}"
	 */
	private String generateTagfile;

	/**
	 * If the ALLEXTERNALS tag is set to YES all external classes will be listed 
	 * in the class index. If set to NO only the inherited external classes 
	 * will be listed.
	 *
	 * @parameter expression="${doxygen.allexternals}" default-value="false"
	 */
	private boolean allexternals;

	/**
	 * If the EXTERNAL_GROUPS tag is set to YES all external groups will be listed 
	 * in the modules index. If set to NO, only the current project's groups will 
	 * be listed.
	 *
	 * @parameter expression="${doxygen.externalGroups}" default-value="true"
	 */
	private boolean externalGroups;

	/**
	 * The PERL_PATH should be the absolute path and name of the perl script 
	 * interpreter (i.e. the result of `which perl').
	 *
	 * @parameter expression="${doxygen.perlPath}" default-value="/usr/bin/perl"
	 */
	private String perlPath;

	/**
	 * If the CLASS_DIAGRAMS tag is set to YES (the default) Doxygen will 
	 * generate a inheritance diagram (in HTML, RTF and LaTeX) for classes with base 
	 * or super classes. Setting the tag to NO turns the diagrams off. Note that 
	 * this option is superseded by the HAVE_DOT option below. This is only a 
	 * fallback. It is recommended to install and use dot, since it yields more 
	 * powerful graphs.
	 *
	 * @parameter expression="${doxygen.classDiagrams}" default-value="true"
	 */
	private boolean classDiagrams;

	/**
	 * You can define message sequence charts within doxygen comments using the \msc 
	 * command. Doxygen will then run the mscgen tool (see 
	 * http://www.mcternan.me.uk/mscgen/) to produce the chart and insert it in the 
	 * documentation. The MSCGEN_PATH tag allows you to specify the directory where 
	 * the mscgen tool resides. If left empty the tool is assumed to be found in the 
	 * default search path.
	 *
	 * @parameter expression="${doxygen.mscgenPath}"
	 */
	private String mscgenPath;

	/**
	 * If set to YES, the inheritance and collaboration graphs will hide 
	 * inheritance and usage relations if the target is undocumented 
	 * or is not a class.
	 *
	 * @parameter expression="${doxygen.hideUndocRelations}" default-value="true"
	 */
	private boolean hideUndocRelations;

	/**
	 * If you set the HAVE_DOT tag to YES then doxygen will assume the dot tool is 
	 * available from the path. This tool is part of Graphviz, a graph visualization 
	 * toolkit from AT&T and Lucent Bell Labs. The other options in this section 
	 * have no effect if this option is set to NO (the default)
	 *
	 * @parameter expression="${doxygen.haveDot}" default-value="true"
	 */
	private boolean haveDot;

	/**
	 * By default doxygen will write a font called FreeSans.ttf to the output 
	 * directory and reference it in all dot files that doxygen generates. This 
	 * font does not include all possible unicode characters however, so when you need 
	 * these (or just want a differently looking font) you can specify the font name 
	 * using DOT_FONTNAME. You need need to make sure dot is able to find the font, 
	 * which can be done by putting it in a standard location or by setting the 
	 * DOTFONTPATH environment variable or by setting DOT_FONTPATH to the directory 
	 * containing the font.
	 *
	 * @parameter expression="${doxygen.dotFontname}" default-value="FreeSans"
	 */
	private String dotFontname;

	/**
	 * The DOT_FONTSIZE tag can be used to set the size of the font of dot graphs. 
	 * The default size is 10pt.
	 *
	 * @parameter expression="${doxygen.dotFontsize}" default-value="10"
	 */
	private Integer dotFontsize;

	/**
	 * By default doxygen will tell dot to use the output directory to look for the 
	 * FreeSans.ttf font (which doxygen will put there itself). If you specify a 
	 * different font using DOT_FONTNAME you can set the path where dot 
	 * can find it using this tag.
	 *
	 * @parameter expression="${doxygen.dotFontpath}"
	 */
	private String dotFontpath;

	/**
	 * If the CLASS_GRAPH and HAVE_DOT tags are set to YES then doxygen 
	 * will generate a graph for each documented class showing the direct and 
	 * indirect inheritance relations. Setting this tag to YES will force the 
	 * the CLASS_DIAGRAMS tag to NO.
	 *
	 * @parameter expression="${doxygen.classGraph}" default-value="true"
	 */
	private boolean classGraph;

	/**
	 * If the COLLABORATION_GRAPH and HAVE_DOT tags are set to YES then doxygen 
	 * will generate a graph for each documented class showing the direct and 
	 * indirect implementation dependencies (inheritance, containment, and 
	 * class references variables) of the class with other documented classes.
	 *
	 * @parameter expression="${doxygen.collaborationGraph}" default-value="true"
	 */
	private boolean collaborationGraph;

	/**
	 * If the GROUP_GRAPHS and HAVE_DOT tags are set to YES then doxygen 
	 * will generate a graph for groups, showing the direct groups dependencies
	 *
	 * @parameter expression="${doxygen.groupGraphs}" default-value="true"
	 */
	private boolean groupGraphs;

	/**
	 * If the UML_LOOK tag is set to YES doxygen will generate inheritance and 
	 * collaboration diagrams in a style similar to the OMG's Unified Modeling 
	 * Language.
	 *
	 * @parameter expression="${doxygen.umlLook}" default-value="false"
	 */
	private boolean umlLook;

	/**
	 * If set to YES, the inheritance and collaboration graphs will show the 
	 * relations between templates and their instances.
	 *
	 * @parameter expression="${doxygen.templateRelations}" default-value="false"
	 */
	private boolean templateRelations;

	/**
	 * If the ENABLE_PREPROCESSING, SEARCH_INCLUDES, INCLUDE_GRAPH, and HAVE_DOT 
	 * tags are set to YES then doxygen will generate a graph for each documented 
	 * file showing the direct and indirect include dependencies of the file with 
	 * other documented files.
	 *
	 * @parameter expression="${doxygen.includeGraph}" default-value="true"
	 */
	private boolean includeGraph;

	/**
	 * If the ENABLE_PREPROCESSING, SEARCH_INCLUDES, INCLUDED_BY_GRAPH, and 
	 * HAVE_DOT tags are set to YES then doxygen will generate a graph for each 
	 * documented header file showing the documented files that directly or 
	 * indirectly include this file.
	 *
	 * @parameter expression="${doxygen.includedByGraph}" default-value="true"
	 */
	private boolean includedByGraph;

	/**
	 * If the CALL_GRAPH and HAVE_DOT options are set to YES then 
	 * doxygen will generate a call dependency graph for every global function 
	 * or class method. Note that enabling this option will significantly increase 
	 * the time of a run. So in most cases it will be better to enable call graphs 
	 * for selected functions only using the \callgraph command.
	 *
	 * @parameter expression="${doxygen.callGraph}" default-value="false"
	 */
	private boolean callGraph;

	/**
	 * If the CALLER_GRAPH and HAVE_DOT tags are set to YES then 
	 * doxygen will generate a caller dependency graph for every global function 
	 * or class method. Note that enabling this option will significantly increase 
	 * the time of a run. So in most cases it will be better to enable caller 
	 * graphs for selected functions only using the \callergraph command.
	 *
	 * @parameter expression="${doxygen.callerGraph}" default-value="false"
	 */
	private boolean callerGraph;

	/**
	 * If the GRAPHICAL_HIERARCHY and HAVE_DOT tags are set to YES then doxygen 
	 * will graphical hierarchy of all classes instead of a textual one.
	 *
	 * @parameter expression="${doxygen.graphicalHierarchy}" default-value="true"
	 */
	private boolean graphicalHierarchy;

	/**
	 * If the DIRECTORY_GRAPH, SHOW_DIRECTORIES and HAVE_DOT tags are set to YES 
	 * then doxygen will show the dependencies a directory has on other directories 
	 * in a graphical way. The dependency relations are determined by the #include
	 * relations between the files in the directories.
	 *
	 * @parameter expression="${doxygen.directoryGraph}" default-value="true"
	 */
	private boolean directoryGraph;

	/**
	 * The DOT_IMAGE_FORMAT tag can be used to set the image format of the images 
	 * generated by dot. Possible values are png, jpg, or gif
	 * If left blank png will be used.
	 *
	 * @parameter expression="${doxygen.dotImageFormat}" default-value="png"
	 */
	private String dotImageFormat;

	/**
	 * The tag DOT_PATH can be used to specify the path where the dot tool can be 
	 * found. If left blank, it is assumed the dot tool can be found in the path.
	 *
	 * @parameter expression="${doxygen.dotPath}" default-value="/usr/bin"
	 */
	private String dotPath;

	/**
	 * The DOTFILE_DIRS tag can be used to specify one or more directories that 
	 * contain dot files that are included in the documentation (see the 
	 * \dotfile command).
	 *
	 * @parameter expression="${doxygen.dotfileDirs}"
	 */
	private String dotfileDirs;

	/**
	 * The DOT_GRAPH_MAX_NODES tag can be used to set the maximum number of 
	 * nodes that will be shown in the graph. If the number of nodes in a graph 
	 * becomes larger than this value, doxygen will truncate the graph, which is 
	 * visualized by representing a node as a red box. Note that doxygen if the 
	 * number of direct children of the root node in a graph is already larger than 
	 * DOT_GRAPH_MAX_NODES then the graph will not be shown at all. Also note 
	 * that the size of a graph can be further restricted by MAX_DOT_GRAPH_DEPTH.
	 *
	 * @parameter expression="${doxygen.dotGraphMaxNodes}" default-value="50"
	 */
	private Integer dotGraphMaxNodes;

	/**
	 * The MAX_DOT_GRAPH_DEPTH tag can be used to set the maximum depth of the 
	 * graphs generated by dot. A depth value of 3 means that only nodes reachable 
	 * from the root by following a path via at most 3 edges will be shown. Nodes 
	 * that lay further from the root node will be omitted. Note that setting this 
	 * option to 1 or 2 may greatly reduce the computation time needed for large 
	 * code bases. Also note that the size of a graph can be further restricted by 
	 * DOT_GRAPH_MAX_NODES. Using a depth of 0 means no depth restriction.
	 *
	 * @parameter expression="${doxygen.maxDotGraphDepth}" default-value="0"
	 */
	private Integer maxDotGraphDepth;

	/**
	 * Set the DOT_TRANSPARENT tag to YES to generate images with a transparent 
	 * background. This is disabled by default, because dot on Windows does not 
	 * seem to support this out of the box. Warning: Depending on the platform used, 
	 * enabling this option may lead to badly anti-aliased labels on the edges of 
	 * a graph (i.e. they become hard to read).
	 *
	 * @parameter expression="${doxygen.dotTransparent}" default-value="true"
	 */
	private boolean dotTransparent;

	/**
	 * Set the DOT_MULTI_TARGETS tag to YES allow dot to generate multiple output 
	 * files in one run (i.e. multiple -o and -T options on the command line). This 
	 * makes dot run faster, but since only newer versions of dot (&gt;1.8.10) 
	 * support this, this feature is disabled by default.
	 *
	 * @parameter expression="${doxygen.dotMultiTargets}" default-value="true"
	 */
	private boolean dotMultiTargets;

	/**
	 * If the GENERATE_LEGEND tag is set to YES (the default) Doxygen will 
	 * generate a legend page explaining the meaning of the various boxes and 
	 * arrows in the dot generated graphs.
	 *
	 * @parameter expression="${doxygen.generateLegend}" default-value="true"
	 */
	private boolean generateLegend;

	/**
	 * If the DOT_CLEANUP tag is set to YES (the default) Doxygen will 
	 * remove the intermediate dot files that are used to generate 
	 * the various graphs.
	 *
	 * @parameter expression="${doxygen.dotCleanup}" default-value="true"
	 */
	private boolean dotCleanup;

	/**
	 * The SEARCHENGINE tag specifies whether or not a search engine should be 
	 * used. If set to NO the values of all tags below this one will be ignored.
	 *
	 * @parameter expression="${doxygen.searchengine}" default-value="false"
	 */
	private boolean searchengine;

	public File getConfigurationFile() {
		return configurationFile;
	}

	public void setConfigurationFile(File configurationFile) {
		this.configurationFile = configurationFile;
	}

	public String getDoxyfileEncoding() {
		return doxyfileEncoding;
	}

	public void setDoxyfileEncoding(String doxyfileEncoding) {
		this.doxyfileEncoding = doxyfileEncoding;
	}

	public String getProjectName() {
		return projectName;
	}

	public void setProjectName(String projectName) {
		this.projectName = projectName;
	}

	public String getProjectNumber() {
		return projectNumber;
	}

	public void setProjectNumber(String projectNumber) {
		this.projectNumber = projectNumber;
	}

//	public String getOutputDirectory() {
//		return outputDirectory;
//	}
//
//	public void setOutputDirectory(String outputDirectory) {
//		this.outputDirectory = outputDirectory;
//	}

	public boolean isCreateSubdirs() {
		return createSubdirs;
	}

	public void setCreateSubdirs(boolean createSubdirs) {
		this.createSubdirs = createSubdirs;
	}

	public String getOutputLanguage() {
		return outputLanguage;
	}

	public void setOutputLanguage(String outputLanguage) {
		this.outputLanguage = outputLanguage;
	}

	public boolean isBriefMemberDesc() {
		return briefMemberDesc;
	}

	public void setBriefMemberDesc(boolean briefMemberDesc) {
		this.briefMemberDesc = briefMemberDesc;
	}

	public boolean isRepeatBrief() {
		return repeatBrief;
	}

	public void setRepeatBrief(boolean repeatBrief) {
		this.repeatBrief = repeatBrief;
	}

	public String getAbbreviateBrief() {
		return abbreviateBrief;
	}

	public void setAbbreviateBrief(String abbreviateBrief) {
		this.abbreviateBrief = abbreviateBrief;
	}

	public boolean isAlwaysDetailedSec() {
		return alwaysDetailedSec;
	}

	public void setAlwaysDetailedSec(boolean alwaysDetailedSec) {
		this.alwaysDetailedSec = alwaysDetailedSec;
	}

	public boolean isInlineInheritedMemb() {
		return inlineInheritedMemb;
	}

	public void setInlineInheritedMemb(boolean inlineInheritedMemb) {
		this.inlineInheritedMemb = inlineInheritedMemb;
	}

	public boolean isFullPathNames() {
		return fullPathNames;
	}

	public void setFullPathNames(boolean fullPathNames) {
		this.fullPathNames = fullPathNames;
	}

	public String getStripFromPath() {
		return stripFromPath;
	}

	public void setStripFromPath(String stripFromPath) {
		this.stripFromPath = stripFromPath;
	}

	public String getStripFromIncPath() {
		return stripFromIncPath;
	}

	public void setStripFromIncPath(String stripFromIncPath) {
		this.stripFromIncPath = stripFromIncPath;
	}

	public boolean isShortNames() {
		return shortNames;
	}

	public void setShortNames(boolean shortNames) {
		this.shortNames = shortNames;
	}

	public boolean isJavadocAutobrief() {
		return javadocAutobrief;
	}

	public void setJavadocAutobrief(boolean javadocAutobrief) {
		this.javadocAutobrief = javadocAutobrief;
	}

	public boolean isQtAutobrief() {
		return qtAutobrief;
	}

	public void setQtAutobrief(boolean qtAutobrief) {
		this.qtAutobrief = qtAutobrief;
	}

	public boolean isMultilineCppIsBrief() {
		return multilineCppIsBrief;
	}

	public void setMultilineCppIsBrief(boolean multilineCppIsBrief) {
		this.multilineCppIsBrief = multilineCppIsBrief;
	}

	public boolean isInheritDocs() {
		return inheritDocs;
	}

	public void setInheritDocs(boolean inheritDocs) {
		this.inheritDocs = inheritDocs;
	}

	public boolean isSeparateMemberPages() {
		return separateMemberPages;
	}

	public void setSeparateMemberPages(boolean separateMemberPages) {
		this.separateMemberPages = separateMemberPages;
	}

	public Integer getTabSize() {
		return tabSize;
	}

	public void setTabSize(Integer tabSize) {
		this.tabSize = tabSize;
	}

	public String getAliases() {
		return aliases;
	}

	public void setAliases(String aliases) {
		this.aliases = aliases;
	}

	public boolean isOptimizeOutputForC() {
		return optimizeOutputForC;
	}

	public void setOptimizeOutputForC(boolean optimizeOutputForC) {
		this.optimizeOutputForC = optimizeOutputForC;
	}

	public boolean isOptimizeOutputJava() {
		return optimizeOutputJava;
	}

	public void setOptimizeOutputJava(boolean optimizeOutputJava) {
		this.optimizeOutputJava = optimizeOutputJava;
	}

	public boolean isOptimizeForFortran() {
		return optimizeForFortran;
	}

	public void setOptimizeForFortran(boolean optimizeForFortran) {
		this.optimizeForFortran = optimizeForFortran;
	}

	public boolean isOptimizeOutputVhdl() {
		return optimizeOutputVhdl;
	}

	public void setOptimizeOutputVhdl(boolean optimizeOutputVhdl) {
		this.optimizeOutputVhdl = optimizeOutputVhdl;
	}

	public boolean isBuiltinStlSupport() {
		return builtinStlSupport;
	}

	public void setBuiltinStlSupport(boolean builtinStlSupport) {
		this.builtinStlSupport = builtinStlSupport;
	}

	public boolean isCppCliSupport() {
		return cppCliSupport;
	}

	public void setCppCliSupport(boolean cppCliSupport) {
		this.cppCliSupport = cppCliSupport;
	}

	public boolean isSipSupport() {
		return sipSupport;
	}

	public void setSipSupport(boolean sipSupport) {
		this.sipSupport = sipSupport;
	}

	public boolean isIdlPropertySupport() {
		return idlPropertySupport;
	}

	public void setIdlPropertySupport(boolean idlPropertySupport) {
		this.idlPropertySupport = idlPropertySupport;
	}

	public boolean isDistributeGroupDoc() {
		return distributeGroupDoc;
	}

	public void setDistributeGroupDoc(boolean distributeGroupDoc) {
		this.distributeGroupDoc = distributeGroupDoc;
	}

	public boolean isSubgrouping() {
		return subgrouping;
	}

	public void setSubgrouping(boolean subgrouping) {
		this.subgrouping = subgrouping;
	}

	public boolean isTypedefHidesStruct() {
		return typedefHidesStruct;
	}

	public void setTypedefHidesStruct(boolean typedefHidesStruct) {
		this.typedefHidesStruct = typedefHidesStruct;
	}

	public Integer getSymbolCacheSize() {
		return symbolCacheSize;
	}

	public void setSymbolCacheSize(Integer symbolCacheSize) {
		this.symbolCacheSize = symbolCacheSize;
	}

	public boolean isExtractAll() {
		return extractAll;
	}

	public void setExtractAll(boolean extractAll) {
		this.extractAll = extractAll;
	}

	public boolean isExtractPrivate() {
		return extractPrivate;
	}

	public void setExtractPrivate(boolean extractPrivate) {
		this.extractPrivate = extractPrivate;
	}

	public boolean isExtractStatic() {
		return extractStatic;
	}

	public void setExtractStatic(boolean extractStatic) {
		this.extractStatic = extractStatic;
	}

	public boolean isExtractLocalClasses() {
		return extractLocalClasses;
	}

	public void setExtractLocalClasses(boolean extractLocalClasses) {
		this.extractLocalClasses = extractLocalClasses;
	}

	public boolean isExtractLocalMethods() {
		return extractLocalMethods;
	}

	public void setExtractLocalMethods(boolean extractLocalMethods) {
		this.extractLocalMethods = extractLocalMethods;
	}

	public boolean isExtractAnonNspaces() {
		return extractAnonNspaces;
	}

	public void setExtractAnonNspaces(boolean extractAnonNspaces) {
		this.extractAnonNspaces = extractAnonNspaces;
	}

	public boolean isHideUndocMembers() {
		return hideUndocMembers;
	}

	public void setHideUndocMembers(boolean hideUndocMembers) {
		this.hideUndocMembers = hideUndocMembers;
	}

	public boolean isHideUndocClasses() {
		return hideUndocClasses;
	}

	public void setHideUndocClasses(boolean hideUndocClasses) {
		this.hideUndocClasses = hideUndocClasses;
	}

	public boolean isHideFriendCompounds() {
		return hideFriendCompounds;
	}

	public void setHideFriendCompounds(boolean hideFriendCompounds) {
		this.hideFriendCompounds = hideFriendCompounds;
	}

	public boolean isHideInBodyDocs() {
		return hideInBodyDocs;
	}

	public void setHideInBodyDocs(boolean hideInBodyDocs) {
		this.hideInBodyDocs = hideInBodyDocs;
	}

	public boolean isInternalDocs() {
		return internalDocs;
	}

	public void setInternalDocs(boolean internalDocs) {
		this.internalDocs = internalDocs;
	}

	public boolean isCaseSenseNames() {
		return caseSenseNames;
	}

	public void setCaseSenseNames(boolean caseSenseNames) {
		this.caseSenseNames = caseSenseNames;
	}

	public boolean isHideScopeNames() {
		return hideScopeNames;
	}

	public void setHideScopeNames(boolean hideScopeNames) {
		this.hideScopeNames = hideScopeNames;
	}

	public boolean isShowIncludeFiles() {
		return showIncludeFiles;
	}

	public void setShowIncludeFiles(boolean showIncludeFiles) {
		this.showIncludeFiles = showIncludeFiles;
	}

	public boolean isInlineInfo() {
		return inlineInfo;
	}

	public void setInlineInfo(boolean inlineInfo) {
		this.inlineInfo = inlineInfo;
	}

	public boolean isSortMemberDocs() {
		return sortMemberDocs;
	}

	public void setSortMemberDocs(boolean sortMemberDocs) {
		this.sortMemberDocs = sortMemberDocs;
	}

	public boolean isSortBriefDocs() {
		return sortBriefDocs;
	}

	public void setSortBriefDocs(boolean sortBriefDocs) {
		this.sortBriefDocs = sortBriefDocs;
	}

	public boolean isSortGroupNames() {
		return sortGroupNames;
	}

	public void setSortGroupNames(boolean sortGroupNames) {
		this.sortGroupNames = sortGroupNames;
	}

	public boolean isSortByScopeName() {
		return sortByScopeName;
	}

	public void setSortByScopeName(boolean sortByScopeName) {
		this.sortByScopeName = sortByScopeName;
	}

	public boolean isGenerateTodolist() {
		return generateTodolist;
	}

	public void setGenerateTodolist(boolean generateTodolist) {
		this.generateTodolist = generateTodolist;
	}

	public boolean isGenerateTestlist() {
		return generateTestlist;
	}

	public void setGenerateTestlist(boolean generateTestlist) {
		this.generateTestlist = generateTestlist;
	}

	public boolean isGenerateBuglist() {
		return generateBuglist;
	}

	public void setGenerateBuglist(boolean generateBuglist) {
		this.generateBuglist = generateBuglist;
	}

	public boolean isGenerateDeprecatedlist() {
		return generateDeprecatedlist;
	}

	public void setGenerateDeprecatedlist(boolean generateDeprecatedlist) {
		this.generateDeprecatedlist = generateDeprecatedlist;
	}

	public String getEnabledSections() {
		return enabledSections;
	}

	public void setEnabledSections(String enabledSections) {
		this.enabledSections = enabledSections;
	}

	public Integer getMaxInitializerLines() {
		return maxInitializerLines;
	}

	public void setMaxInitializerLines(Integer maxInitializerLines) {
		this.maxInitializerLines = maxInitializerLines;
	}

	public boolean isShowUsedFiles() {
		return showUsedFiles;
	}

	public void setShowUsedFiles(boolean showUsedFiles) {
		this.showUsedFiles = showUsedFiles;
	}

	public boolean isShowDirectories() {
		return showDirectories;
	}

	public void setShowDirectories(boolean showDirectories) {
		this.showDirectories = showDirectories;
	}

	public boolean isShowFiles() {
		return showFiles;
	}

	public void setShowFiles(boolean showFiles) {
		this.showFiles = showFiles;
	}

	public boolean isShowNamespaces() {
		return showNamespaces;
	}

	public void setShowNamespaces(boolean showNamespaces) {
		this.showNamespaces = showNamespaces;
	}

	public String getFileVersionFilter() {
		return fileVersionFilter;
	}

	public void setFileVersionFilter(String fileVersionFilter) {
		this.fileVersionFilter = fileVersionFilter;
	}

	public String getLayoutFile() {
		return layoutFile;
	}

	public void setLayoutFile(String layoutFile) {
		this.layoutFile = layoutFile;
	}

	public boolean isQuiet() {
		return quiet;
	}

	public void setQuiet(boolean quiet) {
		this.quiet = quiet;
	}

	public boolean isWarnings() {
		return warnings;
	}

	public void setWarnings(boolean warnings) {
		this.warnings = warnings;
	}

	public boolean isWarnIfUndocumented() {
		return warnIfUndocumented;
	}

	public void setWarnIfUndocumented(boolean warnIfUndocumented) {
		this.warnIfUndocumented = warnIfUndocumented;
	}

	public boolean isWarnIfDocError() {
		return warnIfDocError;
	}

	public void setWarnIfDocError(boolean warnIfDocError) {
		this.warnIfDocError = warnIfDocError;
	}

	public boolean isWarnNoParamdoc() {
		return warnNoParamdoc;
	}

	public void setWarnNoParamdoc(boolean warnNoParamdoc) {
		this.warnNoParamdoc = warnNoParamdoc;
	}

	public String getWarnFormat() {
		return warnFormat;
	}

	public void setWarnFormat(String warnFormat) {
		this.warnFormat = warnFormat;
	}

	public String getWarnLogfile() {
		return warnLogfile;
	}

	public void setWarnLogfile(String warnLogfile) {
		this.warnLogfile = warnLogfile;
	}

	public String getInput() {
		return input;
	}

	public void setInput(String input) {
		this.input = input;
	}

	public String getInputEncoding() {
		return inputEncoding;
	}

	public void setInputEncoding(String inputEncoding) {
		this.inputEncoding = inputEncoding;
	}

	public String getFilePatterns() {
		return filePatterns;
	}

	public void setFilePatterns(String filePatterns) {
		this.filePatterns = filePatterns;
	}

	public boolean isRecursive() {
		return recursive;
	}

	public void setRecursive(boolean recursive) {
		this.recursive = recursive;
	}

	public String getExclude() {
		return exclude;
	}

	public void setExclude(String exclude) {
		this.exclude = exclude;
	}

	public boolean isExcludeSymlinks() {
		return excludeSymlinks;
	}

	public void setExcludeSymlinks(boolean excludeSymlinks) {
		this.excludeSymlinks = excludeSymlinks;
	}

	public String getExcludePatterns() {
		return excludePatterns;
	}

	public void setExcludePatterns(String excludePatterns) {
		this.excludePatterns = excludePatterns;
	}

	public String getExcludeSymbols() {
		return excludeSymbols;
	}

	public void setExcludeSymbols(String excludeSymbols) {
		this.excludeSymbols = excludeSymbols;
	}

	public String getExamplePath() {
		return examplePath;
	}

	public void setExamplePath(String examplePath) {
		this.examplePath = examplePath;
	}

	public String getExamplePatterns() {
		return examplePatterns;
	}

	public void setExamplePatterns(String examplePatterns) {
		this.examplePatterns = examplePatterns;
	}

	public boolean isExampleRecursive() {
		return exampleRecursive;
	}

	public void setExampleRecursive(boolean exampleRecursive) {
		this.exampleRecursive = exampleRecursive;
	}

	public String getImagePath() {
		return imagePath;
	}

	public void setImagePath(String imagePath) {
		this.imagePath = imagePath;
	}

	public String getInputFilter() {
		return inputFilter;
	}

	public void setInputFilter(String inputFilter) {
		this.inputFilter = inputFilter;
	}

	public String getFilterPatterns() {
		return filterPatterns;
	}

	public void setFilterPatterns(String filterPatterns) {
		this.filterPatterns = filterPatterns;
	}

	public boolean isFilterSourceFiles() {
		return filterSourceFiles;
	}

	public void setFilterSourceFiles(boolean filterSourceFiles) {
		this.filterSourceFiles = filterSourceFiles;
	}

	public boolean isSourceBrowser() {
		return sourceBrowser;
	}

	public void setSourceBrowser(boolean sourceBrowser) {
		this.sourceBrowser = sourceBrowser;
	}

	public boolean isInlineSources() {
		return inlineSources;
	}

	public void setInlineSources(boolean inlineSources) {
		this.inlineSources = inlineSources;
	}

	public boolean isStripCodeComments() {
		return stripCodeComments;
	}

	public void setStripCodeComments(boolean stripCodeComments) {
		this.stripCodeComments = stripCodeComments;
	}

	public boolean isReferencedByRelation() {
		return referencedByRelation;
	}

	public void setReferencedByRelation(boolean referencedByRelation) {
		this.referencedByRelation = referencedByRelation;
	}

	public boolean isReferencesRelation() {
		return referencesRelation;
	}

	public void setReferencesRelation(boolean referencesRelation) {
		this.referencesRelation = referencesRelation;
	}

	public boolean isReferencesLinkSource() {
		return referencesLinkSource;
	}

	public void setReferencesLinkSource(boolean referencesLinkSource) {
		this.referencesLinkSource = referencesLinkSource;
	}

	public boolean isUseHtags() {
		return useHtags;
	}

	public void setUseHtags(boolean useHtags) {
		this.useHtags = useHtags;
	}

	public boolean isVerbatimHeaders() {
		return verbatimHeaders;
	}

	public void setVerbatimHeaders(boolean verbatimHeaders) {
		this.verbatimHeaders = verbatimHeaders;
	}

	public boolean isAlphabeticalIndex() {
		return alphabeticalIndex;
	}

	public void setAlphabeticalIndex(boolean alphabeticalIndex) {
		this.alphabeticalIndex = alphabeticalIndex;
	}

	public Integer getColsInAlphaIndex() {
		return colsInAlphaIndex;
	}

	public void setColsInAlphaIndex(Integer colsInAlphaIndex) {
		this.colsInAlphaIndex = colsInAlphaIndex;
	}

	public String getIgnorePrefix() {
		return ignorePrefix;
	}

	public void setIgnorePrefix(String ignorePrefix) {
		this.ignorePrefix = ignorePrefix;
	}

	public boolean isGenerateHtml() {
		return generateHtml;
	}

	public void setGenerateHtml(boolean generateHtml) {
		this.generateHtml = generateHtml;
	}

	public String getHtmlOutput() {
		return htmlOutput;
	}

	public void setHtmlOutput(String htmlOutput) {
		this.htmlOutput = htmlOutput;
	}

	public String getHtmlFileExtension() {
		return htmlFileExtension;
	}

	public void setHtmlFileExtension(String htmlFileExtension) {
		this.htmlFileExtension = htmlFileExtension;
	}

	public String getHtmlHeader() {
		return htmlHeader;
	}

	public void setHtmlHeader(String htmlHeader) {
		this.htmlHeader = htmlHeader;
	}

	public String getHtmlFooter() {
		return htmlFooter;
	}

	public void setHtmlFooter(String htmlFooter) {
		this.htmlFooter = htmlFooter;
	}

	public String getHtmlStylesheet() {
		return htmlStylesheet;
	}

	public void setHtmlStylesheet(String htmlStylesheet) {
		this.htmlStylesheet = htmlStylesheet;
	}

	public boolean isHtmlAlignMembers() {
		return htmlAlignMembers;
	}

	public void setHtmlAlignMembers(boolean htmlAlignMembers) {
		this.htmlAlignMembers = htmlAlignMembers;
	}

	public boolean isHtmlDynamicSections() {
		return htmlDynamicSections;
	}

	public void setHtmlDynamicSections(boolean htmlDynamicSections) {
		this.htmlDynamicSections = htmlDynamicSections;
	}

	public boolean isGenerateDocset() {
		return generateDocset;
	}

	public void setGenerateDocset(boolean generateDocset) {
		this.generateDocset = generateDocset;
	}

	public String getDocsetFeedname() {
		return docsetFeedname;
	}

	public void setDocsetFeedname(String docsetFeedname) {
		this.docsetFeedname = docsetFeedname;
	}

	public String getDocsetBundleId() {
		return docsetBundleId;
	}

	public void setDocsetBundleId(String docsetBundleId) {
		this.docsetBundleId = docsetBundleId;
	}

	public boolean isGenerateHtmlhelp() {
		return generateHtmlhelp;
	}

	public void setGenerateHtmlhelp(boolean generateHtmlhelp) {
		this.generateHtmlhelp = generateHtmlhelp;
	}

	public String getChmFile() {
		return chmFile;
	}

	public void setChmFile(String chmFile) {
		this.chmFile = chmFile;
	}

	public String getHhcLocation() {
		return hhcLocation;
	}

	public void setHhcLocation(String hhcLocation) {
		this.hhcLocation = hhcLocation;
	}

	public boolean isGenerateChi() {
		return generateChi;
	}

	public void setGenerateChi(boolean generateChi) {
		this.generateChi = generateChi;
	}

	public String getChmIndexEncoding() {
		return chmIndexEncoding;
	}

	public void setChmIndexEncoding(String chmIndexEncoding) {
		this.chmIndexEncoding = chmIndexEncoding;
	}

	public boolean isBinaryToc() {
		return binaryToc;
	}

	public void setBinaryToc(boolean binaryToc) {
		this.binaryToc = binaryToc;
	}

	public boolean isTocExpand() {
		return tocExpand;
	}

	public void setTocExpand(boolean tocExpand) {
		this.tocExpand = tocExpand;
	}

	public boolean isGenerateQhp() {
		return generateQhp;
	}

	public void setGenerateQhp(boolean generateQhp) {
		this.generateQhp = generateQhp;
	}

	public String getQchFile() {
		return qchFile;
	}

	public void setQchFile(String qchFile) {
		this.qchFile = qchFile;
	}

	public String getQhpNamespace() {
		return qhpNamespace;
	}

	public void setQhpNamespace(String qhpNamespace) {
		this.qhpNamespace = qhpNamespace;
	}

	public String getQhpVirtualFolder() {
		return qhpVirtualFolder;
	}

	public void setQhpVirtualFolder(String qhpVirtualFolder) {
		this.qhpVirtualFolder = qhpVirtualFolder;
	}

	public String getQhgLocation() {
		return qhgLocation;
	}

	public void setQhgLocation(String qhgLocation) {
		this.qhgLocation = qhgLocation;
	}

	public boolean isDisableIndex() {
		return disableIndex;
	}

	public void setDisableIndex(boolean disableIndex) {
		this.disableIndex = disableIndex;
	}

	public Integer getEnumValuesPerLine() {
		return enumValuesPerLine;
	}

	public void setEnumValuesPerLine(Integer enumValuesPerLine) {
		this.enumValuesPerLine = enumValuesPerLine;
	}

	public String getGenerateTreeview() {
		return generateTreeview;
	}

	public void setGenerateTreeview(String generateTreeview) {
		this.generateTreeview = generateTreeview;
	}

	public Integer getTreeviewWidth() {
		return treeviewWidth;
	}

	public void setTreeviewWidth(Integer treeviewWidth) {
		this.treeviewWidth = treeviewWidth;
	}

	public Integer getFormulaFontsize() {
		return formulaFontsize;
	}

	public void setFormulaFontsize(Integer formulaFontsize) {
		this.formulaFontsize = formulaFontsize;
	}

	public boolean isGenerateLatex() {
		return generateLatex;
	}

	public void setGenerateLatex(boolean generateLatex) {
		this.generateLatex = generateLatex;
	}

	public String getLatexOutput() {
		return latexOutput;
	}

	public void setLatexOutput(String latexOutput) {
		this.latexOutput = latexOutput;
	}

	public String getLatexCmdName() {
		return latexCmdName;
	}

	public void setLatexCmdName(String latexCmdName) {
		this.latexCmdName = latexCmdName;
	}

	public String getMakeindexCmdName() {
		return makeindexCmdName;
	}

	public void setMakeindexCmdName(String makeindexCmdName) {
		this.makeindexCmdName = makeindexCmdName;
	}

	public boolean isCompactLatex() {
		return compactLatex;
	}

	public void setCompactLatex(boolean compactLatex) {
		this.compactLatex = compactLatex;
	}

	public String getPaperType() {
		return paperType;
	}

	public void setPaperType(String paperType) {
		this.paperType = paperType;
	}

	public String getExtraPackages() {
		return extraPackages;
	}

	public void setExtraPackages(String extraPackages) {
		this.extraPackages = extraPackages;
	}

	public String getLatexHeader() {
		return latexHeader;
	}

	public void setLatexHeader(String latexHeader) {
		this.latexHeader = latexHeader;
	}

	public boolean isPdfHyperlinks() {
		return pdfHyperlinks;
	}

	public void setPdfHyperlinks(boolean pdfHyperlinks) {
		this.pdfHyperlinks = pdfHyperlinks;
	}

	public boolean isUsePdflatex() {
		return usePdflatex;
	}

	public void setUsePdflatex(boolean usePdflatex) {
		this.usePdflatex = usePdflatex;
	}

	public boolean isLatexBatchmode() {
		return latexBatchmode;
	}

	public void setLatexBatchmode(boolean latexBatchmode) {
		this.latexBatchmode = latexBatchmode;
	}

	public boolean isLatexHideIndices() {
		return latexHideIndices;
	}

	public void setLatexHideIndices(boolean latexHideIndices) {
		this.latexHideIndices = latexHideIndices;
	}

	public boolean isGenerateRtf() {
		return generateRtf;
	}

	public void setGenerateRtf(boolean generateRtf) {
		this.generateRtf = generateRtf;
	}

	public String getRtfOutput() {
		return rtfOutput;
	}

	public void setRtfOutput(String rtfOutput) {
		this.rtfOutput = rtfOutput;
	}

	public boolean isCompactRtf() {
		return compactRtf;
	}

	public void setCompactRtf(boolean compactRtf) {
		this.compactRtf = compactRtf;
	}

	public boolean isRtfHyperlinks() {
		return rtfHyperlinks;
	}

	public void setRtfHyperlinks(boolean rtfHyperlinks) {
		this.rtfHyperlinks = rtfHyperlinks;
	}

	public String getRtfStylesheetFile() {
		return rtfStylesheetFile;
	}

	public void setRtfStylesheetFile(String rtfStylesheetFile) {
		this.rtfStylesheetFile = rtfStylesheetFile;
	}

	public String getRtfExtensionsFile() {
		return rtfExtensionsFile;
	}

	public void setRtfExtensionsFile(String rtfExtensionsFile) {
		this.rtfExtensionsFile = rtfExtensionsFile;
	}

	public boolean isGenerateMan() {
		return generateMan;
	}

	public void setGenerateMan(boolean generateMan) {
		this.generateMan = generateMan;
	}

	public String getManOutput() {
		return manOutput;
	}

	public void setManOutput(String manOutput) {
		this.manOutput = manOutput;
	}

	public String getManExtension() {
		return manExtension;
	}

	public void setManExtension(String manExtension) {
		this.manExtension = manExtension;
	}

	public boolean isManLinks() {
		return manLinks;
	}

	public void setManLinks(boolean manLinks) {
		this.manLinks = manLinks;
	}

	public boolean isGenerateXml() {
		return generateXml;
	}

	public void setGenerateXml(boolean generateXml) {
		this.generateXml = generateXml;
	}

	public String getXmlOutput() {
		return xmlOutput;
	}

	public void setXmlOutput(String xmlOutput) {
		this.xmlOutput = xmlOutput;
	}

	public String getXmlSchema() {
		return xmlSchema;
	}

	public void setXmlSchema(String xmlSchema) {
		this.xmlSchema = xmlSchema;
	}

	public String getXmlDtd() {
		return xmlDtd;
	}

	public void setXmlDtd(String xmlDtd) {
		this.xmlDtd = xmlDtd;
	}

	public boolean isXmlProgramlisting() {
		return xmlProgramlisting;
	}

	public void setXmlProgramlisting(boolean xmlProgramlisting) {
		this.xmlProgramlisting = xmlProgramlisting;
	}

	public boolean isGenerateAutogenDef() {
		return generateAutogenDef;
	}

	public void setGenerateAutogenDef(boolean generateAutogenDef) {
		this.generateAutogenDef = generateAutogenDef;
	}

	public boolean isGeneratePerlmod() {
		return generatePerlmod;
	}

	public void setGeneratePerlmod(boolean generatePerlmod) {
		this.generatePerlmod = generatePerlmod;
	}

	public boolean isPerlmodLatex() {
		return perlmodLatex;
	}

	public void setPerlmodLatex(boolean perlmodLatex) {
		this.perlmodLatex = perlmodLatex;
	}

	public boolean isPerlmodPretty() {
		return perlmodPretty;
	}

	public void setPerlmodPretty(boolean perlmodPretty) {
		this.perlmodPretty = perlmodPretty;
	}

	public String getPerlmodMakevarPrefix() {
		return perlmodMakevarPrefix;
	}

	public void setPerlmodMakevarPrefix(String perlmodMakevarPrefix) {
		this.perlmodMakevarPrefix = perlmodMakevarPrefix;
	}

	public boolean isEnablePreprocessing() {
		return enablePreprocessing;
	}

	public void setEnablePreprocessing(boolean enablePreprocessing) {
		this.enablePreprocessing = enablePreprocessing;
	}

	public boolean isMacroExpansion() {
		return macroExpansion;
	}

	public void setMacroExpansion(boolean macroExpansion) {
		this.macroExpansion = macroExpansion;
	}

	public boolean isExpandOnlyPredef() {
		return expandOnlyPredef;
	}

	public void setExpandOnlyPredef(boolean expandOnlyPredef) {
		this.expandOnlyPredef = expandOnlyPredef;
	}

	public boolean isSearchIncludes() {
		return searchIncludes;
	}

	public void setSearchIncludes(boolean searchIncludes) {
		this.searchIncludes = searchIncludes;
	}

	public String getIncludePath() {
		return includePath;
	}

	public void setIncludePath(String includePath) {
		this.includePath = includePath;
	}

	public String getIncludeFilePatterns() {
		return includeFilePatterns;
	}

	public void setIncludeFilePatterns(String includeFilePatterns) {
		this.includeFilePatterns = includeFilePatterns;
	}

	public String getPredefined() {
		return predefined;
	}

	public void setPredefined(String predefined) {
		this.predefined = predefined;
	}

	public String getExpandAsDefined() {
		return expandAsDefined;
	}

	public void setExpandAsDefined(String expandAsDefined) {
		this.expandAsDefined = expandAsDefined;
	}

	public boolean isSkipFunctionMacros() {
		return skipFunctionMacros;
	}

	public void setSkipFunctionMacros(boolean skipFunctionMacros) {
		this.skipFunctionMacros = skipFunctionMacros;
	}

	public String getTagfiles() {
		return tagfiles;
	}

	public void setTagfiles(String tagfiles) {
		this.tagfiles = tagfiles;
	}

	public String getGenerateTagfile() {
		return generateTagfile;
	}

	public void setGenerateTagfile(String generateTagfile) {
		this.generateTagfile = generateTagfile;
	}

	public boolean isAllexternals() {
		return allexternals;
	}

	public void setAllexternals(boolean allexternals) {
		this.allexternals = allexternals;
	}

	public boolean isExternalGroups() {
		return externalGroups;
	}

	public void setExternalGroups(boolean externalGroups) {
		this.externalGroups = externalGroups;
	}

	public String getPerlPath() {
		return perlPath;
	}

	public void setPerlPath(String perlPath) {
		this.perlPath = perlPath;
	}

	public boolean isClassDiagrams() {
		return classDiagrams;
	}

	public void setClassDiagrams(boolean classDiagrams) {
		this.classDiagrams = classDiagrams;
	}

	public String getMscgenPath() {
		return mscgenPath;
	}

	public void setMscgenPath(String mscgenPath) {
		this.mscgenPath = mscgenPath;
	}

	public boolean isHideUndocRelations() {
		return hideUndocRelations;
	}

	public void setHideUndocRelations(boolean hideUndocRelations) {
		this.hideUndocRelations = hideUndocRelations;
	}

	public boolean isHaveDot() {
		return haveDot;
	}

	public void setHaveDot(boolean haveDot) {
		this.haveDot = haveDot;
	}

	public String getDotFontname() {
		return dotFontname;
	}

	public void setDotFontname(String dotFontname) {
		this.dotFontname = dotFontname;
	}

	public Integer getDotFontsize() {
		return dotFontsize;
	}

	public void setDotFontsize(Integer dotFontsize) {
		this.dotFontsize = dotFontsize;
	}

	public String getDotFontpath() {
		return dotFontpath;
	}

	public void setDotFontpath(String dotFontpath) {
		this.dotFontpath = dotFontpath;
	}

	public boolean isClassGraph() {
		return classGraph;
	}

	public void setClassGraph(boolean classGraph) {
		this.classGraph = classGraph;
	}

	public boolean isCollaborationGraph() {
		return collaborationGraph;
	}

	public void setCollaborationGraph(boolean collaborationGraph) {
		this.collaborationGraph = collaborationGraph;
	}

	public boolean isGroupGraphs() {
		return groupGraphs;
	}

	public void setGroupGraphs(boolean groupGraphs) {
		this.groupGraphs = groupGraphs;
	}

	public boolean isUmlLook() {
		return umlLook;
	}

	public void setUmlLook(boolean umlLook) {
		this.umlLook = umlLook;
	}

	public boolean isTemplateRelations() {
		return templateRelations;
	}

	public void setTemplateRelations(boolean templateRelations) {
		this.templateRelations = templateRelations;
	}

	public boolean isIncludeGraph() {
		return includeGraph;
	}

	public void setIncludeGraph(boolean includeGraph) {
		this.includeGraph = includeGraph;
	}

	public boolean isIncludedByGraph() {
		return includedByGraph;
	}

	public void setIncludedByGraph(boolean includedByGraph) {
		this.includedByGraph = includedByGraph;
	}

	public boolean isCallGraph() {
		return callGraph;
	}

	public void setCallGraph(boolean callGraph) {
		this.callGraph = callGraph;
	}

	public boolean isCallerGraph() {
		return callerGraph;
	}

	public void setCallerGraph(boolean callerGraph) {
		this.callerGraph = callerGraph;
	}

	public boolean isGraphicalHierarchy() {
		return graphicalHierarchy;
	}

	public void setGraphicalHierarchy(boolean graphicalHierarchy) {
		this.graphicalHierarchy = graphicalHierarchy;
	}

	public boolean isDirectoryGraph() {
		return directoryGraph;
	}

	public void setDirectoryGraph(boolean directoryGraph) {
		this.directoryGraph = directoryGraph;
	}

	public String getDotImageFormat() {
		return dotImageFormat;
	}

	public void setDotImageFormat(String dotImageFormat) {
		this.dotImageFormat = dotImageFormat;
	}

	public String getDotPath() {
		return dotPath;
	}

	public void setDotPath(String dotPath) {
		this.dotPath = dotPath;
	}

	public String getDotfileDirs() {
		return dotfileDirs;
	}

	public void setDotfileDirs(String dotfileDirs) {
		this.dotfileDirs = dotfileDirs;
	}

	public Integer getDotGraphMaxNodes() {
		return dotGraphMaxNodes;
	}

	public void setDotGraphMaxNodes(Integer dotGraphMaxNodes) {
		this.dotGraphMaxNodes = dotGraphMaxNodes;
	}

	public Integer getMaxDotGraphDepth() {
		return maxDotGraphDepth;
	}

	public void setMaxDotGraphDepth(Integer maxDotGraphDepth) {
		this.maxDotGraphDepth = maxDotGraphDepth;
	}

	public boolean isDotTransparent() {
		return dotTransparent;
	}

	public void setDotTransparent(boolean dotTransparent) {
		this.dotTransparent = dotTransparent;
	}

	public boolean isDotMultiTargets() {
		return dotMultiTargets;
	}

	public void setDotMultiTargets(boolean dotMultiTargets) {
		this.dotMultiTargets = dotMultiTargets;
	}

	public boolean isGenerateLegend() {
		return generateLegend;
	}

	public void setGenerateLegend(boolean generateLegend) {
		this.generateLegend = generateLegend;
	}

	public boolean isDotCleanup() {
		return dotCleanup;
	}

	public void setDotCleanup(boolean dotCleanup) {
		this.dotCleanup = dotCleanup;
	}

	public boolean isSearchengine() {
		return searchengine;
	}

	public void setSearchengine(boolean searchengine) {
		this.searchengine = searchengine;
	}

	
}
