/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.license.kafka;

import io.confluent.license.kafka.BrokerDetectionException;
import io.confluent.license.kafka.BrokerDetector;
import io.confluent.license.kafka.BrokerType;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class BrokerDetectorImpl
implements BrokerDetector {
    private static final Logger log = LoggerFactory.getLogger(BrokerDetectorImpl.class);
    private static final String BROKER_INFO_TOPIC = "__internal_confluent_only_broker_info";
    private static final String BROKER_TYPE_KEY = "broker_type";
    private static final Duration POLL_TIMEOUT = Duration.ofMillis(500L);
    private static final int MAX_POLL_RECORDS = 50;
    private static final int MAX_WAIT_MS = 500;
    private static final Map<String, BrokerType> BROKER_TYPE_MAPPING = Map.ofEntries(Map.entry("confluent_cloud", BrokerType.CONFLUENT_CLOUD), Map.entry("confluent_platform", BrokerType.CONFLUENT_PLATFORM));

    BrokerDetectorImpl() {
    }

    @Override
    public BrokerType determineBrokerType(Properties properties) throws BrokerDetectionException {
        return this.doDetectBrokerType(properties);
    }

    @Override
    public BrokerType determineBrokerType(Map<String, Object> properties) throws BrokerDetectionException {
        return this.doDetectBrokerType(this.mapToProperties(properties));
    }

    private BrokerType doDetectBrokerType(Properties properties) throws BrokerDetectionException {
        BrokerType brokerType;
        block9: {
            this.validateInputProperties(properties);
            log.trace("Attempting to detect broker type by reading from topic: {}", (Object)BROKER_INFO_TOPIC);
            Properties consumerProperties = this.createConsumerProperties(properties);
            long startMs = Time.SYSTEM.milliseconds();
            Consumer<String, String> consumer = this.createConsumer(consumerProperties);
            try {
                BrokerType result = this.detectBrokerTypeWithTopicRecords(consumer);
                log.trace("Broker type detection finished in {} ms (result: {})", (Object)(Time.SYSTEM.milliseconds() - startMs), (Object)result);
                brokerType = result;
                if (consumer == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (consumer != null) {
                        try {
                            consumer.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (UnknownTopicOrPartitionException e) {
                    log.debug("Broker info topic {} does not exist, returning UNKNOWN broker type", (Object)BROKER_INFO_TOPIC);
                    return BrokerType.UNKNOWN;
                }
                catch (Exception e) {
                    log.warn("Error while detecting broker type from topic {}: {}", new Object[]{BROKER_INFO_TOPIC, e.getMessage(), e});
                    return BrokerType.UNKNOWN;
                }
            }
            consumer.close();
        }
        return brokerType;
    }

    private Properties mapToProperties(Map<String, Object> originalProps) {
        Properties props = new Properties();
        props.putAll(originalProps);
        return props;
    }

    protected Consumer<String, String> createConsumer(Properties properties) {
        return new KafkaConsumer(properties);
    }

    protected Duration getPollTimeout() {
        return POLL_TIMEOUT;
    }

    private void validateInputProperties(Properties consumerProperties) throws BrokerDetectionException {
        if (consumerProperties == null) {
            throw new BrokerDetectionException("Consumer properties cannot be null");
        }
        Object bootstrapServers = consumerProperties.get("bootstrap.servers");
        if (bootstrapServers == null) {
            throw new BrokerDetectionException("Bootstrap servers configuration is required");
        }
    }

    private Properties createConsumerProperties(Properties originalProps) {
        Properties props = new Properties();
        props.putAll((Map<?, ?>)originalProps);
        props.put("key.deserializer", StringDeserializer.class.getName());
        props.put("value.deserializer", StringDeserializer.class.getName());
        props.put("auto.offset.reset", "earliest");
        props.put("enable.auto.commit", "false");
        props.put("max.poll.records", (Object)50);
        props.put("fetch.max.wait.ms", (Object)500);
        props.put("allow.auto.create.topics", (Object)false);
        return props;
    }

    private BrokerType detectBrokerTypeWithTopicRecords(Consumer<String, String> consumer) {
        TopicPartition topicPartition;
        BrokerType detectedBrokerType = BrokerType.UNKNOWN;
        try {
            List partitions = consumer.partitionsFor(BROKER_INFO_TOPIC);
            if (partitions == null || partitions.isEmpty()) {
                log.debug("Broker info topic {} does not exist in metadata, returning UNKNOWN broker type", (Object)BROKER_INFO_TOPIC);
                return BrokerType.UNKNOWN;
            }
            PartitionInfo p = (PartitionInfo)partitions.get(0);
            topicPartition = new TopicPartition(p.topic(), p.partition());
            long assignStartMs = Time.SYSTEM.milliseconds();
            consumer.assign(Collections.singletonList(topicPartition));
            consumer.seekToBeginning(Collections.singletonList(topicPartition));
            log.trace("Successfully assigned to topic partition: {}, took {} ms )", (Object)topicPartition, (Object)(Time.SYSTEM.milliseconds() - assignStartMs));
        }
        catch (UnknownTopicOrPartitionException e) {
            log.debug("Broker info topic {} does not exist, returning UNKNOWN broker type", (Object)BROKER_INFO_TOPIC);
            return BrokerType.UNKNOWN;
        }
        long endOffset = (Long)consumer.endOffsets(Collections.singletonList(topicPartition)).get(topicPartition);
        while (consumer.position(topicPartition) < endOffset) {
            ConsumerRecords records = consumer.poll(this.getPollTimeout());
            log.trace("Polled {} records from topic {}", (Object)records.count(), (Object)BROKER_INFO_TOPIC);
            BrokerType currentBrokerType = this.processRecords((ConsumerRecords<String, String>)records);
            if (currentBrokerType == null) continue;
            detectedBrokerType = currentBrokerType;
        }
        return detectedBrokerType;
    }

    private BrokerType processRecords(ConsumerRecords<String, String> records) {
        BrokerType detectedBrokerType = null;
        for (ConsumerRecord record : records) {
            log.trace("Processing record: offset={}, key='{}', value='{}'", new Object[]{record.offset(), record.key(), record.value()});
            if (!BROKER_TYPE_KEY.equals(record.key())) continue;
            String brokerTypeValue = (String)record.value();
            detectedBrokerType = BROKER_TYPE_MAPPING.get(brokerTypeValue);
            log.trace("Found matching broker_type key with value: {}, Detected broker type {}", (Object)brokerTypeValue, (Object)detectedBrokerType);
        }
        return detectedBrokerType;
    }
}

