/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.common.security.jetty;

import io.confluent.common.security.auth.CertificatePrincipal;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.function.Function;
import javax.security.auth.login.LoginException;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jetty.security.AbstractLoginService;
import org.eclipse.jetty.security.IdentityService;
import org.eclipse.jetty.security.LoginService;
import org.eclipse.jetty.security.RolePrincipal;
import org.eclipse.jetty.security.UserIdentity;
import org.eclipse.jetty.security.UserPrincipal;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.Session;
import org.eclipse.jetty.util.component.LifeCycle;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompositeLoginService
extends AbstractLoginService {
    private static final Logger log = LoggerFactory.getLogger(CompositeLoginService.class);
    private final LoginService http;
    private final LoginService x509;
    private final JwtConsumer jwtConsumer;
    private boolean isClientAuthRequired = false;
    private boolean isImpersonationTokenValidationEnabled = false;

    public CompositeLoginService(LoginService http, LoginService x509) {
        this.http = http;
        this.x509 = x509;
        this.jwtConsumer = new JwtConsumerBuilder().setSkipSignatureVerification().setDisableRequireSignature().setSkipAllValidators().build();
        super.setIdentityService(null);
    }

    public CompositeLoginService(LoginService http, LoginService x509, boolean isClientAuthRequired, boolean isImpersonationTokenValidationEnabled) {
        this(http, x509);
        this.isClientAuthRequired = isClientAuthRequired;
        this.isImpersonationTokenValidationEnabled = isImpersonationTokenValidationEnabled;
        if (!this.isImpersonationTokenValidationEnabled) {
            log.warn("Skipping impersonation identity validation as impersonation token validation is disabled. Configure 'token.impersonation.validation' to 'true' to enable it.");
        } else if (!this.isClientAuthRequired) {
            log.warn("Skipping impersonation identity validation as client auth is not set. Configure 'ssl.client.authentication' to enhance security.");
        }
    }

    public UserIdentity login(String userName, Object credentials, Request request, Function<Boolean, Session> getOrCreateSession) {
        JwtClaims jwtClaims;
        if (credentials instanceof X509Certificate) {
            log.debug("Credentials provided as X509 certificate, processing to the certificate authorization");
            return this.x509.login(userName, credentials, request, getOrCreateSession);
        }
        log.debug("Processing to the HTTP authorization");
        if (this.isOAuthRequest(request) && (jwtClaims = this.parseJwtToken(credentials)) != null && jwtClaims.hasClaim("cp_proxy")) {
            String proxyCertSN = (String)jwtClaims.getClaimValue("cp_proxy");
            log.debug("Impersonation token detected, trying to validate it for proxy : {}", (Object)proxyCertSN);
            if (this.isImpersonationTokenValidationEnabled && this.isClientAuthRequired) {
                try {
                    this.validateImpersonationIdentity(proxyCertSN, request);
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return this.http.login(userName, credentials, request, getOrCreateSession);
    }

    private boolean isOAuthRequest(Request request) {
        Object isOauth = request.getAttribute("io.confluent.useJWTLoginService");
        return isOauth instanceof Boolean && (Boolean)isOauth != false;
    }

    private JwtClaims parseJwtToken(Object credentials) {
        JwtClaims jwtClaims = null;
        try {
            jwtClaims = this.jwtConsumer.processToClaims((String)credentials);
        }
        catch (Exception e) {
            log.error("Found an invalid JWT token, letting the underlying login service handle it.");
        }
        return jwtClaims;
    }

    public void validateImpersonationIdentity(String proxyCertSN, Request request) throws Exception {
        X509Certificate[] certs = (X509Certificate[])request.getAttribute("jakarta.servlet.request.X509Certificate");
        if (certs == null || certs.length == 0) {
            log.warn("Certificate(s) couldn't be found to validate impersonation token");
        } else {
            Matcher matcher = new Matcher();
            Optional<X509Certificate> matchedCert = Arrays.stream(certs).filter(cert -> matcher.isCertificateValid.test((X509Certificate)cert, proxyCertSN)).findFirst();
            if (matchedCert.isPresent()) {
                log.debug("Impersonation token validation successful. Certificate principal : {}, matches the proxy principal : {}", (Object)matchedCert.get().getSubjectX500Principal().getName(), (Object)proxyCertSN);
            } else {
                throw new LoginException("Impersonation token validation failed. None of the certificate principal matches the proxy principal : " + proxyCertSN);
            }
        }
    }

    public boolean validate(UserIdentity user) {
        if (!user.getSubject().getPrincipals(CertificatePrincipal.class).isEmpty()) {
            log.debug("Claimed identity contains certificate principal, processing to the certificate validation");
            return this.x509.validate(user);
        }
        log.debug("Processing to the HTTP validation");
        return this.http.validate(user);
    }

    public void logout(UserIdentity user) {
        if (!user.getSubject().getPrincipals(CertificatePrincipal.class).isEmpty()) {
            this.x509.logout(user);
        } else {
            this.http.logout(user);
        }
    }

    public void setIdentityService(IdentityService identityService) {
        this.x509.setIdentityService(identityService);
        this.http.setIdentityService(identityService);
        super.setIdentityService(identityService);
    }

    protected UserPrincipal loadUserInfo(String s) {
        throw new UnsupportedOperationException("loadUserInfo");
    }

    protected List<RolePrincipal> loadRoleInfo(UserPrincipal userPrincipal) {
        throw new UnsupportedOperationException("loadRoleInfo");
    }

    protected void doStart() throws Exception {
        if (this.http instanceof LifeCycle) {
            ((LifeCycle)this.http).start();
        }
        if (this.x509 instanceof LifeCycle) {
            ((LifeCycle)this.x509).start();
        }
        super.doStart();
    }

    protected void doStop() throws Exception {
        if (this.x509 instanceof LifeCycle) {
            ((LifeCycle)this.x509).stop();
        }
        if (this.http instanceof LifeCycle) {
            ((LifeCycle)this.http).stop();
        }
        super.doStop();
    }

    private static class Matcher {
        private final BiPredicate<X509Certificate, String> isCertificateValid = (cert, proxyCertSN) -> {
            String certSN = cert.getSubjectX500Principal().getName();
            if (StringUtils.isNotEmpty((CharSequence)certSN) && certSN.equals(proxyCertSN)) {
                log.debug("Impersonation token validated successfully using the certificate");
                return true;
            }
            return false;
        };

        private Matcher() {
        }
    }
}

