package io.split.client;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientRequestFilter;
import javax.ws.rs.core.HttpHeaders;
import java.io.IOException;
import java.net.InetAddress;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Created by adilaijaz on 5/22/15.
 */
public class AddSplitHeadersFilter implements ClientRequestFilter {
    private static final Logger _log = LoggerFactory.getLogger(AddSplitHeadersFilter.class);

    private static final String CLIENT_MACHINE_NAME_HEADER = "SplitSDKMachineName";
    private static final String CLIENT_MACHINE_IP_HEADER = "SplitSDKMachineIP";
    private static final String CLIENT_VERSION = "SplitSDKVersion";

    private final String _apiTokenBearer;
    private final String _hostname;
    private final String _ip;

    public static AddSplitHeadersFilter instance(String apiToken) {
        String hostname = null;
        String ip = null;

        try {
            InetAddress localHost = InetAddress.getLocalHost();
            hostname = localHost.getHostName();
            ip = localHost.getHostAddress();
        } catch (Exception e) {
            _log.error("Could not resolve InetAddress", e);
        }

        return new AddSplitHeadersFilter(apiToken, hostname, ip);
    }

    private AddSplitHeadersFilter(String apiToken, String hostname, String ip) {
        checkNotNull(apiToken);

        _apiTokenBearer = "Bearer " + apiToken;
        _hostname = hostname;
        _ip = ip;
    }

    @Override
    public void filter(ClientRequestContext requestContext) throws IOException {
        requestContext.getHeaders().putSingle(HttpHeaders.AUTHORIZATION, _apiTokenBearer);
        requestContext.getHeaders().putSingle(CLIENT_VERSION, SplitClientConfig.splitSdkVersion);

        if (_hostname != null) {
            requestContext.getHeaders().putSingle(CLIENT_MACHINE_NAME_HEADER, _hostname);
        }

        if (_ip != null) {
            requestContext.getHeaders().putSingle(CLIENT_MACHINE_IP_HEADER, _ip);
        }
    }
}
