package io.split.client;

import io.codigo.dtos.SegmentChange;
import io.split.engine.metrics.Metrics;
import io.split.engine.segments.SegmentChangeFetcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Created by adilaijaz on 5/22/15.
 */
public final class HttpSegmentChangeFetcher implements SegmentChangeFetcher {
    private static final Logger _log = LoggerFactory.getLogger(HttpSegmentChangeFetcher.class);

    private static final String PREFIX = "segmentChangeFetcher";

    private final WebTarget _target;

    private static final String SINCE = "since";

    private final Metrics _metrics;

    public static HttpSegmentChangeFetcher create(WebTarget root) {
        return create(root, new Metrics.NoopMetrics());
    }

    public static HttpSegmentChangeFetcher create(WebTarget root, Metrics metrics) {
        return new HttpSegmentChangeFetcher(root.path("segmentChanges"), metrics);
    }

    private HttpSegmentChangeFetcher(WebTarget target, Metrics metrics) {
        _target = target;
        _metrics = metrics;
        checkNotNull(_target);
    }

    @Override
    public SegmentChange fetch(String segmentName, long since) {
        long start = System.currentTimeMillis();

        try {
            Response response = _target.path(segmentName)
                    .queryParam(SINCE, since)
                    .request(MediaType.APPLICATION_JSON)
                    .acceptEncoding("gzip")
                    .get();

            if (response.getStatus() <  200 || response.getStatus() >= 300) {
                _log.error("Response status was: " + response.getStatus());
                _metrics.count(PREFIX + ".status." + response.getStatus(), 1);
                throw new RuntimeException("Could not retrieve segment changes for " + segmentName + "; http return code " + response.getStatus());
            }

            try {
                return response.readEntity(SegmentChange.class);
            } catch (Throwable t) {
                // log error
                _log.error("Problem reading entity: " + response.getStatus(), t);
                _metrics.count(PREFIX + ".exception", 1);
                throw new RuntimeException(t);
            }

        } finally {
            _metrics.time(PREFIX + ".time", System.currentTimeMillis() - start);
        }

    }
}
