package io.split.client.metrics;

import com.google.common.collect.Lists;
import io.split.api.CounterDTO;
import io.split.api.GaugeDTO;
import io.split.api.LatencyDTO;
import io.split.engine.metrics.Metrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Created by adilaijaz on 9/4/15.
 */
public class HttpMetrics  implements Metrics {
    private static final Logger _log = LoggerFactory.getLogger(HttpMetrics.class);

    private final WebTarget _target;

    public static HttpMetrics create(WebTarget root) {
        return new HttpMetrics(root.path("metrics"));
    }


    public HttpMetrics(WebTarget target) {
        _target = target;
        checkNotNull(_target);
    }


    public void time(LatencyDTO dto) {

        if (dto.latencies().isEmpty()) {
            return;
        }

        post(_target.path("time"), dto);
    }

    public void gauge(GaugeDTO dto) {
        post(_target.path("gauge"), dto);
    }

    public void count(CounterDTO dto) {
        post(_target.path("counter"), dto);
    }

    private void post(WebTarget endpoint, Object dto) {

        try {
            Response response = endpoint
                    .request(MediaType.APPLICATION_JSON)
                    .post(Entity.json(dto));

            if (response.getStatus() != 200) {
                _log.warn("Response status was: " + response.getStatus());
            }

        } catch (Exception e) {
            if (_log.isDebugEnabled()) {
                _log.debug("Exception when posting metric" + dto, e);
            }
        }

    }

    @Override
    public void count(String counter, long delta) {
        try {
            CounterDTO dto = CounterDTO.builder()
                    .name(counter)
                    .delta(delta)
                    .build();
            count(dto);
        } catch (Throwable t) {
            _log.info("Could not count metric " + counter, t);
        }

    }

    @Override
    public void time(String operation, long timeInMs) {
        try {
            LatencyDTO dto = LatencyDTO.builder()
                    .name(operation)
                    .latencies(Lists.newArrayList(timeInMs))
                    .build();
            time(dto);
        } catch (Throwable t) {
            _log.info("Could not time metric " + operation, t);
        }
    }

    @Override
    public void gauge(String gauge, double value) {
        try {
            GaugeDTO dto = GaugeDTO.builder()
                    .name(gauge)
                    .value(value)
                    .build();
            gauge(dto);
        } catch (Throwable t) {
            _log.info("Could not set gauge:" + gauge, t);
        }
    }
}
