/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.header.internals.RecordHeader;
import org.apache.kafka.common.network.TransferableChannel;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.ConvertedRecords;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.LazyDownConversionRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MemoryRecordsBuilder;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.RecordVersion;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.common.record.SimpleRecord;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class FileRecordsTest {
    private byte[][] values = new byte[][]{"abcd".getBytes(), "efgh".getBytes(), "ijkl".getBytes()};
    private FileRecords fileRecords;
    private Time time;

    @BeforeEach
    public void setup() throws IOException {
        this.fileRecords = this.createFileRecords(this.values);
        this.time = new MockTime();
    }

    @AfterEach
    public void cleanup() throws IOException {
        this.fileRecords.close();
    }

    @Test
    public void testAppendProtectsFromOverflow() throws Exception {
        File fileMock = (File)Mockito.mock(File.class);
        FileChannel fileChannelMock = (FileChannel)Mockito.mock(FileChannel.class);
        Mockito.when((Object)fileChannelMock.size()).thenReturn((Object)Integer.MAX_VALUE);
        FileRecords records = new FileRecords(fileMock, fileChannelMock, 0, Integer.MAX_VALUE, false);
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.append(records, this.values));
    }

    @Test
    public void testOpenOversizeFile() throws Exception {
        File fileMock = (File)Mockito.mock(File.class);
        FileChannel fileChannelMock = (FileChannel)Mockito.mock(FileChannel.class);
        Mockito.when((Object)fileChannelMock.size()).thenReturn((Object)0x80000004L);
        Assertions.assertThrows(KafkaException.class, () -> new FileRecords(fileMock, fileChannelMock, 0, Integer.MAX_VALUE, false));
    }

    @Test
    public void testOutOfRangeSlice() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.fileRecords.slice(this.fileRecords.sizeInBytes() + 1, 15).sizeInBytes());
    }

    @Test
    public void testFileSize() throws IOException {
        Assertions.assertEquals((long)this.fileRecords.channel().size(), (long)this.fileRecords.sizeInBytes());
        for (int i = 0; i < 20; ++i) {
            this.fileRecords.append(MemoryRecords.withRecords((CompressionType)CompressionType.NONE, (SimpleRecord[])new SimpleRecord[]{new SimpleRecord("abcd".getBytes())}));
            Assertions.assertEquals((long)this.fileRecords.channel().size(), (long)this.fileRecords.sizeInBytes());
        }
    }

    @Test
    public void testIterationOverPartialAndTruncation() throws IOException {
        this.testPartialWrite(0, this.fileRecords);
        this.testPartialWrite(2, this.fileRecords);
        this.testPartialWrite(4, this.fileRecords);
        this.testPartialWrite(5, this.fileRecords);
        this.testPartialWrite(6, this.fileRecords);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testSliceSizeLimitWithConcurrentWrite() throws Exception {
        FileRecords log = FileRecords.open((File)TestUtils.tempFile());
        ExecutorService executor = Executors.newFixedThreadPool(2);
        int maxSizeInBytes = 16384;
        try {
            Future<Object> readerCompletion = executor.submit(() -> {
                while (log.sizeInBytes() < maxSizeInBytes) {
                    int currentSize = log.sizeInBytes();
                    FileRecords slice = log.slice(0, currentSize);
                    Assertions.assertEquals((int)currentSize, (int)slice.sizeInBytes());
                }
                return null;
            });
            Future<Object> writerCompletion = executor.submit(() -> {
                while (log.sizeInBytes() < maxSizeInBytes) {
                    this.append(log, this.values);
                }
                return null;
            });
            writerCompletion.get();
            readerCompletion.get();
        }
        finally {
            executor.shutdownNow();
        }
    }

    private void testPartialWrite(int size, FileRecords fileRecords) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(size);
        for (int i = 0; i < size; ++i) {
            buffer.put((byte)0);
        }
        buffer.rewind();
        fileRecords.channel().write(buffer);
        Iterator records = fileRecords.records().iterator();
        for (byte[] value : this.values) {
            Assertions.assertTrue((boolean)records.hasNext());
            Assertions.assertEquals((Object)((Record)records.next()).value(), (Object)ByteBuffer.wrap(value));
        }
    }

    @Test
    public void testIterationDoesntChangePosition() throws IOException {
        long position = this.fileRecords.channel().position();
        Iterator records = this.fileRecords.records().iterator();
        for (byte[] value : this.values) {
            Assertions.assertTrue((boolean)records.hasNext());
            Assertions.assertEquals((Object)((Record)records.next()).value(), (Object)ByteBuffer.wrap(value));
        }
        Assertions.assertEquals((long)position, (long)this.fileRecords.channel().position());
    }

    @Test
    public void testRead() throws IOException {
        FileRecords read = this.fileRecords.slice(0, this.fileRecords.sizeInBytes());
        Assertions.assertEquals((int)this.fileRecords.sizeInBytes(), (int)read.sizeInBytes());
        TestUtils.checkEquals(this.fileRecords.batches(), read.batches());
        List<RecordBatch> items = FileRecordsTest.batches((Records)read);
        RecordBatch first = items.get(0);
        read = this.fileRecords.slice(first.sizeInBytes(), this.fileRecords.sizeInBytes() - first.sizeInBytes());
        Assertions.assertEquals((int)(this.fileRecords.sizeInBytes() - first.sizeInBytes()), (int)read.sizeInBytes());
        Assertions.assertEquals(items.subList(1, items.size()), FileRecordsTest.batches((Records)read), (String)"Read starting from the second message");
        read = this.fileRecords.slice(first.sizeInBytes(), this.fileRecords.sizeInBytes());
        Assertions.assertEquals((int)(this.fileRecords.sizeInBytes() - first.sizeInBytes()), (int)read.sizeInBytes());
        Assertions.assertEquals(items.subList(1, items.size()), FileRecordsTest.batches((Records)read), (String)"Read starting from the second message");
        read = this.fileRecords.slice(first.sizeInBytes(), Integer.MAX_VALUE);
        Assertions.assertEquals((int)(this.fileRecords.sizeInBytes() - first.sizeInBytes()), (int)read.sizeInBytes());
        Assertions.assertEquals(items.subList(1, items.size()), FileRecordsTest.batches((Records)read), (String)"Read starting from the second message");
        read = this.fileRecords.slice(1, this.fileRecords.sizeInBytes() - 1).slice(first.sizeInBytes() - 1, this.fileRecords.sizeInBytes());
        Assertions.assertEquals((int)(this.fileRecords.sizeInBytes() - first.sizeInBytes()), (int)read.sizeInBytes());
        Assertions.assertEquals(items.subList(1, items.size()), FileRecordsTest.batches((Records)read), (String)"Read starting from the second message");
        read = this.fileRecords.slice(1, this.fileRecords.sizeInBytes() - 1).slice(first.sizeInBytes() - 1, Integer.MAX_VALUE);
        Assertions.assertEquals((int)(this.fileRecords.sizeInBytes() - first.sizeInBytes()), (int)read.sizeInBytes());
        Assertions.assertEquals(items.subList(1, items.size()), FileRecordsTest.batches((Records)read), (String)"Read starting from the second message");
        RecordBatch second = items.get(1);
        read = this.fileRecords.slice(first.sizeInBytes(), second.sizeInBytes());
        Assertions.assertEquals((int)second.sizeInBytes(), (int)read.sizeInBytes());
        Assertions.assertEquals(Collections.singletonList(second), FileRecordsTest.batches((Records)read), (String)"Read a single message starting from the second message");
    }

    @Test
    public void testSearch() throws IOException {
        SimpleRecord lastMessage = new SimpleRecord("test".getBytes());
        this.fileRecords.append(MemoryRecords.withRecords((long)50L, (CompressionType)CompressionType.NONE, (SimpleRecord[])new SimpleRecord[]{lastMessage}));
        List<RecordBatch> batches = FileRecordsTest.batches((Records)this.fileRecords);
        int position = 0;
        int message1Size = batches.get(0).sizeInBytes();
        Assertions.assertEquals((Object)new FileRecords.LogOffsetPosition(0L, position, message1Size), (Object)this.fileRecords.searchForOffsetWithSize(0L, 0), (String)"Should be able to find the first message by its offset");
        int message2Size = batches.get(1).sizeInBytes();
        Assertions.assertEquals((Object)new FileRecords.LogOffsetPosition(1L, position += message1Size, message2Size), (Object)this.fileRecords.searchForOffsetWithSize(1L, 0), (String)"Should be able to find second message when starting from 0");
        Assertions.assertEquals((Object)new FileRecords.LogOffsetPosition(1L, position, message2Size), (Object)this.fileRecords.searchForOffsetWithSize(1L, position), (String)"Should be able to find second message starting from its offset");
        int message4Size = batches.get(3).sizeInBytes();
        Assertions.assertEquals((Object)new FileRecords.LogOffsetPosition(50L, position += message2Size + batches.get(2).sizeInBytes(), message4Size), (Object)this.fileRecords.searchForOffsetWithSize(3L, position), (String)"Should be able to find fourth message from a non-existent offset");
        Assertions.assertEquals((Object)new FileRecords.LogOffsetPosition(50L, position, message4Size), (Object)this.fileRecords.searchForOffsetWithSize(50L, position), (String)"Should be able to find fourth message by correct offset");
    }

    @Test
    public void testIteratorWithLimits() throws IOException {
        RecordBatch batch = FileRecordsTest.batches((Records)this.fileRecords).get(1);
        int start = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        int size = batch.sizeInBytes();
        FileRecords slice = this.fileRecords.slice(start, size);
        Assertions.assertEquals(Collections.singletonList(batch), FileRecordsTest.batches((Records)slice));
        FileRecords slice2 = this.fileRecords.slice(start, size - 1);
        Assertions.assertEquals(Collections.emptyList(), FileRecordsTest.batches((Records)slice2));
    }

    @Test
    public void testTruncate() throws IOException {
        RecordBatch batch = FileRecordsTest.batches((Records)this.fileRecords).get(0);
        int end = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        this.fileRecords.truncateTo(end);
        Assertions.assertEquals(Collections.singletonList(batch), FileRecordsTest.batches((Records)this.fileRecords));
        Assertions.assertEquals((int)batch.sizeInBytes(), (int)this.fileRecords.sizeInBytes());
    }

    @Test
    public void testTruncateNotCalledIfSizeIsSameAsTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)Mockito.mock(FileChannel.class);
        Mockito.when((Object)channelMock.size()).thenReturn((Object)42L);
        Mockito.when((Object)channelMock.position(42L)).thenReturn(null);
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        fileRecords.truncateTo(42);
        ((FileChannel)Mockito.verify((Object)channelMock, (VerificationMode)Mockito.atLeastOnce())).size();
        ((FileChannel)Mockito.verify((Object)channelMock, (VerificationMode)Mockito.times((int)0))).truncate(ArgumentMatchers.anyLong());
    }

    @Test
    public void testTruncateNotCalledIfSizeIsBiggerThanTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)Mockito.mock(FileChannel.class);
        Mockito.when((Object)channelMock.size()).thenReturn((Object)42L);
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        try {
            fileRecords.truncateTo(43);
            Assertions.fail((String)"Should throw KafkaException");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
        ((FileChannel)Mockito.verify((Object)channelMock, (VerificationMode)Mockito.atLeastOnce())).size();
    }

    @Test
    public void testTruncateIfSizeIsDifferentToTargetSize() throws IOException {
        FileChannel channelMock = (FileChannel)Mockito.mock(FileChannel.class);
        Mockito.when((Object)channelMock.size()).thenReturn((Object)42L);
        Mockito.when((Object)channelMock.truncate(ArgumentMatchers.anyLong())).thenReturn((Object)channelMock);
        FileRecords fileRecords = new FileRecords(TestUtils.tempFile(), channelMock, 0, Integer.MAX_VALUE, false);
        fileRecords.truncateTo(23);
        ((FileChannel)Mockito.verify((Object)channelMock, (VerificationMode)Mockito.atLeastOnce())).size();
        ((FileChannel)Mockito.verify((Object)channelMock)).truncate(23L);
    }

    @Test
    public void testPreallocateTrue() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords fileRecords = FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)true);
        long position = fileRecords.channel().position();
        int size = fileRecords.sizeInBytes();
        Assertions.assertEquals((long)0L, (long)position);
        Assertions.assertEquals((int)0, (int)size);
        Assertions.assertEquals((long)0x100000L, (long)temp.length());
    }

    @Test
    public void testPreallocateFalse() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords set = FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)false);
        long position = set.channel().position();
        int size = set.sizeInBytes();
        Assertions.assertEquals((long)0L, (long)position);
        Assertions.assertEquals((int)0, (int)size);
        Assertions.assertEquals((long)0L, (long)temp.length());
    }

    @Test
    public void testPreallocateClearShutdown() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords fileRecords = FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)true);
        this.append(fileRecords, this.values);
        int oldPosition = (int)fileRecords.channel().position();
        int oldSize = fileRecords.sizeInBytes();
        Assertions.assertEquals((int)this.fileRecords.sizeInBytes(), (int)oldPosition);
        Assertions.assertEquals((int)this.fileRecords.sizeInBytes(), (int)oldSize);
        fileRecords.close();
        File tempReopen = new File(temp.getAbsolutePath());
        FileRecords setReopen = FileRecords.open((File)tempReopen, (boolean)true, (int)0x100000, (boolean)true);
        int position = (int)setReopen.channel().position();
        int size = setReopen.sizeInBytes();
        Assertions.assertEquals((int)oldPosition, (int)position);
        Assertions.assertEquals((int)oldPosition, (int)size);
        Assertions.assertEquals((long)oldPosition, (long)tempReopen.length());
    }

    @Test
    public void testFormatConversionWithPartialMessage() throws IOException {
        RecordBatch batch = FileRecordsTest.batches((Records)this.fileRecords).get(1);
        int start = this.fileRecords.searchForOffsetWithSize((long)1L, (int)0).position;
        int size = batch.sizeInBytes();
        FileRecords slice = this.fileRecords.slice(start, size - 1);
        Records messageV0 = slice.downConvert((byte)0, 0L, this.time).records();
        Assertions.assertTrue((boolean)FileRecordsTest.batches(messageV0).isEmpty(), (String)"No message should be there");
        Assertions.assertEquals((int)(size - 1), (int)messageV0.sizeInBytes(), (String)("There should be " + (size - 1) + " bytes"));
        TopicPartition tp = new TopicPartition("topic-1", 0);
        LazyDownConversionRecords lazyRecords = new LazyDownConversionRecords(tp, (Records)slice, 0, 0L, Time.SYSTEM);
        Iterator it = lazyRecords.iterator(16384L);
        Assertions.assertFalse((boolean)it.hasNext(), (String)"No messages should be returned");
    }

    @Test
    public void testFormatConversionWithNoMessages() throws IOException {
        TopicPartition tp = new TopicPartition("topic-1", 0);
        LazyDownConversionRecords lazyRecords = new LazyDownConversionRecords(tp, (Records)MemoryRecords.EMPTY, 0, 0L, Time.SYSTEM);
        Assertions.assertEquals((int)0, (int)lazyRecords.sizeInBytes());
        Iterator it = lazyRecords.iterator(16384L);
        Assertions.assertFalse((boolean)it.hasNext(), (String)"No messages should be returned");
    }

    @Test
    public void testSearchForTimestamp() throws IOException {
        for (RecordVersion version : RecordVersion.values()) {
            this.testSearchForTimestamp(version);
        }
    }

    private void testSearchForTimestamp(RecordVersion version) throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords fileRecords = FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)true);
        this.appendWithOffsetAndTimestamp(fileRecords, version, 10L, 5L, 0);
        this.appendWithOffsetAndTimestamp(fileRecords, version, 11L, 6L, 1);
        this.assertFoundTimestamp(new FileRecords.FileTimestampAndOffset(10L, 5L, Optional.of(0)), fileRecords.searchForTimestamp(9L, 0, 0L), version);
        this.assertFoundTimestamp(new FileRecords.FileTimestampAndOffset(10L, 5L, Optional.of(0)), fileRecords.searchForTimestamp(10L, 0, 0L), version);
        this.assertFoundTimestamp(new FileRecords.FileTimestampAndOffset(11L, 6L, Optional.of(1)), fileRecords.searchForTimestamp(11L, 0, 0L), version);
        Assertions.assertNull((Object)fileRecords.searchForTimestamp(12L, 0, 0L));
    }

    private void assertFoundTimestamp(FileRecords.FileTimestampAndOffset expected, FileRecords.FileTimestampAndOffset actual, RecordVersion version) {
        if (version == RecordVersion.V0) {
            Assertions.assertNull((Object)actual, (String)"Expected no match for message format v0");
        } else {
            Assertions.assertNotNull((Object)actual, (String)("Expected to find timestamp for message format " + version));
            Assertions.assertEquals((long)expected.timestamp, (long)actual.timestamp, (String)("Expected matching timestamps for message format" + version));
            Assertions.assertEquals((long)expected.offset, (long)actual.offset, (String)("Expected matching offsets for message format " + version));
            Optional expectedLeaderEpoch = version.value >= RecordVersion.V2.value ? expected.leaderEpoch : Optional.empty();
            Assertions.assertEquals((Object)expectedLeaderEpoch, (Object)actual.leaderEpoch, (String)("Non-matching leader epoch for version " + version));
        }
    }

    private void appendWithOffsetAndTimestamp(FileRecords fileRecords, RecordVersion recordVersion, long timestamp, long offset, int leaderEpoch) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(128);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)recordVersion.value, (CompressionType)CompressionType.NONE, (TimestampType)TimestampType.CREATE_TIME, (long)offset, (long)timestamp, (int)leaderEpoch);
        builder.append(new SimpleRecord(timestamp, new byte[0], new byte[0]));
        fileRecords.append(builder.build());
    }

    @Test
    public void testDownconversionAfterMessageFormatDowngrade() throws IOException {
        Random random = new Random();
        byte[] bytes = new byte[3000];
        random.nextBytes(bytes);
        CompressionType compressionType = CompressionType.GZIP;
        List<Long> offsets = Arrays.asList(0L, 1L);
        List<Byte> magic = Arrays.asList((byte)2, (byte)1);
        List<SimpleRecord> records = Arrays.asList(new SimpleRecord(1L, "k1".getBytes(), bytes), new SimpleRecord(2L, "k2".getBytes(), bytes));
        byte toMagic = 1;
        ByteBuffer buffer = ByteBuffer.allocate(8000);
        for (int i = 0; i < records.size(); ++i) {
            MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)magic.get(i), (CompressionType)compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)0L);
            builder.appendWithOffset(offsets.get(i).longValue(), records.get(i));
            builder.close();
        }
        buffer.flip();
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(MemoryRecords.readableRecords((ByteBuffer)buffer));
            fileRecords.flush();
            this.downConvertAndVerifyRecords(records, offsets, fileRecords, compressionType, toMagic, 0L, this.time);
        }
    }

    @Test
    public void testConversion() throws IOException {
        this.doTestConversion(CompressionType.NONE, (byte)0);
        this.doTestConversion(CompressionType.GZIP, (byte)0);
        this.doTestConversion(CompressionType.NONE, (byte)1);
        this.doTestConversion(CompressionType.GZIP, (byte)1);
        this.doTestConversion(CompressionType.NONE, (byte)2);
        this.doTestConversion(CompressionType.GZIP, (byte)2);
    }

    @Test
    public void testNonEmptyFileAlreadyExists() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords records = FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)false);
        records.append(MemoryRecords.withRecords((CompressionType)CompressionType.NONE, (SimpleRecord[])new SimpleRecord[]{new SimpleRecord("abcd".getBytes())}));
        int size = records.sizeInBytes();
        records.close();
        Assertions.assertThrows(IllegalStateException.class, () -> FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)false));
        Assertions.assertThrows(IllegalStateException.class, () -> FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)true));
        records = FileRecords.open((File)temp, (boolean)true, (int)0x100000, (boolean)false);
        Assertions.assertEquals((int)size, (int)records.sizeInBytes());
        records.close();
        records = FileRecords.open((File)temp, (boolean)true, (int)0x100000, (boolean)true);
        Assertions.assertEquals((int)size, (int)records.sizeInBytes());
        records.close();
    }

    @Test
    public void testEmptyFileAlreadyExists() throws IOException {
        File temp = TestUtils.tempFile();
        FileRecords records = FileRecords.open((File)temp, (boolean)true, (int)0x100000, (boolean)false);
        Assertions.assertEquals((int)0, (int)records.sizeInBytes());
        records.close();
        records = FileRecords.open((File)temp, (boolean)true, (int)0x100000, (boolean)true);
        Assertions.assertEquals((int)0, (int)records.sizeInBytes());
        records.close();
        records = FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)false);
        Assertions.assertEquals((int)0, (int)records.sizeInBytes());
        records.close();
        records = FileRecords.open((File)temp, (boolean)false, (int)0x100000, (boolean)true);
        Assertions.assertEquals((int)0, (int)records.sizeInBytes());
        records.close();
    }

    @Test
    public void testBytesLengthOfWriteTo() throws IOException {
        int size = this.fileRecords.sizeInBytes();
        long firstWritten = size / 3;
        TransferableChannel channel = (TransferableChannel)Mockito.mock(TransferableChannel.class);
        this.fileRecords.writeTo(channel, 0L, (int)firstWritten);
        ((TransferableChannel)Mockito.verify((Object)channel)).transferFrom((FileChannel)ArgumentMatchers.any(), ArgumentMatchers.anyLong(), ArgumentMatchers.eq((long)firstWritten));
        int secondWrittenLength = size - (int)firstWritten + 1;
        this.fileRecords.writeTo(channel, firstWritten, secondWrittenLength);
        ((TransferableChannel)Mockito.verify((Object)channel)).transferFrom((FileChannel)ArgumentMatchers.any(), ArgumentMatchers.anyLong(), ArgumentMatchers.eq((long)((long)size - firstWritten)));
    }

    private void doTestConversion(CompressionType compressionType, byte toMagic) throws IOException {
        int i;
        List<Long> offsets = Arrays.asList(0L, 2L, 3L, 9L, 11L, 15L, 16L, 17L, 22L, 24L);
        Header[] headers = new Header[]{new RecordHeader("headerKey1", "headerValue1".getBytes()), new RecordHeader("headerKey2", "headerValue2".getBytes()), new RecordHeader("headerKey3", "headerValue3".getBytes())};
        List<SimpleRecord> records = Arrays.asList(new SimpleRecord(1L, "k1".getBytes(), "hello".getBytes()), new SimpleRecord(2L, "k2".getBytes(), "goodbye".getBytes()), new SimpleRecord(3L, "k3".getBytes(), "hello again".getBytes()), new SimpleRecord(4L, "k4".getBytes(), "goodbye for now".getBytes()), new SimpleRecord(5L, "k5".getBytes(), "hello again".getBytes()), new SimpleRecord(6L, "k6".getBytes(), "I sense indecision".getBytes()), new SimpleRecord(7L, "k7".getBytes(), "what now".getBytes()), new SimpleRecord(8L, "k8".getBytes(), "running out".getBytes(), headers), new SimpleRecord(9L, "k9".getBytes(), "ok, almost done".getBytes()), new SimpleRecord(10L, "k10".getBytes(), "finally".getBytes(), headers));
        Assertions.assertEquals((int)offsets.size(), (int)records.size(), (String)"incorrect test setup");
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)0, (CompressionType)compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)0L);
        for (i = 0; i < 3; ++i) {
            builder.appendWithOffset(offsets.get(i).longValue(), records.get(i));
        }
        builder.close();
        builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)1, (CompressionType)compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)0L);
        for (i = 3; i < 6; ++i) {
            builder.appendWithOffset(offsets.get(i).longValue(), records.get(i));
        }
        builder.close();
        builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (CompressionType)compressionType, (TimestampType)TimestampType.CREATE_TIME, (long)0L);
        for (i = 6; i < 10; ++i) {
            builder.appendWithOffset(offsets.get(i).longValue(), records.get(i));
        }
        builder.close();
        buffer.flip();
        try (FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());){
            fileRecords.append(MemoryRecords.readableRecords((ByteBuffer)buffer));
            fileRecords.flush();
            this.downConvertAndVerifyRecords(records, offsets, fileRecords, compressionType, toMagic, 0L, this.time);
            if (toMagic <= 1 && compressionType == CompressionType.NONE) {
                long firstOffset = toMagic == 0 ? 11L : 17L;
                ArrayList<Long> filteredOffsets = new ArrayList<Long>(offsets);
                ArrayList<SimpleRecord> filteredRecords = new ArrayList<SimpleRecord>(records);
                int index = filteredOffsets.indexOf(firstOffset) - 1;
                filteredRecords.remove(index);
                filteredOffsets.remove(index);
                this.downConvertAndVerifyRecords(filteredRecords, filteredOffsets, fileRecords, compressionType, toMagic, firstOffset, this.time);
            } else {
                this.downConvertAndVerifyRecords(records, offsets, fileRecords, compressionType, toMagic, 10L, this.time);
            }
        }
    }

    private void downConvertAndVerifyRecords(List<SimpleRecord> initialRecords, List<Long> initialOffsets, FileRecords fileRecords, CompressionType compressionType, byte toMagic, long firstOffset, Time time) {
        long minBatchSize = Long.MAX_VALUE;
        long maxBatchSize = Long.MIN_VALUE;
        for (RecordBatch batch : fileRecords.batches()) {
            minBatchSize = Math.min(minBatchSize, (long)batch.sizeInBytes());
            maxBatchSize = Math.max(maxBatchSize, (long)batch.sizeInBytes());
        }
        ArrayList<Records> convertedRecords = new ArrayList<Records>();
        convertedRecords.add(fileRecords.downConvert(toMagic, firstOffset, time).records());
        this.verifyConvertedRecords(initialRecords, initialOffsets, convertedRecords, compressionType, toMagic);
        convertedRecords.clear();
        List<Long> maximumReadSize = Arrays.asList(16384L, fileRecords.sizeInBytes(), (long)fileRecords.sizeInBytes() - 1L, (long)fileRecords.sizeInBytes() / 4L, maxBatchSize + 1L, 1L);
        for (long readSize : maximumReadSize) {
            TopicPartition tp = new TopicPartition("topic-1", 0);
            LazyDownConversionRecords lazyRecords = new LazyDownConversionRecords(tp, (Records)fileRecords, toMagic, firstOffset, Time.SYSTEM);
            Iterator it = lazyRecords.iterator(readSize);
            while (it.hasNext()) {
                convertedRecords.add(((ConvertedRecords)it.next()).records());
            }
            this.verifyConvertedRecords(initialRecords, initialOffsets, convertedRecords, compressionType, toMagic);
            convertedRecords.clear();
        }
    }

    private void verifyConvertedRecords(List<SimpleRecord> initialRecords, List<Long> initialOffsets, List<Records> convertedRecordsList, CompressionType compressionType, byte magicByte) {
        int i = 0;
        for (Records convertedRecords : convertedRecordsList) {
            for (RecordBatch batch : convertedRecords.batches()) {
                Assertions.assertTrue((batch.magic() <= magicByte ? 1 : 0) != 0, (String)("Magic byte should be lower than or equal to " + magicByte));
                if (batch.magic() == 0) {
                    Assertions.assertEquals((Object)TimestampType.NO_TIMESTAMP_TYPE, (Object)batch.timestampType());
                } else {
                    Assertions.assertEquals((Object)TimestampType.CREATE_TIME, (Object)batch.timestampType());
                }
                Assertions.assertEquals((Object)compressionType, (Object)batch.compressionType(), (String)"Compression type should not be affected by conversion");
                for (Record record : batch) {
                    Assertions.assertTrue((boolean)record.hasMagic(batch.magic()), (String)("Inner record should have magic " + magicByte));
                    Assertions.assertEquals((long)initialOffsets.get(i), (long)record.offset(), (String)"Offset should not change");
                    Assertions.assertEquals((Object)Utils.utf8((ByteBuffer)initialRecords.get(i).key()), (Object)Utils.utf8((ByteBuffer)record.key()), (String)"Key should not change");
                    Assertions.assertEquals((Object)Utils.utf8((ByteBuffer)initialRecords.get(i).value()), (Object)Utils.utf8((ByteBuffer)record.value()), (String)"Value should not change");
                    Assertions.assertFalse((boolean)record.hasTimestampType(TimestampType.LOG_APPEND_TIME));
                    if (batch.magic() == 0) {
                        Assertions.assertEquals((long)-1L, (long)record.timestamp());
                        Assertions.assertFalse((boolean)record.hasTimestampType(TimestampType.CREATE_TIME));
                        Assertions.assertTrue((boolean)record.hasTimestampType(TimestampType.NO_TIMESTAMP_TYPE));
                    } else if (batch.magic() == 1) {
                        Assertions.assertEquals((long)initialRecords.get(i).timestamp(), (long)record.timestamp(), (String)"Timestamp should not change");
                        Assertions.assertTrue((boolean)record.hasTimestampType(TimestampType.CREATE_TIME));
                        Assertions.assertFalse((boolean)record.hasTimestampType(TimestampType.NO_TIMESTAMP_TYPE));
                    } else {
                        Assertions.assertEquals((long)initialRecords.get(i).timestamp(), (long)record.timestamp(), (String)"Timestamp should not change");
                        Assertions.assertFalse((boolean)record.hasTimestampType(TimestampType.CREATE_TIME));
                        Assertions.assertFalse((boolean)record.hasTimestampType(TimestampType.NO_TIMESTAMP_TYPE));
                        Assertions.assertArrayEquals((Object[])initialRecords.get(i).headers(), (Object[])record.headers(), (String)"Headers should not change");
                    }
                    ++i;
                }
            }
        }
        Assertions.assertEquals((int)initialOffsets.size(), (int)i);
    }

    private static List<RecordBatch> batches(Records buffer) {
        return TestUtils.toList(buffer.batches());
    }

    private FileRecords createFileRecords(byte[][] values) throws IOException {
        FileRecords fileRecords = FileRecords.open((File)TestUtils.tempFile());
        this.append(fileRecords, values);
        return fileRecords;
    }

    private void append(FileRecords fileRecords, byte[][] values) throws IOException {
        long offset = 0L;
        for (byte[] value : values) {
            ByteBuffer buffer = ByteBuffer.allocate(128);
            MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)2, (CompressionType)CompressionType.NONE, (TimestampType)TimestampType.CREATE_TIME, (long)offset);
            builder.appendWithOffset(offset++, System.currentTimeMillis(), null, value);
            fileRecords.append(builder.build());
        }
        fileRecords.flush();
    }
}

