/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.security.oauthbearer;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.confluent.kafka.security.PrivateKeyUtils;
import io.confluent.kafka.security.oauthbearer.PrivateKeyClientAssertion;
import java.io.File;
import java.security.Key;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.utils.FileWatchService;
import org.apache.kafka.test.TestUtils;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.jwt.consumer.JwtContext;
import org.jose4j.jwx.Headers;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

class PrivateKeyClientAssertionTest {
    private String pemFilepath;
    private PublicKey publicKey;
    private String clientAssertionConfigFilePath;

    PrivateKeyClientAssertionTest() {
    }

    static Stream<Arguments> headerTestCaseProvider() {
        Map<String, Object> presentKeyValueCase1 = PrivateKeyClientAssertionTest.createMap("kid", "test", "x5t", "test", "x5t#S256", "test");
        HashMap absentKeyValueCase1 = new HashMap();
        Map<String, Object> presentKeyValueCase2 = PrivateKeyClientAssertionTest.createMap("kid", "test", "x5t", "test");
        Map<String, Object> absentKeyValueCase2 = PrivateKeyClientAssertionTest.createMap("x5t#S256", "test");
        Map<String, Object> presentKeyValueCase3 = PrivateKeyClientAssertionTest.createMap("kid", "test", "x5t", "test");
        Map<String, Object> absentKeyValueCase3 = PrivateKeyClientAssertionTest.createMap("gibberishHeader", "test");
        return Stream.of(Arguments.of((Object[])new Object[]{presentKeyValueCase1, absentKeyValueCase1}), Arguments.of((Object[])new Object[]{presentKeyValueCase2, absentKeyValueCase2}), Arguments.of((Object[])new Object[]{presentKeyValueCase3, absentKeyValueCase3}));
    }

    static Stream<Arguments> payloadTestCaseProvider() {
        Map<String, Object> presentKeyValueCase1 = PrivateKeyClientAssertionTest.createMap("additonal_aud", "test", "customClaim", 10L, "tenant", "test");
        HashMap absentKeyValueCase1 = new HashMap();
        Map<String, Object> presentKeyValueCase2 = PrivateKeyClientAssertionTest.createMap("additonal_aud", "test", "customClaim", 10L);
        Map<String, Object> absentKeyValueCase2 = PrivateKeyClientAssertionTest.createMap("tenant", "test");
        Map<String, Object> presentKeyValueCase3 = PrivateKeyClientAssertionTest.createMap("kid", "test", "x5t", "test");
        Map<String, Object> absentKeyValueCase3 = PrivateKeyClientAssertionTest.createMap("gibberishHeader", "test");
        return Stream.of(Arguments.of((Object[])new Object[]{presentKeyValueCase1, absentKeyValueCase1}), Arguments.of((Object[])new Object[]{presentKeyValueCase2, absentKeyValueCase2}), Arguments.of((Object[])new Object[]{presentKeyValueCase3, absentKeyValueCase3}));
    }

    private static Map<String, Object> createMap(Object ... keyValues) {
        HashMap<String, Object> map = new HashMap<String, Object>();
        for (int i = 0; i < keyValues.length; i += 2) {
            map.put((String)keyValues[i], keyValues[i + 1]);
        }
        return map;
    }

    private static void validateHeaders(String clientAssertionJwt, Map<String, Object> expectedHeaderMap) throws InvalidJwtException {
        JwtConsumer jwtConsumer = new JwtConsumerBuilder().setSkipSignatureVerification().setExpectedAudience(new String[]{"http://keycloak:8080/realms/cp/protocol/openid-connect/token"}).build();
        JwtContext jwtContext = jwtConsumer.process(clientAssertionJwt);
        JsonWebSignature jws = (JsonWebSignature)jwtContext.getJoseObjects().get(0);
        Headers actualHeaders = jws.getHeaders();
        expectedHeaderMap.forEach((key, value) -> Assertions.assertEquals((Object)value, (Object)actualHeaders.getStringHeaderValue(key)));
    }

    @BeforeEach
    void setUp() throws Exception {
        FileWatchService.useHighSensitivity();
        KeyPair keyPair = PrivateKeyUtils.getRsaKeyPair();
        this.publicKey = keyPair.getPublic();
        String contents = PrivateKeyUtils.getPkcs1PemKey(keyPair);
        this.pemFilepath = PrivateKeyClientAssertionTest.getFilePath(contents);
    }

    @AfterEach
    void tearDown() {
    }

    private static String getFilePath(String fileContent) throws Exception {
        return TestUtils.tempFile(fileContent).getAbsolutePath();
    }

    private void updateFile(String filepath, String contents) throws Exception {
        File file = new File(filepath);
        TestUtils.writeToFile(file, contents);
    }

    @Test
    void testInitialization() {
        PrivateKeyClientAssertion clientAssertion = new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, null);
        Assertions.assertNotNull((Object)clientAssertion);
    }

    @Test
    void testInitializationThrowsFileException() {
        Exception exception = (Exception)Assertions.assertThrows(KafkaException.class, () -> new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), "invalid file path", null, null));
        String expectedMessage = "invalid file path";
        String actualMessage = exception.getMessage();
        Assertions.assertTrue((boolean)actualMessage.contains(expectedMessage));
    }

    @Test
    void testInitializationThrowsJsonParsingException() throws Exception {
        String invalidJsonContent = "{ invalid json content without proper formatting";
        String invalidJsonFilePath = PrivateKeyClientAssertionTest.getFilePath(invalidJsonContent);
        KafkaException exception = (KafkaException)Assertions.assertThrows(KafkaException.class, () -> new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, invalidJsonFilePath));
        String expectedMessage = "Error parsing the json file at path:" + invalidJsonFilePath;
        String actualMessage = exception.getMessage();
        Assertions.assertEquals((Object)expectedMessage, (Object)actualMessage);
    }

    @Test
    void testInitializationThrowsJsonParsingExceptionWithMalformedJson() throws Exception {
        String malformedJsonContent = "not json at all - just plain text";
        String malformedJsonFilePath = PrivateKeyClientAssertionTest.getFilePath(malformedJsonContent);
        KafkaException exception = (KafkaException)Assertions.assertThrows(KafkaException.class, () -> new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, malformedJsonFilePath));
        String expectedMessage = "Error parsing the json file at path:" + malformedJsonFilePath;
        String actualMessage = exception.getMessage();
        Assertions.assertEquals((Object)expectedMessage, (Object)actualMessage);
    }

    @Test
    void testInitializationThrowsJsonParsingExceptionWithEmptyFile() throws Exception {
        String emptyContent = "";
        String emptyFilePath = PrivateKeyClientAssertionTest.getFilePath(emptyContent);
        KafkaException exception = (KafkaException)Assertions.assertThrows(KafkaException.class, () -> new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, emptyFilePath));
        String expectedMessage = "Error parsing the json file at path:" + emptyFilePath;
        String actualMessage = exception.getMessage();
        Assertions.assertEquals((Object)expectedMessage, (Object)actualMessage);
    }

    @Test
    public void testReloadsPrivateKey() throws Exception {
        PrivateKeyClientAssertion clientAssertion = new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, null);
        PrivateKey originalPrivateKey = clientAssertion.getPrivateKey();
        KeyPair keyPair = PrivateKeyUtils.getRsaKeyPair();
        String contents = PrivateKeyUtils.getPkcs1PemKey(keyPair);
        this.updateFile(this.pemFilepath, contents);
        TestUtils.waitForCondition(() -> !originalPrivateKey.equals(clientAssertion.getPrivateKey()), "Private key not reloaded");
    }

    @Test
    public void testReloadsPrivateKeyAndHeaders() throws Exception {
        Map<String, Object> expectedHeaderMap = PrivateKeyClientAssertionTest.createMap("kid", "test", "x5t", "test");
        Map<String, Object> expectedUpdatedHeaderMap = PrivateKeyClientAssertionTest.createMap("kid", "updated-test", "x5t", "updated-test");
        this.createClientAssertionConfigFile(PrivateKeyClientAssertionTest.createMap("headers", expectedHeaderMap));
        PrivateKeyClientAssertion clientAssertion = new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, this.clientAssertionConfigFilePath);
        PrivateKey originalPrivateKey = clientAssertion.getPrivateKey();
        String originalClientAssertionJwt = clientAssertion.getJwt();
        PrivateKeyClientAssertionTest.validateHeaders(originalClientAssertionJwt, expectedHeaderMap);
        KeyPair keyPair = PrivateKeyUtils.getRsaKeyPair();
        String contents = PrivateKeyUtils.getPkcs1PemKey(keyPair);
        this.updateClientAssertionConfigFile(PrivateKeyClientAssertionTest.createMap("headers", expectedUpdatedHeaderMap));
        this.updateFile(this.pemFilepath, contents);
        TestUtils.waitForCondition(() -> !originalPrivateKey.equals(clientAssertion.getPrivateKey()), "Private key not reloaded");
        String updatedClientAssertionJwt = clientAssertion.getJwt();
        PrivateKeyClientAssertionTest.validateHeaders(updatedClientAssertionJwt, expectedUpdatedHeaderMap);
    }

    @Test
    public void testGetJwt() throws Exception {
        PrivateKeyClientAssertion clientAssertion = new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, null);
        String clientAssertionJwt = clientAssertion.getJwt();
        JwtConsumer jwtConsumer = new JwtConsumerBuilder().setVerificationKey((Key)this.publicKey).setExpectedAudience(new String[]{"http://keycloak:8080/realms/cp/protocol/openid-connect/token"}).build();
        JwtClaims jwtClaims = jwtConsumer.processToClaims(clientAssertionJwt);
        Assertions.assertNotNull((Object)jwtClaims);
    }

    @ParameterizedTest
    @MethodSource(value={"headerTestCaseProvider"})
    public void testGetJwtWithAdditionalHeaders(Map<String, String> expectedPresentHeaders, Map<String, String> expectedAbsentHeaders) throws Exception {
        HashMap<String, Object> headerClaimsJson = new HashMap<String, Object>();
        headerClaimsJson.put("headers", expectedPresentHeaders);
        headerClaimsJson.putAll(expectedAbsentHeaders);
        this.createClientAssertionConfigFile(headerClaimsJson);
        PrivateKeyClientAssertion clientAssertion = new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, this.clientAssertionConfigFilePath);
        String clientAssertionJwt = clientAssertion.getJwt();
        JwtConsumer jwtConsumer = new JwtConsumerBuilder().setVerificationKey((Key)this.publicKey).setExpectedAudience(new String[]{"http://keycloak:8080/realms/cp/protocol/openid-connect/token"}).build();
        JwtClaims jwtClaims = jwtConsumer.processToClaims(clientAssertionJwt);
        Assertions.assertNotNull((Object)jwtClaims);
        JwtContext jwtContext = jwtConsumer.process(clientAssertionJwt);
        JsonWebSignature jws = (JsonWebSignature)jwtContext.getJoseObjects().get(0);
        Headers actualHeaders = jws.getHeaders();
        expectedPresentHeaders.forEach((key, value) -> Assertions.assertEquals((Object)value, (Object)actualHeaders.getStringHeaderValue(key)));
        expectedAbsentHeaders.forEach((key, value) -> Assertions.assertNull((Object)actualHeaders.getStringHeaderValue(key)));
    }

    @ParameterizedTest
    @MethodSource(value={"payloadTestCaseProvider"})
    public void testGetJwtWithAdditionalPayloadClaims(Map<String, Object> expectedPresentClaims, Map<String, Object> expectedAbsentClaims) throws Exception {
        HashMap<String, Object> headerClaimsJson = new HashMap<String, Object>();
        headerClaimsJson.put("payload", expectedPresentClaims);
        headerClaimsJson.putAll(expectedAbsentClaims);
        this.createClientAssertionConfigFile(headerClaimsJson);
        PrivateKeyClientAssertion clientAssertion = new PrivateKeyClientAssertion("client_app1", "http://keycloak:8080/realms/cp/protocol/openid-connect/token", "client_app1", Integer.valueOf(10), Boolean.valueOf(true), Boolean.valueOf(true), this.pemFilepath, null, this.clientAssertionConfigFilePath);
        String clientAssertionJwt = clientAssertion.getJwt();
        JwtConsumer jwtConsumer = new JwtConsumerBuilder().setVerificationKey((Key)this.publicKey).setExpectedAudience(new String[]{"http://keycloak:8080/realms/cp/protocol/openid-connect/token"}).build();
        JwtClaims jwtClaims = jwtConsumer.processToClaims(clientAssertionJwt);
        Assertions.assertNotNull((Object)jwtClaims);
        expectedPresentClaims.forEach((key, value) -> Assertions.assertEquals((Object)value, (Object)jwtClaims.getClaimValue(key)));
        expectedAbsentClaims.forEach((key, value) -> Assertions.assertNull((Object)jwtClaims.getClaimValue(key)));
    }

    private void createClientAssertionConfigFile(Map<String, Object> headerClaimsJson) {
        ObjectMapper mapper = new ObjectMapper();
        try {
            String jsonString = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(headerClaimsJson);
            this.clientAssertionConfigFilePath = PrivateKeyClientAssertionTest.getFilePath(jsonString);
        }
        catch (Exception e) {
            throw new RuntimeException("Error writing JSON to file", e);
        }
    }

    private void updateClientAssertionConfigFile(Map<String, Object> headerClaimsJson) {
        ObjectMapper mapper = new ObjectMapper();
        try {
            String jsonString = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(headerClaimsJson);
            this.updateFile(this.clientAssertionConfigFilePath, jsonString);
        }
        catch (Exception e) {
            throw new RuntimeException("Error writing JSON to file", e);
        }
    }
}

