/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.ssl;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.errors.InvalidConfigurationException;
import org.apache.kafka.common.metrics.KafkaMetric;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.security.ssl.DefaultSslEngineFactory;
import org.apache.kafka.common.security.ssl.RenewalFailureSslEngineFactory;
import org.apache.kafka.common.utils.FileWatchService;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class DefaultSslEngineFactoryTest {
    protected static final String CA1 = "-----BEGIN CERTIFICATE-----\nMIIC0zCCAbugAwIBAgIEStdXHTANBgkqhkiG9w0BAQsFADASMRAwDgYDVQQDEwdU\nZXN0Q0ExMCAXDTIwMDkyODA5MDI0MFoYDzIxMjAwOTA0MDkwMjQwWjASMRAwDgYD\nVQQDEwdUZXN0Q0ExMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAo3Gr\nWJAkjnvgcuIfjArDhNdtAlRTt094WMUXhYDibgGtd+CLcWqA+c4PEoK4oybnKZqU\n6MlDfPgesIK2YiNBuSVWMtZ2doageOBnd80Iwbg8DqGtQpUsvw8X5fOmuza+4inv\n/8IpiTizq8YjSMT4nYDmIjyyRCSNY4atjgMnskutJ0v6i69+ZAA520Y6nn2n4RD5\n8Yc+y7yCkbZXnYS5xBOFEExmtc0Xa7S9nM157xqKws9Z+rTKZYLrryaHI9JNcXgG\nkzQEH9fBePASeWfi9AGRvAyS2GMSIBOsihIDIha/mqQcJOGCEqTMtefIj2FaErO2\nbL9yU7OpW53iIC8y0QIDAQABoy8wLTAMBgNVHRMEBTADAQH/MB0GA1UdDgQWBBRf\nsvKcoQ9ZBvjwyUSV2uMFzlkOWDANBgkqhkiG9w0BAQsFAAOCAQEAEE1ZG2MGE248\nglO83ROrHbxmnVWSQHt/JZANR1i362sY1ekL83wlhkriuvGVBlHQYWezIfo/4l9y\nJTHNX3Mrs9eWUkaDXADkHWj3AyLXN3nfeU307x1wA7OvI4YKpwvfb4aYS8RTPz9d\nJtrfR0r8aGTgsXvCe4SgwDBKv7bckctOwD3S7D/b6y3w7X0s7JCU5+8ZjgoYfcLE\ngNqQEaOwdT2LHCvxHmGn/2VGs/yatPQIYYuufe5i8yX7pp4Xbd2eD6LULYkHFs3x\nuJzMRI7BukmIIWuBbAkYI0atxLQIysnVFXdL9pBgvgso2nA3FgP/XeORhkyHVvtL\nREH2YTlftQ==\n-----END CERTIFICATE-----";
    protected static final String CA2 = "-----BEGIN CERTIFICATE-----\nMIIC0zCCAbugAwIBAgIEfk9e9DANBgkqhkiG9w0BAQsFADASMRAwDgYDVQQDEwdU\nZXN0Q0EyMCAXDTIwMDkyODA5MDI0MVoYDzIxMjAwOTA0MDkwMjQxWjASMRAwDgYD\nVQQDEwdUZXN0Q0EyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAvCh0\nUO5op9eHfz7mvZ7IySK7AOCTC56QYFJcU+hD6yk1wKg2qot7naI5ozAc8n7c4pMt\nLjI3D0VtC/oHC29R2HNMSWyHcxIXw8z127XeCLRkCqYWuVAl3nBuWfWVPObjKetH\nTWlQANYWAfk1VbS6wfzgp9cMaK7wQ+VoGEo4x3pjlrdlyg4k4O2yubcpWmJ2TjxS\ngg7TfKGizUVAvF9wUG9Q4AlCg4uuww5RN9w6vnzDKGhWJhkQ6pf/m1xB+WueFOeU\naASGhGqCTqiz3p3M3M4OZzG3KptjQ/yb67x4T5U5RxqoiN4L57E7ZJLREpa6ZZNs\nps/gQ8dR9Uo/PRyAkQIDAQABoy8wLTAMBgNVHRMEBTADAQH/MB0GA1UdDgQWBBRg\nIAOVH5LeE6nZmdScEE3JO/AhvTANBgkqhkiG9w0BAQsFAAOCAQEAHkk1iybwy/Lf\niEQMVRy7XfuC008O7jfCUBMgUvE+oO2RadH5MmsXHG3YerdsDM90dui4JqQNZOUh\nkF8dIWPQHE0xDsR9jiUsemZFpVMN7DcvVZ3eFhbvJA8Q50rxcNGA+tn9xT/xdQ6z\n1eRq9IPoYcRexQ7s9mincM4T4lLm8GGcd7ZPHy8kw0Bp3E/enRHWaF5b8KbXezXD\nI3SEYUyRL2K3px4FImT4X9XQm2EX6EONlu4GRcJpD6RPc0zC7c9dwEnSo+0NnewR\ngjgO34CLzShB/kASLS9VQXcUC6bsggAVK2rWQMmy35SOEUufSuvg8kUFoyuTzfhn\nhL+PVwIu7g==\n-----END CERTIFICATE-----";
    protected static final String CERTCHAIN = "Bag Attributes\n    friendlyName: server\n    localKeyID: 54 69 6D 65 20 31 36 30 31 32 38 33 37 36 35 34 32 33 \nsubject=/CN=TestBroker\nissuer=/CN=TestCA1\n-----BEGIN CERTIFICATE-----\nMIIC/zCCAeegAwIBAgIEatBnEzANBgkqhkiG9w0BAQsFADASMRAwDgYDVQQDEwdU\nZXN0Q0ExMCAXDTIwMDkyODA5MDI0NFoYDzIxMjAwOTA0MDkwMjQ0WjAVMRMwEQYD\nVQQDEwpUZXN0QnJva2VyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA\npkw1AS71ej/iOMvzVgVL1dkQOYzI842NcPmx0yFFsue2umL8WVd3085NgWRb3SS1\n4X676t7zxjPGzYi7jwmA8stCrDt0NAPWd/Ko6ErsCs87CUs4u1Cinf+b3o9NF5u0\nUPYBQLF4Ir8T1jQ+tKiqsChGDt6urRAg1Cro5i7r10jN1uofY2tBs+r8mALhJ17c\nT5LKawXeYwNOQ86c5djClbcP0RrfcPyRyj1/Cp1axo28iO0fXFyO2Zf3a4vtt+Ih\nPW+A2tL+t3JTBd8g7Fl3ozzpcotAi7MDcZaYA9GiTP4DOiKUeDt6yMYQQr3VEqGa\npXp4fKY+t9slqnAmcBZ4kQIDAQABo1gwVjAfBgNVHSMEGDAWgBRfsvKcoQ9ZBvjw\nyUSV2uMFzlkOWDAUBgNVHREEDTALgglsb2NhbGhvc3QwHQYDVR0OBBYEFGWt+27P\nINk/S5X+PRV/jW3WOhtaMA0GCSqGSIb3DQEBCwUAA4IBAQCLHCjFFvqa+0GcG9eq\nv1QWaXDohY5t5CCwD8Z+lT9wcSruTxDPwL7LrR36h++D6xJYfiw4iaRighoA40xP\nW6+0zGK/UtWV4t+ODTDzyAWgls5w+0R5ki6447qGqu5tXlW5DCHkkxWiozMnhNU2\nG3P/Drh7DhmADDBjtVLsu5M1sagF/xwTP/qCLMdChlJNdeqyLnAUa9SYG1eNZS/i\nwrCC8m9RUQb4+OlQuFtr0KhaaCkBXfmhigQAmh44zSyO+oa3qQDEavVFo/Mcui9o\nWBYetcgVbXPNoti+hQEMqmJYBHlLbhxMnkooGn2fa70f453Bdu/Xh6Yphi5NeCHn\n1I+y\n-----END CERTIFICATE-----\nBag Attributes\n    friendlyName: CN=TestCA1\nsubject=/CN=TestCA1\nissuer=/CN=TestCA1\n-----BEGIN CERTIFICATE-----\nMIIC0zCCAbugAwIBAgIEStdXHTANBgkqhkiG9w0BAQsFADASMRAwDgYDVQQDEwdU\nZXN0Q0ExMCAXDTIwMDkyODA5MDI0MFoYDzIxMjAwOTA0MDkwMjQwWjASMRAwDgYD\nVQQDEwdUZXN0Q0ExMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAo3Gr\nWJAkjnvgcuIfjArDhNdtAlRTt094WMUXhYDibgGtd+CLcWqA+c4PEoK4oybnKZqU\n6MlDfPgesIK2YiNBuSVWMtZ2doageOBnd80Iwbg8DqGtQpUsvw8X5fOmuza+4inv\n/8IpiTizq8YjSMT4nYDmIjyyRCSNY4atjgMnskutJ0v6i69+ZAA520Y6nn2n4RD5\n8Yc+y7yCkbZXnYS5xBOFEExmtc0Xa7S9nM157xqKws9Z+rTKZYLrryaHI9JNcXgG\nkzQEH9fBePASeWfi9AGRvAyS2GMSIBOsihIDIha/mqQcJOGCEqTMtefIj2FaErO2\nbL9yU7OpW53iIC8y0QIDAQABoy8wLTAMBgNVHRMEBTADAQH/MB0GA1UdDgQWBBRf\nsvKcoQ9ZBvjwyUSV2uMFzlkOWDANBgkqhkiG9w0BAQsFAAOCAQEAEE1ZG2MGE248\nglO83ROrHbxmnVWSQHt/JZANR1i362sY1ekL83wlhkriuvGVBlHQYWezIfo/4l9y\nJTHNX3Mrs9eWUkaDXADkHWj3AyLXN3nfeU307x1wA7OvI4YKpwvfb4aYS8RTPz9d\nJtrfR0r8aGTgsXvCe4SgwDBKv7bckctOwD3S7D/b6y3w7X0s7JCU5+8ZjgoYfcLE\ngNqQEaOwdT2LHCvxHmGn/2VGs/yatPQIYYuufe5i8yX7pp4Xbd2eD6LULYkHFs3x\nuJzMRI7BukmIIWuBbAkYI0atxLQIysnVFXdL9pBgvgso2nA3FgP/XeORhkyHVvtL\nREH2YTlftQ==\n-----END CERTIFICATE-----";
    protected static final String KEY = "Bag Attributes\n    friendlyName: server\n    localKeyID: 54 69 6D 65 20 31 36 30 31 32 38 33 37 36 35 34 32 33\nKey Attributes: <No Attributes>\n-----BEGIN PRIVATE KEY-----\nMIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKYwggSiAgEAAoIBAQCmTDUBLvV6P+I4\ny/NWBUvV2RA5jMjzjY1w+bHTIUWy57a6YvxZV3fTzk2BZFvdJLXhfrvq3vPGM8bN\niLuPCYDyy0KsO3Q0A9Z38qjoSuwKzzsJSzi7UKKd/5vej00Xm7RQ9gFAsXgivxPW\nND60qKqwKEYO3q6tECDUKujmLuvXSM3W6h9ja0Gz6vyYAuEnXtxPksprBd5jA05D\nzpzl2MKVtw/RGt9w/JHKPX8KnVrGjbyI7R9cXI7Zl/dri+234iE9b4Da0v63clMF\n3yDsWXejPOlyi0CLswNxlpgD0aJM/gM6IpR4O3rIxhBCvdUSoZqlenh8pj632yWq\ncCZwFniRAgMBAAECggEAOfC/XwQvf0KW3VciF0yNGZshbgvBUCp3p284J+ml0Smu\nns4yQiaZl3B/zJ9c6nYJ8OEpNDIuGVac46vKPZIAHZf4SO4GFMFpji078IN6LmH5\nnclZoNn9brNKaYbgQ2N6teKgmRu8Uc7laHKXjnZd0jaWAkRP8/h0l7fDob+jaERj\noJBx4ux2Z62TTCP6W4VY3KZgSL1p6dQswqlukPVytMeI2XEwWnO+w8ED0BxCxM4F\nK//dw7nUMGS9GUNkgyDcH1akYSCDzdBeymQBp2latBotVfGNK1hq9nC1iaxmRkJL\nsYjwVc24n37u+txOovy3daq2ySj9trF7ySAPVYkh4QKBgQDWeN/MR6cy1TLF2j3g\neMMeM32LxXArIPsar+mft+uisKWk5LDpsKpph93sl0JjFi4x0t1mqw23h23I+B2c\nJWiPAHUG3FGvvkPPcfMUvd7pODyE2XaXi+36UZAH7qc94VZGJEb+sPITckSruREE\nQErWZyrbBRgvQXsmVme5B2/kRQKBgQDGf2HQH0KHl54O2r9vrhiQxWIIMSWlizJC\nhjboY6DkIsAMwnXp3wn3Bk4tSgeLk8DEVlmEaE3gvGpiIp0vQnSOlME2TXfEthdM\nuS3+BFXN4Vxxx/qjKL2WfZloyzdaaaF7s+LIwmXgLsFFCUSq+uLtBqfpH2Qv+paX\nXqm7LN3V3QKBgH5ssj/Q3RZx5oQKqf7wMNRUteT2dbB2uI56s9SariQwzPPuevrG\nUS30ETWt1ExkfsaP7kLfAi71fhnBaHLq+j+RnWp15REbrw1RtmC7q/L+W25UYjvj\nGF0+RxDl9V/cvOaL6+2mkIw2B5TSet1uqK7KEdEZp6/zgYyP0oSXhbWhAoGAdnlZ\nHCtMPjnUcPFHCZVTvDTTSihrW9805FfPNe0g/olvLy5xymEBRZtR1d41mq1ZhNY1\nH75RnS1YIbKfNrHnd6J5n7ulHJfCWFy+grp7rCIyVwcRJYkPf17/zXhdVW1uoLLB\nTSoaPDAr0tSxU4vjHa23UoEV/z0F3Nr3W2xwC1ECgYBHKjv6ekLhx7HbP797+Ai+\nwkHvS2L/MqEBxuHzcQ9G6Mj3ANAeyDB8YSC8qGtDQoEyukv2dO73lpodNgbR8P+Q\nPDBb6eyntAo2sSeo0jZkiXvDOfRaGuGVrxjuTfaqcVB33jC6BYfi61/3Sr5oG9Nd\ntDGh1HlOIRm1jD9KQNVZ/Q==\n-----END PRIVATE KEY-----";
    protected static final String ENCRYPTED_KEY = "-----BEGIN ENCRYPTED PRIVATE KEY-----\nMIIE6jAcBgoqhkiG9w0BDAEDMA4ECGyAEWAXlaXzAgIIAASCBMgt7QD1Bbz7MAHI\nNi0eTrwNiuAPluHirLXzsV57d1O9i4EXVp5nzRy6753cjXbGXARbBeaJD+/+jbZp\nCBZTHMG8rTCfbsg5kMqxT6XuuqWlKLKc4gaq+QNgHHleKqnpwZQmOQ+awKWEK/Ow\nZ0KxXqkp+b4/qJK3MqKZDsJtVdyUhO0tLVxd+BHDg9B93oExc87F16h3R0+T4rxE\nTvz2c2upBqva49AbLDxpWXLCJC8CRkxM+KHrPkYjpNx3jCjtwiiXfzJCWjuCkVrL\n2F4bqvpYPIseoPtMvWaplNtoPwhpzBB/hoJ+R+URr4XHX3Y+bz6k6iQnhoCOIviy\noEEUvWtKnaEEKSauR+Wyj3MoeB64g9NWMEHv7+SQeA4WqlgV2s4txwRxFGKyKLPq\ncaMSpfxvYujtSh0DOv9GI3cVHPM8WsebCz9cNrbKSR8/8JufcoonTitwF/4vm1Et\nAdmCuH9JIYVvmFKFVxY9SvRAvo43OQaPmJQHMUa4yDfMtpTSgmB/7HFgxtksYs++\nGbrq6F/hon+0bLx+bMz2FK635UU+iVno+qaScKWN3BFqDl+KnZprBhLSXTT3aHmp\nfisQit/HWp71a0Vzq85WwI4ucMKNc8LemlwNBxWLLiJDp7sNPLb5dIl8yIwSEIgd\nvC5px9KWEdt3GxTUEqtIeBmagbBhahcv+c9Dq924DLI+Slv6TJKZpIcMqUECgzvi\nhb8gegyEscBEcDSzl0ojlFVz4Va5eZS/linTjNJhnkx8BKLn/QFco7FpEE6uOmQ3\n0kF64M2Rv67cJbYVrhD46TgIzH3Y/FOMSi1zFHQ14nVXWMu0yAlBX+QGk7Xl+/aF\nBIq+i9WcBqbttR3CwyeTnIFXkdC66iTZYhDl9HT6yMcazql2Or2TjIIWr6tfNWH/\n5dWSEHYM5m8F2/wF0ANWJyR1oPr4ckcUsfl5TfOWVj5wz4QVF6EGV7FxEnQHrdx0\n6rXThRKFjqxUubsNt1yUEwdlTNz2UFhobGF9MmFeB97BZ6T4v8G825de/Caq9FzO\nyMFFCRcGC7gIzMXRPEjHIvBdTThm9rbNzKPXHqw0LHG478yIqzxvraCYTRw/4eWN\nQ+hyOL/5T5QNXHpR8Udp/7sptw7HfRnecQ/Vz9hOKShQq3h4Sz6eQMQm7P9qGo/N\nbltEAIECRVcNYLN8LuEORfeecNcV3BX+4BBniFtdD2bIRsWC0ZUsGf14Yhr4P1OA\nPtMJzy99mrcq3h+o+hEW6bhIj1gA88JSMJ4iRuwTLRKE81w7EyziScDsotYKvDPu\nw4+PFbQO3fr/Zga3LgYis8/DMqZoWjVCjAeVoypuOZreieZYC/BgBS8qSUAmDPKq\njK+T5pwMMchfXbkV80LTu1kqLfKWdE0AmZfGy8COE/NNZ/FeiWZPdwu2Ix6u/RoY\nLTjNy4YLIBdVELFXaFJF2GfzLpnwrW5tyNPVVrGmUoiyOzgx8gMyCLGavGtduyoY\ntBiUTmd05Ugscn4Rz9X30S4NbnjL/h+bWl1m6/M+9FHEe85FPxmt/GRmJPbFPMR5\nq5EgQGkt4ifiaP6qvyFulwvVwx+m0bf1q6Vb/k3clIyLMcVZWFE1TqNH2Ife46AE\n2I39ZnGTt0mbWskpHBA=\n-----END ENCRYPTED PRIVATE KEY-----";
    protected static final Password KEY_PASSWORD = new Password("key-password");
    protected DefaultSslEngineFactory factory;
    protected Map<String, Object> configs = new HashMap<String, Object>();
    protected Metrics metrics = new Metrics();

    @BeforeEach
    public void setUp() throws IOException {
        FileWatchService.useHighSensitivity();
        this.factory = this.sslEngineFactory();
        this.factory.setMetrics(this.metrics);
        this.configs.clear();
        this.configs.put("ssl.protocol", "TLSv1.2");
    }

    @AfterEach
    public void teardown() throws IOException {
        this.factory.close();
    }

    protected String getEncryptedKey() {
        return ENCRYPTED_KEY;
    }

    protected Password getKeyPassword() {
        return KEY_PASSWORD;
    }

    protected String getKeyStoreType() {
        return "PKCS12";
    }

    protected DefaultSslEngineFactory sslEngineFactory() {
        return new DefaultSslEngineFactory();
    }

    protected Class<? extends Throwable> getKeyException() {
        return null;
    }

    private void verifyGetKeyWithNoKey(KeyStore store, String alias) throws Exception {
        if (this.getKeyException() == null) {
            Assertions.assertNull((Object)store.getKey(alias, null), (String)"Unexpected private key");
        } else {
            Assertions.assertThrows(this.getKeyException(), () -> store.getKey(alias, null), (String)"Unexpected private key");
        }
    }

    @Test
    public void testPemTrustStoreConfigWithOneCert() throws Exception {
        this.configs.put("ssl.truststore.certificates", DefaultSslEngineFactoryTest.pemAsConfigValue(CA1));
        this.configs.put("ssl.truststore.type", "PEM");
        this.factory.configure(this.configs);
        KeyStore trustStore = this.factory.truststore();
        ArrayList<String> aliases = Collections.list(trustStore.aliases());
        Assertions.assertEquals(Collections.singletonList("kafka0"), aliases);
        Assertions.assertNotNull((Object)trustStore.getCertificate("kafka0"), (String)"Certificate not loaded");
        this.verifyGetKeyWithNoKey(trustStore, "kafka0");
    }

    @Test
    public void testPemTrustStoreConfigWithMultipleCerts() throws Exception {
        this.configs.put("ssl.truststore.certificates", DefaultSslEngineFactoryTest.pemAsConfigValue(CA1, CA2));
        this.configs.put("ssl.truststore.type", "PEM");
        this.factory.configure(this.configs);
        KeyStore trustStore = this.factory.truststore();
        ArrayList<String> aliases = Collections.list(trustStore.aliases());
        Collections.sort(aliases);
        Assertions.assertEquals(Arrays.asList("kafka0", "kafka1"), aliases);
        Assertions.assertNotNull((Object)trustStore.getCertificate("kafka0"), (String)"Certificate not loaded");
        this.verifyGetKeyWithNoKey(trustStore, "kafka0");
        Assertions.assertNotNull((Object)trustStore.getCertificate("kafka1"), (String)"Certificate not loaded");
        this.verifyGetKeyWithNoKey(trustStore, "kafka1");
        Assertions.assertEquals((Object)this.getKeyStoreType(), (Object)trustStore.getType());
    }

    @Test
    public void testPemKeyStoreConfigNoPassword() throws Exception {
        this.verifyPemKeyStoreConfig(KEY, null);
    }

    @Test
    public void testPemKeyStoreConfigWithKeyPassword() throws Exception {
        this.verifyPemKeyStoreConfig(this.getEncryptedKey(), this.getKeyPassword());
    }

    @Test
    public void testTrailingNewLines() throws Exception {
        this.verifyPemKeyStoreConfig(this.getEncryptedKey() + "\n\n", this.getKeyPassword());
    }

    @Test
    public void testLeadingNewLines() throws Exception {
        this.verifyPemKeyStoreConfig("\n\n" + this.getEncryptedKey(), this.getKeyPassword());
    }

    @Test
    public void testCarriageReturnLineFeed() throws Exception {
        this.verifyPemKeyStoreConfig(this.getEncryptedKey().replaceAll("\n", "\r\n"), this.getKeyPassword());
    }

    private void verifyPemKeyStoreConfig(String keyFileName, Password keyPassword) throws Exception {
        this.configs.put("ssl.keystore.key", DefaultSslEngineFactoryTest.pemAsConfigValue(keyFileName));
        this.configs.put("ssl.keystore.certificate.chain", DefaultSslEngineFactoryTest.pemAsConfigValue(CERTCHAIN));
        this.configs.put("ssl.key.password", keyPassword);
        this.configs.put("ssl.keystore.type", "PEM");
        this.factory.configure(this.configs);
        KeyStore keyStore = this.factory.keystore();
        ArrayList<String> aliases = Collections.list(keyStore.aliases());
        Assertions.assertEquals(Collections.singletonList("kafka"), aliases);
        Assertions.assertNotNull((Object)keyStore.getCertificate("kafka"), (String)"Certificate not loaded");
        Assertions.assertNotNull((Object)keyStore.getKey("kafka", keyPassword == null ? null : keyPassword.value().toCharArray()), (String)"Private key not loaded");
        Assertions.assertEquals((Object)this.getKeyStoreType(), (Object)keyStore.getType());
    }

    @Test
    public void testPemTrustStoreFile() throws Exception {
        this.configs.put("ssl.truststore.location", DefaultSslEngineFactoryTest.pemFilePath(CA1));
        this.configs.put("ssl.truststore.type", "PEM");
        this.factory.configure(this.configs);
        KeyStore trustStore = this.factory.truststore();
        ArrayList<String> aliases = Collections.list(trustStore.aliases());
        Assertions.assertEquals(Collections.singletonList("kafka0"), aliases);
        Assertions.assertNotNull((Object)trustStore.getCertificate("kafka0"), (String)"Certificate not found");
        this.verifyGetKeyWithNoKey(trustStore, "kafka0");
    }

    @Test
    public void testPemKeyStoreFileNoKeyPassword() throws Exception {
        this.configs.put("ssl.keystore.location", DefaultSslEngineFactoryTest.pemFilePath(DefaultSslEngineFactoryTest.pemAsConfigValue(KEY, CERTCHAIN).value()));
        this.configs.put("ssl.keystore.type", "PEM");
        this.configs.put("ssl.key.password", null);
        this.factory.configure(this.configs);
        KeyStore keyStore = this.factory.keystore();
        ArrayList<String> aliases = Collections.list(keyStore.aliases());
        Assertions.assertEquals(Collections.singletonList("kafka"), aliases);
        Assertions.assertNotNull((Object)keyStore.getCertificate("kafka"), (String)"Certificate not loaded");
        Assertions.assertNotNull((Object)keyStore.getKey("kafka", null), (String)"Private key not loaded");
    }

    @Test
    public void testPemKeyStoreFileWithKeyPassword() throws Exception {
        this.configs.put("ssl.keystore.location", DefaultSslEngineFactoryTest.pemFilePath(DefaultSslEngineFactoryTest.pemAsConfigValue(this.getEncryptedKey(), CERTCHAIN).value()));
        this.configs.put("ssl.key.password", this.getKeyPassword());
        this.configs.put("ssl.keystore.type", "PEM");
        this.factory.configure(this.configs);
        KeyStore keyStore = this.factory.keystore();
        ArrayList<String> aliases = Collections.list(keyStore.aliases());
        Assertions.assertEquals(Collections.singletonList("kafka"), aliases);
        Assertions.assertNotNull((Object)keyStore.getCertificate("kafka"), (String)"Certificate not found");
        if (this.getKeyPassword() != null) {
            Assertions.assertNotNull((Object)keyStore.getKey("kafka", this.getKeyPassword().value().toCharArray()), (String)"Private key not found");
        }
    }

    public static String pemFilePath(String pem) throws Exception {
        return TestUtils.tempFile(pem).getAbsolutePath();
    }

    public static Password pemAsConfigValue(String ... pemValues) {
        StringBuilder builder = new StringBuilder();
        for (String pem : pemValues) {
            builder.append(pem);
            builder.append("\n");
        }
        return new Password(builder.toString().trim());
    }

    @Test
    public void testReloadingOnClosed() throws Exception {
        this.configs.put("ssl.protocol", "TLSv1.2");
        this.configs.put("confluent.ssl.enable.dynamic.store.update", "true");
        String filePath = DefaultSslEngineFactoryTest.pemFilePath(DefaultSslEngineFactoryTest.pemAsConfigValue(this.getEncryptedKey(), CERTCHAIN).value());
        this.configs.put("ssl.keystore.location", filePath);
        this.configs.put("ssl.key.password", this.getKeyPassword());
        this.configs.put("ssl.keystore.type", "PEM");
        DefaultSslEngineFactory factory = this.sslEngineFactory();
        factory.configure(this.configs);
        TestUtils.waitForCondition(() -> factory.keyStoreFileWatchListener().initCompleted(), "FileWatchListener init is not completed");
        Assertions.assertNotNull((Object)this.getKeyStoreFromFileWatchListener(factory));
        factory.close();
        factory.createAndSetSSLContext();
        Assertions.assertTrue((boolean)DefaultSslEngineFactory.isEmptyWatch());
    }

    @Test
    public void testKeyStoreFileTriggerReload() throws Exception {
        this.configs.put("ssl.protocol", "TLSv1.2");
        this.configs.put("confluent.ssl.enable.dynamic.store.update", "true");
        String filePath = DefaultSslEngineFactoryTest.pemFilePath(DefaultSslEngineFactoryTest.pemAsConfigValue(this.getEncryptedKey(), CERTCHAIN).value());
        this.configs.put("ssl.keystore.location", filePath);
        this.configs.put("ssl.key.password", this.getKeyPassword());
        this.configs.put("ssl.keystore.type", "PEM");
        this.factory.configure(this.configs);
        TestUtils.waitForCondition(() -> this.factory.keyStoreFileWatchListener().initCompleted(), "FileWatchListener init is not completed");
        Assertions.assertNotNull((Object)this.getKeyStoreFromFileWatchListener(this.factory));
        Thread.sleep(FileWatchService.MIN_WATCH_INTERVAL.toMillis());
        KeyStore oldStore = this.getKeyStoreFromFileWatchListener(this.factory);
        try (FileWriter writer = new FileWriter(filePath);){
            writer.write(DefaultSslEngineFactoryTest.pemAsConfigValue(KEY, CERTCHAIN).value());
            writer.flush();
        }
        TestUtils.waitForCondition(() -> this.factory.keyStoreFileWatchListener().lastLoadFailure() instanceof InvalidConfigurationException, "key store not reloaded or encountered expected failure");
        Assertions.assertEquals((Object)oldStore, (Object)this.getKeyStoreFromFileWatchListener(this.factory));
        writer = new FileWriter(filePath);
        try {
            writer.write(DefaultSslEngineFactoryTest.pemAsConfigValue(this.getEncryptedKey(), CERTCHAIN).value());
            writer.flush();
        }
        finally {
            writer.close();
        }
        TestUtils.waitForCondition(() -> this.factory.keyStoreFileWatchListener().lastLoadFailure() == null, "key store is reloaded");
        Assertions.assertEquals((Object)oldStore, (Object)this.getKeyStoreFromFileWatchListener(this.factory));
        writer = new FileWriter(filePath);
        try {
            writer.write(DefaultSslEngineFactoryTest.pemAsConfigValue(KEY, CERTCHAIN).value());
            writer.flush();
        }
        finally {
            writer.close();
        }
        TestUtils.waitForCondition(() -> this.factory.keyStoreFileWatchListener().lastLoadFailure() instanceof InvalidConfigurationException, "key store not reloaded or encountered expected failure");
        Assertions.assertEquals((Object)oldStore, (Object)this.getKeyStoreFromFileWatchListener(this.factory));
    }

    @Test
    public void testTrustStoreFileTriggerReload() throws Exception {
        this.configs.put("ssl.protocol", "TLSv1.2");
        this.configs.put("confluent.ssl.enable.dynamic.store.update", "true");
        String filePath = DefaultSslEngineFactoryTest.pemFilePath(CA1);
        this.configs.put("ssl.truststore.location", filePath);
        this.configs.put("ssl.truststore.type", "PEM");
        this.factory.configure(this.configs);
        TestUtils.waitForCondition(() -> this.factory.trustStoreFileWatchListener().initCompleted(), "FileWatchListener init is not completed");
        Thread.sleep(FileWatchService.MIN_WATCH_INTERVAL.toMillis());
        FileWriter writer = new FileWriter(filePath);
        writer.write(DefaultSslEngineFactoryTest.pemAsConfigValue(CA1, CA2).value());
        writer.flush();
        writer.close();
        this.verifyTrustStoreUpdate(this.factory);
    }

    @Test
    public void testTrustStoreFileFailedIfNotExistOrEmpty() throws Exception {
        File truststoreFile = TestUtils.tempFile("truststore", ".jks");
        Assertions.assertEquals((long)0L, (long)truststoreFile.length());
        this.configs.put("ssl.truststore.location", truststoreFile.getAbsolutePath());
        this.configs.put("ssl.truststore.password", KEY_PASSWORD);
        this.configs.put("ssl.truststore.type", "JKS");
        Assertions.assertThrows(KafkaException.class, () -> this.factory.configure(this.configs));
    }

    @Test
    public void testTrustStoreFileAutoCreateForMTls() throws Exception {
        String truststoreType = "JKS";
        if (this.getClass().getSimpleName().toLowerCase(Locale.ENGLISH).contains("fips")) {
            truststoreType = "BCFKS";
        }
        File truststoreFile = TestUtils.tempFile("truststore", "." + truststoreType.toLowerCase(Locale.ENGLISH));
        Assertions.assertEquals((long)0L, (long)truststoreFile.length());
        this.configs.put("confluent.mtls.enable", "true");
        this.configs.put("ssl.truststore.location", truststoreFile.getAbsolutePath());
        this.configs.put("ssl.truststore.password", KEY_PASSWORD);
        this.configs.put("ssl.truststore.type", truststoreType);
        this.factory.configure(this.configs);
        Assertions.assertTrue((truststoreFile.length() > 0L ? 1 : 0) != 0);
    }

    @Test
    public void testFileBasedStoreMaybeReload() throws Exception {
        String testFileName = DefaultSslEngineFactoryTest.pemFilePath(DefaultSslEngineFactoryTest.pemAsConfigValue(CERTCHAIN).value());
        String testFileHash = "CD28C63477CD627F35D7C8CE58B3B38F13639CCE772C7D602E77A7373FC56484";
        byte[] checksum = DefaultSslEngineFactory.FileBasedStore.checksum((String)testFileName);
        Assertions.assertEquals((Object)testFileHash, (Object)TestUtils.toHexStringUpperCase(checksum));
        String emptyFileName = TestUtils.tempFile().getAbsolutePath();
        String emptyFileHash = "E3B0C44298FC1C149AFBF4C8996FB92427AE41E4649B934CA495991B7852B855";
        checksum = DefaultSslEngineFactory.FileBasedStore.checksum((String)emptyFileName);
        Assertions.assertEquals((Object)emptyFileHash, (Object)TestUtils.toHexStringUpperCase(checksum));
        String fileDoesNotExist = "test-file-does-not-exist";
        Assertions.assertNull((Object)DefaultSslEngineFactory.FileBasedStore.checksum((String)fileDoesNotExist), (String)"checksum should be null if file does not exist");
    }

    @Test
    void testFileStoreWatchListenerOnUpdateFailure() throws Exception {
        double expectedFailureCount = 1.0;
        this.factory = new RenewalFailureSslEngineFactory(this.metrics);
        this.configs.put("ssl.key.password", this.getKeyPassword());
        this.factory.configure(this.configs);
        this.factory.keyStoreFileWatchListener().onUpdate();
        KafkaMetric certUpdateFailureTotal = this.getMetric("ssl-cert-renew-failure-total");
        Assertions.assertEquals((Object)expectedFailureCount, (Object)certUpdateFailureTotal.metricValue());
    }

    @Test
    public void returnsCipherSuitesWhenConfigured() {
        this.configs.put("ssl.cipher.suites", Arrays.asList("TLS_AES_128_GCM_SHA256", "TLS_AES_256_GCM_SHA384"));
        this.factory.configure(this.configs);
        Assertions.assertArrayEquals((Object[])new String[]{"TLS_AES_128_GCM_SHA256", "TLS_AES_256_GCM_SHA384"}, (Object[])this.factory.cipherSuites());
    }

    @Test
    public void returnsNullWhenCipherSuitesNotConfigured() {
        this.factory.configure(this.configs);
        Assertions.assertNull((Object)this.factory.cipherSuites());
    }

    private void verifyTrustStoreUpdate(DefaultSslEngineFactory factory) throws Exception {
        TestUtils.waitForCondition(() -> {
            ArrayList<String> aliases = Collections.list(factory.truststore().aliases());
            Collections.sort(aliases);
            return Arrays.asList("kafka0", "kafka1").equals(aliases);
        }, "store not reloaded");
        KeyStore trustStore = factory.truststore();
        Assertions.assertNotNull((Object)trustStore.getCertificate("kafka0"), (String)"Certificate not loaded");
        this.verifyGetKeyWithNoKey(trustStore, "kafka0");
        Assertions.assertNotNull((Object)trustStore.getCertificate("kafka1"), (String)"Certificate not loaded");
        this.verifyGetKeyWithNoKey(trustStore, "kafka1");
    }

    private KeyStore getKeyStoreFromFileWatchListener(DefaultSslEngineFactory factory) {
        return factory.keyStoreFileWatchListener() == null ? null : factory.keyStoreFileWatchListener().secretStore().get();
    }

    private KafkaMetric getMetric(String name) throws Exception {
        Optional<Map.Entry> metric = this.metrics.metrics().entrySet().stream().filter(entry -> ((MetricName)entry.getKey()).name().equals(name)).findFirst();
        if (!metric.isPresent()) {
            throw new Exception(String.format("Could not find metric called %s", name));
        }
        return (KafkaMetric)metric.get().getValue();
    }
}

