/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network.netty;

import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.group.ChannelGroup;
import io.netty.channel.group.DefaultChannelGroup;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2Settings;
import io.netty.handler.ssl.SslContext;
import io.netty.util.AttributeKey;
import io.netty.util.concurrent.DefaultEventExecutor;
import io.netty.util.concurrent.EventExecutor;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadFactory;
import java.util.function.BiFunction;
import org.apache.kafka.common.network.ConnectionMode;
import org.apache.kafka.common.network.EventLoopThreadFactory;
import org.apache.kafka.common.network.netty.NettyHttp2ConnectionInitializer;
import org.apache.kafka.common.network.netty.NettyRawBytesConnectionInitializer;
import org.apache.kafka.common.network.netty.NettyStream;
import org.apache.kafka.common.network.netty.Utils;
import org.apache.kafka.common.utils.LogContext;
import org.slf4j.Logger;

public class NettyServer {
    private final Logger log;
    private final ServerBootstrap serverBootstrap;
    private final ChannelGroup trackedChannels;
    private Channel listenerChannel;

    NettyServer(Logger log, ServerBootstrap serverBootstrap, ChannelGroup trackedChannels) {
        this.serverBootstrap = serverBootstrap;
        this.log = log;
        this.trackedChannels = trackedChannels;
    }

    public CompletableFuture<Integer> start(int port) {
        try {
            return this.start(new InetSocketAddress(port)).thenApply(address -> ((InetSocketAddress)address).getPort());
        }
        catch (Exception e) {
            this.log.error("Failed to start server on port {}", (Object)port, (Object)e);
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<SocketAddress> start(SocketAddress address) {
        CompletableFuture<SocketAddress> startFuture = new CompletableFuture<SocketAddress>();
        try {
            ChannelFuture channelFuture = this.serverBootstrap.bind(address);
            channelFuture.addListener(future -> {
                if (future.isSuccess()) {
                    this.listenerChannel = channelFuture.channel();
                    SocketAddress boundAddress = this.listenerChannel.localAddress();
                    this.log.info("Server started on address {}", (Object)boundAddress);
                    startFuture.complete(boundAddress);
                } else {
                    this.log.error("Failed to start server on address {}", (Object)address, (Object)future.cause());
                    startFuture.completeExceptionally(future.cause());
                }
            });
        }
        catch (Exception e) {
            this.log.error("Failed to start server on address {}", (Object)address, (Object)e);
            startFuture.completeExceptionally(e);
        }
        return startFuture;
    }

    public CompletableFuture<Void> shutdown() {
        CompletableFuture<Void> listenerChannelClose = this.shutdownListenerChannel();
        CompletableFuture<Void> trackedChannelsClose = this.shutdownTrackedChannels();
        CompletableFuture<Void> bossEventLoopGroupClose = this.shutdownBossEventLoopGroup();
        CompletableFuture<Void> workerEventLoopGroupClose = this.getShutdownWorkerEventLoopGroup();
        return CompletableFuture.allOf(listenerChannelClose, trackedChannelsClose, bossEventLoopGroupClose, workerEventLoopGroupClose);
    }

    public EventLoopGroup bossGroup() {
        return this.serverBootstrap.config().group();
    }

    public EventLoopGroup workerGroup() {
        return this.serverBootstrap.config().childGroup();
    }

    public Optional<SocketAddress> localAddress() {
        return this.listenerChannel != null ? Optional.ofNullable(this.listenerChannel.localAddress()) : Optional.empty();
    }

    private CompletableFuture<Void> getShutdownWorkerEventLoopGroup() {
        CompletableFuture<Void> shutdownFuture = new CompletableFuture<Void>();
        EventLoopGroup workerGroup = this.workerGroup();
        if (workerGroup != null && !workerGroup.isShuttingDown()) {
            workerGroup.shutdownGracefully().addListener(future -> {
                if (future.isSuccess()) {
                    this.log.info("Worker event loop group shutdown successfully");
                    shutdownFuture.complete(null);
                } else {
                    this.log.error("Failed to shutdown worker event loop group", future.cause());
                    shutdownFuture.completeExceptionally(future.cause());
                }
            });
        } else {
            this.log.warn("No worker event loop group found, skipping shutdown");
            shutdownFuture.complete(null);
        }
        return shutdownFuture;
    }

    private CompletableFuture<Void> shutdownBossEventLoopGroup() {
        CompletableFuture<Void> shutdownFuture = new CompletableFuture<Void>();
        EventLoopGroup bossGroup = this.bossGroup();
        if (bossGroup != null && !bossGroup.isShuttingDown()) {
            bossGroup.shutdownGracefully().addListener(future -> {
                if (future.isSuccess()) {
                    this.log.info("Boss event loop group shutdown successfully");
                    shutdownFuture.complete(null);
                } else {
                    this.log.error("Failed to shutdown boss event loop group", future.cause());
                    shutdownFuture.completeExceptionally(future.cause());
                }
            });
        } else {
            this.log.warn("No boss event loop group found, skipping shutdown");
            shutdownFuture.complete(null);
        }
        return shutdownFuture;
    }

    private CompletableFuture<Void> shutdownTrackedChannels() {
        CompletableFuture<Void> shutdownFuture = new CompletableFuture<Void>();
        this.trackedChannels.close().addListener(future -> {
            if (future.isSuccess()) {
                this.log.info("All child channels closed successfully");
                shutdownFuture.complete(null);
            } else {
                this.log.error("Failed to close all child channels", future.cause());
                shutdownFuture.completeExceptionally(future.cause());
            }
        });
        return shutdownFuture;
    }

    public ChannelGroup trackedChannels() {
        return this.trackedChannels;
    }

    private CompletableFuture<Void> shutdownListenerChannel() {
        CompletableFuture<Void> shutdownFuture = new CompletableFuture<Void>();
        if (this.listenerChannel != null) {
            this.listenerChannel.close().addListener(future -> {
                if (future.isSuccess()) {
                    this.log.info("Listener channel closed successfully");
                    shutdownFuture.complete(null);
                } else {
                    this.log.error("Failed to close listener channel", future.cause());
                    shutdownFuture.completeExceptionally(future.cause());
                }
            });
            this.listenerChannel = null;
        } else {
            this.log.warn("No listener channel found, skipping close");
            shutdownFuture.complete(null);
        }
        return shutdownFuture;
    }

    public static class Builder {
        private int bossThreads = 1;
        private int workerThreads = 0;
        private EventLoopGroup workerGroup;
        private final LogContext logContext;
        private final NettyServerProtocol protocol;
        private final ThreadFactory bossThreadFactory;
        private final ThreadFactory workerThreadFactory;
        private final Map<ChannelOption<?>, Object> options = new LinkedHashMap();
        private final Map<AttributeKey<?>, Object> attrs = new HashMap();
        private ChannelHandler handler;
        private final Map<ChannelOption<?>, Object> childOptions = new LinkedHashMap();
        private final Map<AttributeKey<?>, Object> childAttrs = new HashMap();
        private final SslContext sslContext;
        private final boolean addLoggingHandler;
        private final BiFunction<NettyStream, Http2Headers, NettyStream.StreamHandler> handlerCreator;
        private Http2Settings http2Settings;
        private boolean flowControlEnabled = false;
        private int http2ConnectionWindowSize = 0x1E00000;

        Builder(LogContext logContext, String listenerName, NettyServerProtocol protocol, SslContext sslContext, BiFunction<NettyStream, Http2Headers, NettyStream.StreamHandler> handlerCreator, boolean addLoggingHandler) {
            this.logContext = logContext;
            this.protocol = protocol;
            this.sslContext = sslContext;
            this.addLoggingHandler = addLoggingHandler;
            this.handlerCreator = handlerCreator;
            this.bossThreadFactory = EventLoopThreadFactory.create(listenerName + "-netty-" + protocol.name().toLowerCase(Locale.ROOT) + "-server-boss-");
            this.workerThreadFactory = EventLoopThreadFactory.create(listenerName + "-netty-" + protocol.name().toLowerCase(Locale.ROOT) + "-server-worker-");
        }

        public static Builder newServerBuilder(LogContext logContext, String listenerName, SslContext sslContext, BiFunction<NettyStream, Http2Headers, NettyStream.StreamHandler> handlerCreator, NettyServerProtocol protocol, boolean addLoggingHandler) {
            return new Builder(logContext, listenerName, protocol, sslContext, handlerCreator, addLoggingHandler);
        }

        public Builder withBossThreads(int nThreads) {
            if (nThreads <= 0) {
                throw new IllegalArgumentException("Number of boss threads must be greater than 0");
            }
            this.bossThreads = nThreads;
            return this;
        }

        public Builder withWorkerGroup(EventLoopGroup workerGroup) {
            if (this.workerThreads > 0) {
                throw new IllegalArgumentException("Cannot set both worker group and worker threads");
            }
            this.workerGroup = workerGroup;
            return this;
        }

        public Builder withWorkerThreads(int nThreads) {
            if (nThreads <= 0) {
                throw new IllegalArgumentException("Number of worker threads must be greater than 0");
            }
            if (this.workerGroup != null) {
                throw new IllegalArgumentException("Cannot set both worker group and worker threads");
            }
            this.workerThreads = nThreads;
            return this;
        }

        public Builder withOption(ChannelOption<?> option, Object value) {
            this.options.put(option, value);
            return this;
        }

        public Builder withAttr(AttributeKey<?> key, Object value) {
            this.attrs.put(key, value);
            return this;
        }

        public Builder withHandler(ChannelHandler handler) {
            this.handler = handler;
            return this;
        }

        public Builder withHttp2Settings(Http2Settings http2Settings) {
            this.http2Settings = http2Settings;
            return this;
        }

        public Builder withHttp2ConnectionWindowSize(int http2ConnectionWindowSize) {
            this.http2ConnectionWindowSize = http2ConnectionWindowSize;
            return this;
        }

        public Builder withFlowControlEnabled(boolean flowControlEnabled) {
            this.flowControlEnabled = flowControlEnabled;
            return this;
        }

        public Builder withChildOption(ChannelOption<?> option, Object value) {
            this.childOptions.put(option, value);
            return this;
        }

        public Builder withChildAttr(AttributeKey<?> key, Object value) {
            this.childAttrs.put(key, value);
            return this;
        }

        public NettyServer build() {
            return this.build(false);
        }

        private ChannelHandler rawBytesConnectionInitializer(ChannelGroup trackedChannels) {
            if (this.handlerCreator == null) {
                throw new IllegalArgumentException("Handler creator should be set.");
            }
            return new NettyRawBytesConnectionInitializer(trackedChannels, this.sslContext, this.handlerCreator, this.logContext, this.addLoggingHandler, this.flowControlEnabled);
        }

        private ChannelHandler http2ConnectionInitializer(ChannelGroup trackedChannels) {
            if (this.handlerCreator == null) {
                throw new IllegalArgumentException("Handler creator should be set.");
            }
            if (this.http2Settings == null) {
                throw new IllegalArgumentException("Http2Settings should be set.");
            }
            return new NettyHttp2ConnectionInitializer(ConnectionMode.SERVER, trackedChannels, this.sslContext, this.http2Settings, (ChannelHandler)new NettyHttp2ConnectionInitializer.NettyHttp2StreamInitializer(this.handlerCreator, trackedChannels, this.logContext, this.flowControlEnabled), this.logContext, this.http2ConnectionWindowSize, this.addLoggingHandler);
        }

        public NettyServer build(boolean isUds) {
            if (this.workerThreads == 0 && this.workerGroup == null) {
                throw new IllegalArgumentException("Either worker group or worker threads must be set");
            }
            if (this.workerGroup == null) {
                this.workerGroup = Utils.createEventLoopGroup(this.workerThreads, this.workerThreadFactory);
            }
            EventLoopGroup bossGroup = Utils.createEventLoopGroup(this.bossThreads, this.bossThreadFactory);
            ServerBootstrap bootstrap = new ServerBootstrap();
            bootstrap.group(bossGroup, this.workerGroup);
            bootstrap.channel(Utils.serverChannelTypeForEventLoop(this.workerGroup, isUds));
            for (Map.Entry<ChannelOption<?>, Object> entry : this.options.entrySet()) {
                bootstrap.option(entry.getKey(), entry.getValue());
            }
            for (Map.Entry<ChannelOption<?>, Object> entry : this.attrs.entrySet()) {
                bootstrap.attr((AttributeKey)entry.getKey(), entry.getValue());
            }
            if (this.handler != null) {
                bootstrap.handler(this.handler);
            }
            for (Map.Entry<ChannelOption<?>, Object> entry : this.childOptions.entrySet()) {
                bootstrap.childOption(entry.getKey(), entry.getValue());
            }
            for (Map.Entry<ChannelOption<?>, Object> entry : this.childAttrs.entrySet()) {
                bootstrap.childAttr((AttributeKey)entry.getKey(), entry.getValue());
            }
            DefaultChannelGroup trackedChannels = new DefaultChannelGroup((EventExecutor)new DefaultEventExecutor());
            ChannelHandler channelHandler = this.protocol == NettyServerProtocol.HTTP2 ? this.http2ConnectionInitializer((ChannelGroup)trackedChannels) : this.rawBytesConnectionInitializer((ChannelGroup)trackedChannels);
            bootstrap.childHandler(channelHandler);
            return new NettyServer(this.logContext.logger(NettyServer.class), bootstrap, (ChannelGroup)trackedChannels);
        }
    }

    public static enum NettyServerProtocol {
        HTTP2,
        SOCKET;

    }
}

