/*
 * Decompiled with CFR 0.152.
 */
package apoc.spatial;

import apoc.ApocConfiguration;
import apoc.util.JsonUtil;
import apoc.util.MapUtil;
import apoc.util.Util;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.logging.Log;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;
import org.neo4j.procedure.TerminationGuard;

public class Geocode {
    public static final int MAX_RESULTS = 100;
    public static final String PREFIX = "spatial.geocode";
    public static final String GEOCODE_PROVIDER_KEY = "provider";
    @Context
    public GraphDatabaseService db;
    @Context
    public TerminationGuard terminationGuard;
    @Context
    public Log log;

    private GeocodeSupplier getSupplier() {
        Map<String, Object> activeConfig = ApocConfiguration.get(PREFIX);
        if (activeConfig.containsKey(GEOCODE_PROVIDER_KEY)) {
            String supplier;
            switch (supplier = activeConfig.get(GEOCODE_PROVIDER_KEY).toString().toLowerCase()) {
                case "google": {
                    return new GoogleSupplier(activeConfig, this.terminationGuard);
                }
                case "osm": {
                    return new OSMSupplier(activeConfig, this.terminationGuard);
                }
            }
            return new SupplierWithKey(activeConfig, this.terminationGuard, supplier);
        }
        return new OSMSupplier(activeConfig, this.terminationGuard);
    }

    @Procedure
    @Description(value="apoc.spatial.geocodeOnce('address') YIELD location, latitude, longitude, description, osmData - look up geographic location of address from openstreetmap geocoding service")
    public Stream<GeoCodeResult> geocodeOnce(@Name(value="location") String address) throws UnsupportedEncodingException {
        return this.geocode(address, 1L);
    }

    @Procedure
    @Description(value="apoc.spatial.geocode('address') YIELD location, latitude, longitude, description, osmData - look up geographic location of address from openstreetmap geocoding service")
    public Stream<GeoCodeResult> geocode(@Name(value="location") String address, @Name(value="maxResults") long maxResults) {
        return this.getSupplier().geocode(address, maxResults == 0L ? 100L : Math.min(Math.max(maxResults, 1L), 100L));
    }

    public static class GeoCodeResult {
        public final Map<String, Object> location;
        public final Map<String, Object> data;
        public final Double latitude;
        public final Double longitude;
        public final String description;

        public GeoCodeResult(Double latitude, Double longitude, String description, Map<String, Object> data) {
            this.data = data;
            this.latitude = latitude;
            this.longitude = longitude;
            this.description = description;
            this.location = MapUtil.map("latitude", latitude, "longitude", longitude, "description", description);
        }
    }

    class GoogleSupplier
    implements GeocodeSupplier {
        private final Throttler throttler;
        private String baseUrl;

        public GoogleSupplier(Map<String, Object> config, TerminationGuard terminationGuard) {
            this.throttler = new Throttler(terminationGuard, Util.toLong(config.getOrDefault("google.throttle", Throttler.DEFAULT_THROTTLE)));
            this.baseUrl = String.format("https://maps.googleapis.com/maps/api/geocode/json?%s&address=", this.credentials(config));
        }

        private String credentials(Map<String, Object> config) {
            if (config.containsKey("google.client") && config.containsKey("google.signature")) {
                return "client=" + config.get("google.client") + "&signature=" + config.get("google.signature");
            }
            if (config.containsKey("google.key")) {
                return "key=" + config.get("google.key");
            }
            return "auth=free";
        }

        @Override
        public Stream<GeoCodeResult> geocode(String address, long maxResults) {
            Object results;
            if (address.length() < 1) {
                return Stream.empty();
            }
            this.throttler.waitForThrottle();
            Object value = JsonUtil.loadJson(this.baseUrl + Util.encodeUrlComponent(address)).findFirst().orElse(null);
            if (value instanceof Map && (results = ((Map)value).get("results")) instanceof List) {
                return ((List)results).stream().limit(maxResults).map(data -> {
                    Map location = (Map)((Map)data.get("geometry")).get("location");
                    return new GeoCodeResult(Util.toDouble(location.get("lat")), Util.toDouble(location.get("lng")), String.valueOf(data.get("formatted_address")), (Map<String, Object>)data);
                });
            }
            throw new RuntimeException("Can't parse geocoding results " + value);
        }
    }

    private static class OSMSupplier
    implements GeocodeSupplier {
        public static final String OSM_URL = "http://nominatim.openstreetmap.org/search.php?format=json&q=";
        private Throttler throttler;

        public OSMSupplier(Map<String, Object> config, TerminationGuard terminationGuard) {
            this.throttler = new Throttler(terminationGuard, Util.toLong(config.getOrDefault("osm.throttle", Throttler.DEFAULT_THROTTLE)));
        }

        @Override
        public Stream<GeoCodeResult> geocode(String address, long maxResults) {
            this.throttler.waitForThrottle();
            Object value = JsonUtil.loadJson(OSM_URL + Util.encodeUrlComponent(address)).findFirst().orElse(null);
            if (value instanceof List) {
                return ((List)value).stream().limit(maxResults).map(data -> new GeoCodeResult(Util.toDouble(data.get("lat")), Util.toDouble(data.get("lon")), String.valueOf(data.get("display_name")), (Map<String, Object>)data));
            }
            throw new RuntimeException("Can't parse geocoding results " + value);
        }
    }

    private static class SupplierWithKey
    implements GeocodeSupplier {
        private static final String[] FORMATTED_KEYS = new String[]{"formatted", "formatted_address", "address", "description", "display_name"};
        private static final String[] LAT_KEYS = new String[]{"lat", "latitude"};
        private static final String[] LNG_KEYS = new String[]{"lng", "longitude", "lon"};
        private Throttler throttler;
        private String configBase;
        private String urlTemplate;

        public SupplierWithKey(Map<String, Object> config, TerminationGuard terminationGuard, String provider) {
            this.configBase = provider;
            if (!config.containsKey(this.configKey("url"))) {
                throw new IllegalArgumentException("Missing 'url' for geocode provider: " + provider);
            }
            this.urlTemplate = config.get(this.configKey("url")).toString();
            if (!this.urlTemplate.contains("PLACE")) {
                throw new IllegalArgumentException("Missing 'PLACE' in url template: " + this.urlTemplate);
            }
            if (this.urlTemplate.contains("KEY") && !config.containsKey(this.configKey("key"))) {
                throw new IllegalArgumentException("Missing 'key' for geocode provider: " + provider);
            }
            String key = config.get(this.configKey("key")).toString();
            this.urlTemplate = this.urlTemplate.replace("KEY", key);
            this.throttler = new Throttler(terminationGuard, Util.toLong(ApocConfiguration.get(this.configKey("throttle"), Throttler.DEFAULT_THROTTLE)));
        }

        @Override
        public Stream<GeoCodeResult> geocode(String address, long maxResults) {
            Object results;
            this.throttler.waitForThrottle();
            String url = this.urlTemplate.replace("PLACE", Util.encodeUrlComponent(address));
            Object value = JsonUtil.loadJson(url).findFirst().orElse(null);
            if (value instanceof List) {
                return this.findResults(value, maxResults);
            }
            if (value instanceof Map && (results = ((Map)value).get("results")) instanceof List) {
                return this.findResults((List)results, maxResults);
            }
            throw new RuntimeException("Can't parse geocoding results " + value);
        }

        private Stream<GeoCodeResult> findResults(List<Map<String, Object>> results, long maxResults) {
            return results.stream().limit(maxResults).map(data -> {
                String description = this.findFirstEntry((Map<String, Object>)data, FORMATTED_KEYS);
                Map location = (Map)data.get("geometry");
                if (location.containsKey("location")) {
                    location = (Map)location.get("location");
                }
                String lat = this.findFirstEntry(location, LAT_KEYS);
                String lng = this.findFirstEntry(location, LNG_KEYS);
                return new GeoCodeResult(Util.toDouble(lat), Util.toDouble(lng), description, (Map<String, Object>)data);
            });
        }

        private String findFirstEntry(Map<String, Object> map, String[] keys) {
            for (String key : keys) {
                if (!map.containsKey(key)) continue;
                return String.valueOf(map.get(key));
            }
            return "";
        }

        private String configKey(String name) {
            return this.configBase + "." + name;
        }
    }

    private static class Throttler {
        private final TerminationGuard terminationGuard;
        private long throttleInMs;
        private static long lastCallTime = 0L;
        private static long DEFAULT_THROTTLE = 5000L;
        private static long MAX_THROTTLE = 3600000L;

        public Throttler(TerminationGuard terminationGuard, long throttle) {
            this.terminationGuard = terminationGuard;
            throttle = Math.min(throttle, MAX_THROTTLE);
            if (throttle < 0L) {
                throttle = DEFAULT_THROTTLE;
            }
            this.throttleInMs = throttle;
        }

        private void waitForThrottle() {
            long msSinceLastCall = System.currentTimeMillis() - lastCallTime;
            while (msSinceLastCall < this.throttleInMs) {
                try {
                    this.terminationGuard.check();
                    long msToWait = this.throttleInMs - msSinceLastCall;
                    Thread.sleep(Math.min(msToWait, 1000L));
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                msSinceLastCall = System.currentTimeMillis() - lastCallTime;
            }
            lastCallTime = System.currentTimeMillis();
        }
    }

    static interface GeocodeSupplier {
        public Stream<GeoCodeResult> geocode(String var1, long var2);
    }
}

