/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a condition to be compared with an attribute value. This condition can be used with
 * <code>DeleteItem</code>, <code>PutItem</code>, or <code>UpdateItem</code> operations; if the comparison evaluates to
 * true, the operation succeeds; if not, the operation fails. You can use <code>ExpectedAttributeValue</code> in one of
 * two different ways:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Use <code>AttributeValueList</code> to specify one or more values to compare against an attribute. Use
 * <code>ComparisonOperator</code> to specify how you want to perform the comparison. If the comparison evaluates to
 * true, then the conditional operation succeeds.
 * </p>
 * </li>
 * <li>
 * <p>
 * Use <code>Value</code> to specify a value that DynamoDB will compare against an attribute. If the values match, then
 * <code>ExpectedAttributeValue</code> evaluates to true and the conditional operation succeeds. Optionally, you can
 * also set <code>Exists</code> to false, indicating that you <i>do not</i> expect to find the attribute value in the
 * table. In this case, the conditional operation succeeds only if the comparison evaluates to false.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <code>Value</code> and <code>Exists</code> are incompatible with <code>AttributeValueList</code> and
 * <code>ComparisonOperator</code>. Note that if you use both sets of parameters at once, DynamoDB will return a
 * <code>ValidationException</code> exception.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExpectedAttributeValue implements SdkPojo, Serializable,
        ToCopyableBuilder<ExpectedAttributeValue.Builder, ExpectedAttributeValue> {
    private static final SdkField<AttributeValue> VALUE_FIELD = SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
            .getter(getter(ExpectedAttributeValue::value)).setter(setter(Builder::value)).constructor(AttributeValue::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<Boolean> EXISTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ExpectedAttributeValue::exists)).setter(setter(Builder::exists))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exists").build()).build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ExpectedAttributeValue::comparisonOperatorAsString)).setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final SdkField<List<AttributeValue>> ATTRIBUTE_VALUE_LIST_FIELD = SdkField
            .<List<AttributeValue>> builder(MarshallingType.LIST)
            .getter(getter(ExpectedAttributeValue::attributeValueList))
            .setter(setter(Builder::attributeValueList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeValueList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VALUE_FIELD, EXISTS_FIELD,
            COMPARISON_OPERATOR_FIELD, ATTRIBUTE_VALUE_LIST_FIELD));

    private static final long serialVersionUID = 1L;

    private final AttributeValue value;

    private final Boolean exists;

    private final String comparisonOperator;

    private final List<AttributeValue> attributeValueList;

    private ExpectedAttributeValue(BuilderImpl builder) {
        this.value = builder.value;
        this.exists = builder.exists;
        this.comparisonOperator = builder.comparisonOperator;
        this.attributeValueList = builder.attributeValueList;
    }

    /**
     * <p>
     * Represents the data for the expected attribute.
     * </p>
     * <p>
     * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data
     * itself.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
     * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * 
     * @return Represents the data for the expected attribute.</p>
     *         <p>
     *         Each attribute value is described as a name-value pair. The name is the data type, and the value is the
     *         data itself.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
     *         >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public AttributeValue value() {
        return value;
    }

    /**
     * <p>
     * Causes DynamoDB to evaluate the value before attempting a conditional operation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>Exists</code> is <code>true</code>, DynamoDB will check to see if that attribute value already exists in
     * the table. If it is found, then the operation succeeds. If it is not found, the operation fails with a
     * <code>ConditionCheckFailedException</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>Exists</code> is <code>false</code>, DynamoDB assumes that the attribute value does not exist in the
     * table. If in fact the value does not exist, then the assumption is valid and the operation succeeds. If the value
     * is found, despite the assumption that it does not exist, the operation fails with a
     * <code>ConditionCheckFailedException</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The default setting for <code>Exists</code> is <code>true</code>. If you supply a <code>Value</code> all by
     * itself, DynamoDB assumes the attribute exists: You don't have to set <code>Exists</code> to <code>true</code>,
     * because it is implied.
     * </p>
     * <p>
     * DynamoDB returns a <code>ValidationException</code> if:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Exists</code> is <code>true</code> but there is no <code>Value</code> to check. (You expect a value to
     * exist, but don't specify what that value is.)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Exists</code> is <code>false</code> but you also provide a <code>Value</code>. (You cannot expect an
     * attribute to have a value, while also expecting it not to exist.)
     * </p>
     * </li>
     * </ul>
     * 
     * @return Causes DynamoDB to evaluate the value before attempting a conditional operation:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>Exists</code> is <code>true</code>, DynamoDB will check to see if that attribute value already
     *         exists in the table. If it is found, then the operation succeeds. If it is not found, the operation fails
     *         with a <code>ConditionCheckFailedException</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>Exists</code> is <code>false</code>, DynamoDB assumes that the attribute value does not exist in
     *         the table. If in fact the value does not exist, then the assumption is valid and the operation succeeds.
     *         If the value is found, despite the assumption that it does not exist, the operation fails with a
     *         <code>ConditionCheckFailedException</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The default setting for <code>Exists</code> is <code>true</code>. If you supply a <code>Value</code> all
     *         by itself, DynamoDB assumes the attribute exists: You don't have to set <code>Exists</code> to
     *         <code>true</code>, because it is implied.
     *         </p>
     *         <p>
     *         DynamoDB returns a <code>ValidationException</code> if:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Exists</code> is <code>true</code> but there is no <code>Value</code> to check. (You expect a value
     *         to exist, but don't specify what that value is.)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Exists</code> is <code>false</code> but you also provide a <code>Value</code>. (You cannot expect
     *         an attribute to have a value, while also expecting it not to exist.)
     *         </p>
     *         </li>
     */
    public Boolean exists() {
        return exists;
    }

    /**
     * <p>
     * A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals, greater than,
     * less than, etc.
     * </p>
     * <p>
     * The following comparison operators are available:
     * </p>
     * <p>
     * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     * </p>
     * <p>
     * The following are descriptions of each comparison operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> element of a
     * different type than the one provided in the request, the value does not match. For example,
     * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     * <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, Binary,
     * String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a different type
     * than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not
     * equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LE</code> : Less than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LT</code> : Less than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
     * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than the
     * one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GE</code> : Greater than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GT</code> : Greater than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not relevant to
     * the <code>NOT_NULL</code> comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean <code>false</code>.
     * This is because the attribute "<code>a</code>" exists; its data type is not relevant to the <code>NULL</code>
     * comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is of type String, then the operator checks
     * for a substring match. If the target attribute of the comparison is of type Binary, then the operator looks for a
     * subsequence of the target that matches the input. If the target attribute of the comparison is a set ("
     * <code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it finds an
     * exact match with any member of the set.
     * </p>
     * <p>
     * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is a String, then the operator checks for
     * the absence of a substring match. If the target attribute of the comparison is Binary, then the operator checks
     * for the absence of a subsequence of the target that matches the input. If the target attribute of the comparison
     * is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it
     * <i>does not</i> find an exact match with any member of the set.
     * </p>
     * <p>
     * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BEGINS_WITH</code> : Checks for a prefix.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary (not a
     * Number or a set type). The target attribute of the comparison must be of type String or Binary (not a Number or a
     * set type).
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>IN</code> : Checks for matching elements in a list.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
     * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of the
     * input are equal to the item attribute, the expression evaluates to true.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
     * </p>
     * <p>
     * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type, either
     * String, Number, or Binary (not a set type). A target attribute matches if the target value is greater than, or
     * equal to, the first element and less than, or equal to, the second element. If an item contains an
     * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does not
     * match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
     * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals,
     *         greater than, less than, etc.</p>
     *         <p>
     *         The following comparison operators are available:
     *         </p>
     *         <p>
     *         <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     *         </p>
     *         <p>
     *         The following are descriptions of each comparison operator.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
     *         element of a different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LE</code> : Less than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LT</code> : Less than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
     *         type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code>
     *         does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GE</code> : Greater than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GT</code> : Greater than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     *         <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     *         <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
     *         relevant to the <code>NOT_NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the nonexistence of an attribute, not its data type. If the data type of
     *         attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
     *         <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant
     *         to the <code>NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
     *         operator checks for a substring match. If the target attribute of the comparison is of type Binary, then
     *         the operator looks for a subsequence of the target that matches the input. If the target attribute of the
     *         comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator
     *         evaluates to true if it finds an exact match with any member of the set.
     *         </p>
     *         <p>
     *         CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the
     *         operator checks for the absence of a substring match. If the target attribute of the comparison is
     *         Binary, then the operator checks for the absence of a subsequence of the target that matches the input.
     *         If the target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "
     *         <code>BS</code>"), then the operator evaluates to true if it <i>does not</i> find an exact match with any
     *         member of the set.
     *         </p>
     *         <p>
     *         NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BEGINS_WITH</code> : Checks for a prefix.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
     *         (not a Number or a set type). The target attribute of the comparison must be of type String or Binary
     *         (not a Number or a set type).
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>IN</code> : Checks for matching elements in a list.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
     *         String, Number, or Binary. These attributes are compared against an existing attribute of an item. If any
     *         elements of the input are equal to the item attribute, the expression evaluates to true.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
     *         value.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
     *         either String, Number, or Binary (not a set type). A target attribute matches if the target value is
     *         greater than, or equal to, the first element and less than, or equal to, the second element. If an item
     *         contains an <code>AttributeValue</code> element of a different type than the one provided in the request,
     *         the value does not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>.
     *         Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     *         </p>
     *         </li>
     * @see ComparisonOperator
     */
    public ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals, greater than,
     * less than, etc.
     * </p>
     * <p>
     * The following comparison operators are available:
     * </p>
     * <p>
     * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     * </p>
     * <p>
     * The following are descriptions of each comparison operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> element of a
     * different type than the one provided in the request, the value does not match. For example,
     * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     * <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, Binary,
     * String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a different type
     * than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not
     * equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LE</code> : Less than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LT</code> : Less than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
     * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than the
     * one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GE</code> : Greater than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GT</code> : Greater than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not relevant to
     * the <code>NOT_NULL</code> comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean <code>false</code>.
     * This is because the attribute "<code>a</code>" exists; its data type is not relevant to the <code>NULL</code>
     * comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is of type String, then the operator checks
     * for a substring match. If the target attribute of the comparison is of type Binary, then the operator looks for a
     * subsequence of the target that matches the input. If the target attribute of the comparison is a set ("
     * <code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it finds an
     * exact match with any member of the set.
     * </p>
     * <p>
     * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is a String, then the operator checks for
     * the absence of a substring match. If the target attribute of the comparison is Binary, then the operator checks
     * for the absence of a subsequence of the target that matches the input. If the target attribute of the comparison
     * is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it
     * <i>does not</i> find an exact match with any member of the set.
     * </p>
     * <p>
     * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BEGINS_WITH</code> : Checks for a prefix.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary (not a
     * Number or a set type). The target attribute of the comparison must be of type String or Binary (not a Number or a
     * set type).
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>IN</code> : Checks for matching elements in a list.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
     * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of the
     * input are equal to the item attribute, the expression evaluates to true.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
     * </p>
     * <p>
     * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type, either
     * String, Number, or Binary (not a set type). A target attribute matches if the target value is greater than, or
     * equal to, the first element and less than, or equal to, the second element. If an item contains an
     * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does not
     * match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
     * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals,
     *         greater than, less than, etc.</p>
     *         <p>
     *         The following comparison operators are available:
     *         </p>
     *         <p>
     *         <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     *         </p>
     *         <p>
     *         The following are descriptions of each comparison operator.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
     *         element of a different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LE</code> : Less than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LT</code> : Less than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
     *         type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code>
     *         does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GE</code> : Greater than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GT</code> : Greater than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     *         <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     *         <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
     *         relevant to the <code>NOT_NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the nonexistence of an attribute, not its data type. If the data type of
     *         attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
     *         <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant
     *         to the <code>NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
     *         operator checks for a substring match. If the target attribute of the comparison is of type Binary, then
     *         the operator looks for a subsequence of the target that matches the input. If the target attribute of the
     *         comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator
     *         evaluates to true if it finds an exact match with any member of the set.
     *         </p>
     *         <p>
     *         CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the
     *         operator checks for the absence of a substring match. If the target attribute of the comparison is
     *         Binary, then the operator checks for the absence of a subsequence of the target that matches the input.
     *         If the target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "
     *         <code>BS</code>"), then the operator evaluates to true if it <i>does not</i> find an exact match with any
     *         member of the set.
     *         </p>
     *         <p>
     *         NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BEGINS_WITH</code> : Checks for a prefix.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
     *         (not a Number or a set type). The target attribute of the comparison must be of type String or Binary
     *         (not a Number or a set type).
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>IN</code> : Checks for matching elements in a list.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
     *         String, Number, or Binary. These attributes are compared against an existing attribute of an item. If any
     *         elements of the input are equal to the item attribute, the expression evaluates to true.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
     *         value.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
     *         either String, Number, or Binary (not a set type). A target attribute matches if the target value is
     *         greater than, or equal to, the first element and less than, or equal to, the second element. If an item
     *         contains an <code>AttributeValue</code> element of a different type than the one provided in the request,
     *         the value does not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>.
     *         Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     *         </p>
     *         </li>
     * @see ComparisonOperator
     */
    public String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * Returns true if the AttributeValueList property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAttributeValueList() {
        return attributeValueList != null && !(attributeValueList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more values to evaluate against the supplied attribute. The number of values in the list depends on the
     * <code>ComparisonOperator</code> being used.
     * </p>
     * <p>
     * For type Number, value comparisons are numeric.
     * </p>
     * <p>
     * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
     * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>. For a
     * list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://en.wikipedia
     * .org/wiki/ASCII#ASCII_printable_characters</a>.
     * </p>
     * <p>
     * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
     * </p>
     * <p>
     * For information on specifying data types in JSON, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataFormat.html">JSON Data Format</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeValueList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more values to evaluate against the supplied attribute. The number of values in the list depends
     *         on the <code>ComparisonOperator</code> being used.</p>
     *         <p>
     *         For type Number, value comparisons are numeric.
     *         </p>
     *         <p>
     *         String value comparisons for greater than, equals, or less than are based on ASCII character code values.
     *         For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
     *         <code>B</code>. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
     *         >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
     *         </p>
     *         <p>
     *         For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
     *         </p>
     *         <p>
     *         For information on specifying data types in JSON, see <a
     *         href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataFormat.html">JSON Data
     *         Format</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public List<AttributeValue> attributeValueList() {
        return attributeValueList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(exists());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(attributeValueList());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExpectedAttributeValue)) {
            return false;
        }
        ExpectedAttributeValue other = (ExpectedAttributeValue) obj;
        return Objects.equals(value(), other.value()) && Objects.equals(exists(), other.exists())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(attributeValueList(), other.attributeValueList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ExpectedAttributeValue").add("Value", value()).add("Exists", exists())
                .add("ComparisonOperator", comparisonOperatorAsString()).add("AttributeValueList", attributeValueList()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "Exists":
            return Optional.ofNullable(clazz.cast(exists()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "AttributeValueList":
            return Optional.ofNullable(clazz.cast(attributeValueList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ExpectedAttributeValue, T> g) {
        return obj -> g.apply((ExpectedAttributeValue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExpectedAttributeValue> {
        /**
         * <p>
         * Represents the data for the expected attribute.
         * </p>
         * <p>
         * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data
         * itself.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
         * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param value
         *        Represents the data for the expected attribute.</p>
         *        <p>
         *        Each attribute value is described as a name-value pair. The name is the data type, and the value is
         *        the data itself.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
         *        >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(AttributeValue value);

        /**
         * <p>
         * Represents the data for the expected attribute.
         * </p>
         * <p>
         * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data
         * itself.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
         * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link AttributeValue.Builder} avoiding the need to
         * create one manually via {@link AttributeValue#builder()}.
         *
         * When the {@link Consumer} completes, {@link AttributeValue.Builder#build()} is called immediately and its
         * result is passed to {@link #value(AttributeValue)}.
         * 
         * @param value
         *        a consumer that will call methods on {@link AttributeValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #value(AttributeValue)
         */
        default Builder value(Consumer<AttributeValue.Builder> value) {
            return value(AttributeValue.builder().applyMutation(value).build());
        }

        /**
         * <p>
         * Causes DynamoDB to evaluate the value before attempting a conditional operation:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>Exists</code> is <code>true</code>, DynamoDB will check to see if that attribute value already
         * exists in the table. If it is found, then the operation succeeds. If it is not found, the operation fails
         * with a <code>ConditionCheckFailedException</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>Exists</code> is <code>false</code>, DynamoDB assumes that the attribute value does not exist in the
         * table. If in fact the value does not exist, then the assumption is valid and the operation succeeds. If the
         * value is found, despite the assumption that it does not exist, the operation fails with a
         * <code>ConditionCheckFailedException</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The default setting for <code>Exists</code> is <code>true</code>. If you supply a <code>Value</code> all by
         * itself, DynamoDB assumes the attribute exists: You don't have to set <code>Exists</code> to <code>true</code>
         * , because it is implied.
         * </p>
         * <p>
         * DynamoDB returns a <code>ValidationException</code> if:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Exists</code> is <code>true</code> but there is no <code>Value</code> to check. (You expect a value to
         * exist, but don't specify what that value is.)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Exists</code> is <code>false</code> but you also provide a <code>Value</code>. (You cannot expect an
         * attribute to have a value, while also expecting it not to exist.)
         * </p>
         * </li>
         * </ul>
         * 
         * @param exists
         *        Causes DynamoDB to evaluate the value before attempting a conditional operation:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>Exists</code> is <code>true</code>, DynamoDB will check to see if that attribute value
         *        already exists in the table. If it is found, then the operation succeeds. If it is not found, the
         *        operation fails with a <code>ConditionCheckFailedException</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>Exists</code> is <code>false</code>, DynamoDB assumes that the attribute value does not exist
         *        in the table. If in fact the value does not exist, then the assumption is valid and the operation
         *        succeeds. If the value is found, despite the assumption that it does not exist, the operation fails
         *        with a <code>ConditionCheckFailedException</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The default setting for <code>Exists</code> is <code>true</code>. If you supply a <code>Value</code>
         *        all by itself, DynamoDB assumes the attribute exists: You don't have to set <code>Exists</code> to
         *        <code>true</code>, because it is implied.
         *        </p>
         *        <p>
         *        DynamoDB returns a <code>ValidationException</code> if:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Exists</code> is <code>true</code> but there is no <code>Value</code> to check. (You expect a
         *        value to exist, but don't specify what that value is.)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Exists</code> is <code>false</code> but you also provide a <code>Value</code>. (You cannot
         *        expect an attribute to have a value, while also expecting it not to exist.)
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exists(Boolean exists);

        /**
         * <p>
         * A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals, greater
         * than, less than, etc.
         * </p>
         * <p>
         * The following comparison operators are available:
         * </p>
         * <p>
         * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         * </p>
         * <p>
         * The following are descriptions of each comparison operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
         * element of a different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
         * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
         * different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LE</code> : Less than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LT</code> : Less than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
         * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
         * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
         * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GE</code> : Greater than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GT</code> : Greater than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
         * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
         * relevant to the <code>NOT_NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
         * <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant to
         * the <code>NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
         * operator checks for a substring match. If the target attribute of the comparison is of type Binary, then the
         * operator looks for a subsequence of the target that matches the input. If the target attribute of the
         * comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates
         * to true if it finds an exact match with any member of the set.
         * </p>
         * <p>
         * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the operator
         * checks for the absence of a substring match. If the target attribute of the comparison is Binary, then the
         * operator checks for the absence of a subsequence of the target that matches the input. If the target
         * attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the
         * operator evaluates to true if it <i>does not</i> find an exact match with any member of the set.
         * </p>
         * <p>
         * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BEGINS_WITH</code> : Checks for a prefix.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
         * (not a Number or a set type). The target attribute of the comparison must be of type String or Binary (not a
         * Number or a set type).
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>IN</code> : Checks for matching elements in a list.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
         * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of
         * the input are equal to the item attribute, the expression evaluates to true.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
         * </p>
         * <p>
         * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
         * either String, Number, or Binary (not a set type). A target attribute matches if the target value is greater
         * than, or equal to, the first element and less than, or equal to, the second element. If an item contains an
         * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does
         * not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
         * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param comparisonOperator
         *        A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals,
         *        greater than, less than, etc.</p>
         *        <p>
         *        The following comparison operators are available:
         *        </p>
         *        <p>
         *        <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         *        </p>
         *        <p>
         *        The following are descriptions of each comparison operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> element of a different type than the one provided in the request, the
         *        value does not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and
         *        maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> of a different type than the one provided in the request, the value does
         *        not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LE</code> : Less than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LT</code> : Less than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
         *        different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GE</code> : Greater than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GT</code> : Greater than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the existence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a
         *        Boolean <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type
         *        is not relevant to the <code>NOT_NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the nonexistence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a
         *        Boolean <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is
         *        not relevant to the <code>NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is of type
         *        String, then the operator checks for a substring match. If the target attribute of the comparison is
         *        of type Binary, then the operator looks for a subsequence of the target that matches the input. If the
         *        target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>
         *        "), then the operator evaluates to true if it finds an exact match with any member of the set.
         *        </p>
         *        <p>
         *        CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is a String,
         *        then the operator checks for the absence of a substring match. If the target attribute of the
         *        comparison is Binary, then the operator checks for the absence of a subsequence of the target that
         *        matches the input. If the target attribute of the comparison is a set ("<code>SS</code>", "
         *        <code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it <i>does not</i>
         *        find an exact match with any member of the set.
         *        </p>
         *        <p>
         *        NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BEGINS_WITH</code> : Checks for a prefix.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or
         *        Binary (not a Number or a set type). The target attribute of the comparison must be of type String or
         *        Binary (not a Number or a set type).
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>IN</code> : Checks for matching elements in a list.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
         *        String, Number, or Binary. These attributes are compared against an existing attribute of an item. If
         *        any elements of the input are equal to the item attribute, the expression evaluates to true.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
         *        value.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same
         *        type, either String, Number, or Binary (not a set type). A target attribute matches if the target
         *        value is greater than, or equal to, the first element and less than, or equal to, the second element.
         *        If an item contains an <code>AttributeValue</code> element of a different type than the one provided
         *        in the request, the value does not match. For example, <code>{"S":"6"}</code> does not compare to
         *        <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         *        <code>{"NS":["6", "2", "1"]}</code>
         *        </p>
         *        </li>
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals, greater
         * than, less than, etc.
         * </p>
         * <p>
         * The following comparison operators are available:
         * </p>
         * <p>
         * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         * </p>
         * <p>
         * The following are descriptions of each comparison operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
         * element of a different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
         * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
         * different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LE</code> : Less than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LT</code> : Less than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
         * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
         * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
         * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GE</code> : Greater than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GT</code> : Greater than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
         * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
         * relevant to the <code>NOT_NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
         * <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant to
         * the <code>NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
         * operator checks for a substring match. If the target attribute of the comparison is of type Binary, then the
         * operator looks for a subsequence of the target that matches the input. If the target attribute of the
         * comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates
         * to true if it finds an exact match with any member of the set.
         * </p>
         * <p>
         * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the operator
         * checks for the absence of a substring match. If the target attribute of the comparison is Binary, then the
         * operator checks for the absence of a subsequence of the target that matches the input. If the target
         * attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the
         * operator evaluates to true if it <i>does not</i> find an exact match with any member of the set.
         * </p>
         * <p>
         * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BEGINS_WITH</code> : Checks for a prefix.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
         * (not a Number or a set type). The target attribute of the comparison must be of type String or Binary (not a
         * Number or a set type).
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>IN</code> : Checks for matching elements in a list.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
         * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of
         * the input are equal to the item attribute, the expression evaluates to true.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
         * </p>
         * <p>
         * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
         * either String, Number, or Binary (not a set type). A target attribute matches if the target value is greater
         * than, or equal to, the first element and less than, or equal to, the second element. If an item contains an
         * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does
         * not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
         * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param comparisonOperator
         *        A comparator for evaluating attributes in the <code>AttributeValueList</code>. For example, equals,
         *        greater than, less than, etc.</p>
         *        <p>
         *        The following comparison operators are available:
         *        </p>
         *        <p>
         *        <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         *        </p>
         *        <p>
         *        The following are descriptions of each comparison operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> element of a different type than the one provided in the request, the
         *        value does not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and
         *        maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> of a different type than the one provided in the request, the value does
         *        not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LE</code> : Less than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LT</code> : Less than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
         *        different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GE</code> : Greater than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GT</code> : Greater than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the existence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a
         *        Boolean <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type
         *        is not relevant to the <code>NOT_NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the nonexistence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a
         *        Boolean <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is
         *        not relevant to the <code>NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is of type
         *        String, then the operator checks for a substring match. If the target attribute of the comparison is
         *        of type Binary, then the operator looks for a subsequence of the target that matches the input. If the
         *        target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>
         *        "), then the operator evaluates to true if it finds an exact match with any member of the set.
         *        </p>
         *        <p>
         *        CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is a String,
         *        then the operator checks for the absence of a substring match. If the target attribute of the
         *        comparison is Binary, then the operator checks for the absence of a subsequence of the target that
         *        matches the input. If the target attribute of the comparison is a set ("<code>SS</code>", "
         *        <code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it <i>does not</i>
         *        find an exact match with any member of the set.
         *        </p>
         *        <p>
         *        NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BEGINS_WITH</code> : Checks for a prefix.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or
         *        Binary (not a Number or a set type). The target attribute of the comparison must be of type String or
         *        Binary (not a Number or a set type).
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>IN</code> : Checks for matching elements in a list.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
         *        String, Number, or Binary. These attributes are compared against an existing attribute of an item. If
         *        any elements of the input are equal to the item attribute, the expression evaluates to true.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
         *        value.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same
         *        type, either String, Number, or Binary (not a set type). A target attribute matches if the target
         *        value is greater than, or equal to, the first element and less than, or equal to, the second element.
         *        If an item contains an <code>AttributeValue</code> element of a different type than the one provided
         *        in the request, the value does not match. For example, <code>{"S":"6"}</code> does not compare to
         *        <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         *        <code>{"NS":["6", "2", "1"]}</code>
         *        </p>
         *        </li>
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * <p>
         * For information on specifying data types in JSON, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataFormat.html">JSON Data Format</a>
         * in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param attributeValueList
         *        One or more values to evaluate against the supplied attribute. The number of values in the list
         *        depends on the <code>ComparisonOperator</code> being used.</p>
         *        <p>
         *        For type Number, value comparisons are numeric.
         *        </p>
         *        <p>
         *        String value comparisons for greater than, equals, or less than are based on ASCII character code
         *        values. For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
         *        <code>B</code>. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         *        </p>
         *        <p>
         *        For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         *        </p>
         *        <p>
         *        For information on specifying data types in JSON, see <a
         *        href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataFormat.html">JSON Data
         *        Format</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueList(Collection<AttributeValue> attributeValueList);

        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * <p>
         * For information on specifying data types in JSON, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataFormat.html">JSON Data Format</a>
         * in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param attributeValueList
         *        One or more values to evaluate against the supplied attribute. The number of values in the list
         *        depends on the <code>ComparisonOperator</code> being used.</p>
         *        <p>
         *        For type Number, value comparisons are numeric.
         *        </p>
         *        <p>
         *        String value comparisons for greater than, equals, or less than are based on ASCII character code
         *        values. For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
         *        <code>B</code>. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         *        </p>
         *        <p>
         *        For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         *        </p>
         *        <p>
         *        For information on specifying data types in JSON, see <a
         *        href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataFormat.html">JSON Data
         *        Format</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueList(AttributeValue... attributeValueList);

        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * <p>
         * For information on specifying data types in JSON, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataFormat.html">JSON Data Format</a>
         * in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeValue>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeValue>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeValue>.Builder#build()} is called immediately and
         * its result is passed to {@link #attributeValueList(List<AttributeValue>)}.
         * 
         * @param attributeValueList
         *        a consumer that will call methods on {@link List<AttributeValue>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributeValueList(List<AttributeValue>)
         */
        Builder attributeValueList(Consumer<AttributeValue.Builder>... attributeValueList);
    }

    static final class BuilderImpl implements Builder {
        private AttributeValue value;

        private Boolean exists;

        private String comparisonOperator;

        private List<AttributeValue> attributeValueList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ExpectedAttributeValue model) {
            value(model.value);
            exists(model.exists);
            comparisonOperator(model.comparisonOperator);
            attributeValueList(model.attributeValueList);
        }

        public final AttributeValue.Builder getValue() {
            return value != null ? value.toBuilder() : null;
        }

        @Override
        public final Builder value(AttributeValue value) {
            this.value = value;
            return this;
        }

        public final void setValue(AttributeValue.BuilderImpl value) {
            this.value = value != null ? value.build() : null;
        }

        public final Boolean getExists() {
            return exists;
        }

        @Override
        public final Builder exists(Boolean exists) {
            this.exists = exists;
            return this;
        }

        public final void setExists(Boolean exists) {
            this.exists = exists;
        }

        public final String getComparisonOperatorAsString() {
            return comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        public final Collection<AttributeValue.Builder> getAttributeValueList() {
            return attributeValueList != null ? attributeValueList.stream().map(AttributeValue::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder attributeValueList(Collection<AttributeValue> attributeValueList) {
            this.attributeValueList = AttributeValueListCopier.copy(attributeValueList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeValueList(AttributeValue... attributeValueList) {
            attributeValueList(Arrays.asList(attributeValueList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeValueList(Consumer<AttributeValue.Builder>... attributeValueList) {
            attributeValueList(Stream.of(attributeValueList).map(c -> AttributeValue.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAttributeValueList(Collection<AttributeValue.BuilderImpl> attributeValueList) {
            this.attributeValueList = AttributeValueListCopier.copyFromBuilder(attributeValueList);
        }

        @Override
        public ExpectedAttributeValue build() {
            return new ExpectedAttributeValue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
