/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The capacity units consumed by an operation. The data returned includes the total provisioned throughput consumed,
 * along with statistics for the table and any indexes involved in the operation. <code>ConsumedCapacity</code> is only
 * returned if the request asked for it. For more information, see <a
 * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ProvisionedThroughputIntro.html">Provisioned
 * Throughput</a> in the <i>Amazon DynamoDB Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConsumedCapacity implements SdkPojo, Serializable,
        ToCopyableBuilder<ConsumedCapacity.Builder, ConsumedCapacity> {
    private static final SdkField<String> TABLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TableName").getter(getter(ConsumedCapacity::tableName)).setter(setter(Builder::tableName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TableName").build()).build();

    private static final SdkField<Double> CAPACITY_UNITS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("CapacityUnits").getter(getter(ConsumedCapacity::capacityUnits)).setter(setter(Builder::capacityUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityUnits").build()).build();

    private static final SdkField<Double> READ_CAPACITY_UNITS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("ReadCapacityUnits").getter(getter(ConsumedCapacity::readCapacityUnits))
            .setter(setter(Builder::readCapacityUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadCapacityUnits").build()).build();

    private static final SdkField<Double> WRITE_CAPACITY_UNITS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("WriteCapacityUnits").getter(getter(ConsumedCapacity::writeCapacityUnits))
            .setter(setter(Builder::writeCapacityUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WriteCapacityUnits").build())
            .build();

    private static final SdkField<Capacity> TABLE_FIELD = SdkField.<Capacity> builder(MarshallingType.SDK_POJO)
            .memberName("Table").getter(getter(ConsumedCapacity::table)).setter(setter(Builder::table))
            .constructor(Capacity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Table").build()).build();

    private static final SdkField<Map<String, Capacity>> LOCAL_SECONDARY_INDEXES_FIELD = SdkField
            .<Map<String, Capacity>> builder(MarshallingType.MAP)
            .memberName("LocalSecondaryIndexes")
            .getter(getter(ConsumedCapacity::localSecondaryIndexes))
            .setter(setter(Builder::localSecondaryIndexes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalSecondaryIndexes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Capacity> builder(MarshallingType.SDK_POJO)
                                            .constructor(Capacity::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, Capacity>> GLOBAL_SECONDARY_INDEXES_FIELD = SdkField
            .<Map<String, Capacity>> builder(MarshallingType.MAP)
            .memberName("GlobalSecondaryIndexes")
            .getter(getter(ConsumedCapacity::globalSecondaryIndexes))
            .setter(setter(Builder::globalSecondaryIndexes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlobalSecondaryIndexes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Capacity> builder(MarshallingType.SDK_POJO)
                                            .constructor(Capacity::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TABLE_NAME_FIELD,
            CAPACITY_UNITS_FIELD, READ_CAPACITY_UNITS_FIELD, WRITE_CAPACITY_UNITS_FIELD, TABLE_FIELD,
            LOCAL_SECONDARY_INDEXES_FIELD, GLOBAL_SECONDARY_INDEXES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String tableName;

    private final Double capacityUnits;

    private final Double readCapacityUnits;

    private final Double writeCapacityUnits;

    private final Capacity table;

    private final Map<String, Capacity> localSecondaryIndexes;

    private final Map<String, Capacity> globalSecondaryIndexes;

    private ConsumedCapacity(BuilderImpl builder) {
        this.tableName = builder.tableName;
        this.capacityUnits = builder.capacityUnits;
        this.readCapacityUnits = builder.readCapacityUnits;
        this.writeCapacityUnits = builder.writeCapacityUnits;
        this.table = builder.table;
        this.localSecondaryIndexes = builder.localSecondaryIndexes;
        this.globalSecondaryIndexes = builder.globalSecondaryIndexes;
    }

    /**
     * <p>
     * The name of the table that was affected by the operation.
     * </p>
     * 
     * @return The name of the table that was affected by the operation.
     */
    public final String tableName() {
        return tableName;
    }

    /**
     * <p>
     * The total number of capacity units consumed by the operation.
     * </p>
     * 
     * @return The total number of capacity units consumed by the operation.
     */
    public final Double capacityUnits() {
        return capacityUnits;
    }

    /**
     * <p>
     * The total number of read capacity units consumed by the operation.
     * </p>
     * 
     * @return The total number of read capacity units consumed by the operation.
     */
    public final Double readCapacityUnits() {
        return readCapacityUnits;
    }

    /**
     * <p>
     * The total number of write capacity units consumed by the operation.
     * </p>
     * 
     * @return The total number of write capacity units consumed by the operation.
     */
    public final Double writeCapacityUnits() {
        return writeCapacityUnits;
    }

    /**
     * <p>
     * The amount of throughput consumed on the table affected by the operation.
     * </p>
     * 
     * @return The amount of throughput consumed on the table affected by the operation.
     */
    public final Capacity table() {
        return table;
    }

    /**
     * For responses, this returns true if the service returned a value for the LocalSecondaryIndexes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasLocalSecondaryIndexes() {
        return localSecondaryIndexes != null && !(localSecondaryIndexes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The amount of throughput consumed on each local index affected by the operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocalSecondaryIndexes} method.
     * </p>
     * 
     * @return The amount of throughput consumed on each local index affected by the operation.
     */
    public final Map<String, Capacity> localSecondaryIndexes() {
        return localSecondaryIndexes;
    }

    /**
     * For responses, this returns true if the service returned a value for the GlobalSecondaryIndexes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasGlobalSecondaryIndexes() {
        return globalSecondaryIndexes != null && !(globalSecondaryIndexes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The amount of throughput consumed on each global index affected by the operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGlobalSecondaryIndexes} method.
     * </p>
     * 
     * @return The amount of throughput consumed on each global index affected by the operation.
     */
    public final Map<String, Capacity> globalSecondaryIndexes() {
        return globalSecondaryIndexes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(capacityUnits());
        hashCode = 31 * hashCode + Objects.hashCode(readCapacityUnits());
        hashCode = 31 * hashCode + Objects.hashCode(writeCapacityUnits());
        hashCode = 31 * hashCode + Objects.hashCode(table());
        hashCode = 31 * hashCode + Objects.hashCode(hasLocalSecondaryIndexes() ? localSecondaryIndexes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasGlobalSecondaryIndexes() ? globalSecondaryIndexes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConsumedCapacity)) {
            return false;
        }
        ConsumedCapacity other = (ConsumedCapacity) obj;
        return Objects.equals(tableName(), other.tableName()) && Objects.equals(capacityUnits(), other.capacityUnits())
                && Objects.equals(readCapacityUnits(), other.readCapacityUnits())
                && Objects.equals(writeCapacityUnits(), other.writeCapacityUnits()) && Objects.equals(table(), other.table())
                && hasLocalSecondaryIndexes() == other.hasLocalSecondaryIndexes()
                && Objects.equals(localSecondaryIndexes(), other.localSecondaryIndexes())
                && hasGlobalSecondaryIndexes() == other.hasGlobalSecondaryIndexes()
                && Objects.equals(globalSecondaryIndexes(), other.globalSecondaryIndexes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConsumedCapacity").add("TableName", tableName()).add("CapacityUnits", capacityUnits())
                .add("ReadCapacityUnits", readCapacityUnits()).add("WriteCapacityUnits", writeCapacityUnits())
                .add("Table", table()).add("LocalSecondaryIndexes", hasLocalSecondaryIndexes() ? localSecondaryIndexes() : null)
                .add("GlobalSecondaryIndexes", hasGlobalSecondaryIndexes() ? globalSecondaryIndexes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "CapacityUnits":
            return Optional.ofNullable(clazz.cast(capacityUnits()));
        case "ReadCapacityUnits":
            return Optional.ofNullable(clazz.cast(readCapacityUnits()));
        case "WriteCapacityUnits":
            return Optional.ofNullable(clazz.cast(writeCapacityUnits()));
        case "Table":
            return Optional.ofNullable(clazz.cast(table()));
        case "LocalSecondaryIndexes":
            return Optional.ofNullable(clazz.cast(localSecondaryIndexes()));
        case "GlobalSecondaryIndexes":
            return Optional.ofNullable(clazz.cast(globalSecondaryIndexes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConsumedCapacity, T> g) {
        return obj -> g.apply((ConsumedCapacity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConsumedCapacity> {
        /**
         * <p>
         * The name of the table that was affected by the operation.
         * </p>
         * 
         * @param tableName
         *        The name of the table that was affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * The total number of capacity units consumed by the operation.
         * </p>
         * 
         * @param capacityUnits
         *        The total number of capacity units consumed by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityUnits(Double capacityUnits);

        /**
         * <p>
         * The total number of read capacity units consumed by the operation.
         * </p>
         * 
         * @param readCapacityUnits
         *        The total number of read capacity units consumed by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readCapacityUnits(Double readCapacityUnits);

        /**
         * <p>
         * The total number of write capacity units consumed by the operation.
         * </p>
         * 
         * @param writeCapacityUnits
         *        The total number of write capacity units consumed by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder writeCapacityUnits(Double writeCapacityUnits);

        /**
         * <p>
         * The amount of throughput consumed on the table affected by the operation.
         * </p>
         * 
         * @param table
         *        The amount of throughput consumed on the table affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder table(Capacity table);

        /**
         * <p>
         * The amount of throughput consumed on the table affected by the operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link Capacity.Builder} avoiding the need to
         * create one manually via {@link Capacity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Capacity.Builder#build()} is called immediately and its result is
         * passed to {@link #table(Capacity)}.
         * 
         * @param table
         *        a consumer that will call methods on {@link Capacity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #table(Capacity)
         */
        default Builder table(Consumer<Capacity.Builder> table) {
            return table(Capacity.builder().applyMutation(table).build());
        }

        /**
         * <p>
         * The amount of throughput consumed on each local index affected by the operation.
         * </p>
         * 
         * @param localSecondaryIndexes
         *        The amount of throughput consumed on each local index affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localSecondaryIndexes(Map<String, Capacity> localSecondaryIndexes);

        /**
         * <p>
         * The amount of throughput consumed on each global index affected by the operation.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        The amount of throughput consumed on each global index affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(Map<String, Capacity> globalSecondaryIndexes);
    }

    static final class BuilderImpl implements Builder {
        private String tableName;

        private Double capacityUnits;

        private Double readCapacityUnits;

        private Double writeCapacityUnits;

        private Capacity table;

        private Map<String, Capacity> localSecondaryIndexes = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, Capacity> globalSecondaryIndexes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConsumedCapacity model) {
            tableName(model.tableName);
            capacityUnits(model.capacityUnits);
            readCapacityUnits(model.readCapacityUnits);
            writeCapacityUnits(model.writeCapacityUnits);
            table(model.table);
            localSecondaryIndexes(model.localSecondaryIndexes);
            globalSecondaryIndexes(model.globalSecondaryIndexes);
        }

        public final String getTableName() {
            return tableName;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final Double getCapacityUnits() {
            return capacityUnits;
        }

        public final void setCapacityUnits(Double capacityUnits) {
            this.capacityUnits = capacityUnits;
        }

        @Override
        public final Builder capacityUnits(Double capacityUnits) {
            this.capacityUnits = capacityUnits;
            return this;
        }

        public final Double getReadCapacityUnits() {
            return readCapacityUnits;
        }

        public final void setReadCapacityUnits(Double readCapacityUnits) {
            this.readCapacityUnits = readCapacityUnits;
        }

        @Override
        public final Builder readCapacityUnits(Double readCapacityUnits) {
            this.readCapacityUnits = readCapacityUnits;
            return this;
        }

        public final Double getWriteCapacityUnits() {
            return writeCapacityUnits;
        }

        public final void setWriteCapacityUnits(Double writeCapacityUnits) {
            this.writeCapacityUnits = writeCapacityUnits;
        }

        @Override
        public final Builder writeCapacityUnits(Double writeCapacityUnits) {
            this.writeCapacityUnits = writeCapacityUnits;
            return this;
        }

        public final Capacity.Builder getTable() {
            return table != null ? table.toBuilder() : null;
        }

        public final void setTable(Capacity.BuilderImpl table) {
            this.table = table != null ? table.build() : null;
        }

        @Override
        public final Builder table(Capacity table) {
            this.table = table;
            return this;
        }

        public final Map<String, Capacity.Builder> getLocalSecondaryIndexes() {
            Map<String, Capacity.Builder> result = SecondaryIndexesCapacityMapCopier.copyToBuilder(this.localSecondaryIndexes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setLocalSecondaryIndexes(Map<String, Capacity.BuilderImpl> localSecondaryIndexes) {
            this.localSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copyFromBuilder(localSecondaryIndexes);
        }

        @Override
        public final Builder localSecondaryIndexes(Map<String, Capacity> localSecondaryIndexes) {
            this.localSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copy(localSecondaryIndexes);
            return this;
        }

        public final Map<String, Capacity.Builder> getGlobalSecondaryIndexes() {
            Map<String, Capacity.Builder> result = SecondaryIndexesCapacityMapCopier.copyToBuilder(this.globalSecondaryIndexes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setGlobalSecondaryIndexes(Map<String, Capacity.BuilderImpl> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copyFromBuilder(globalSecondaryIndexes);
        }

        @Override
        public final Builder globalSecondaryIndexes(Map<String, Capacity> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copy(globalSecondaryIndexes);
            return this;
        }

        @Override
        public ConsumedCapacity build() {
            return new ConsumedCapacity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
