/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesis.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input for <code>CreateStream</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateStreamRequest extends KinesisRequest implements
        ToCopyableBuilder<CreateStreamRequest.Builder, CreateStreamRequest> {
    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(CreateStreamRequest::streamName)).setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<Integer> SHARD_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ShardCount").getter(getter(CreateStreamRequest::shardCount)).setter(setter(Builder::shardCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShardCount").build()).build();

    private static final SdkField<StreamModeDetails> STREAM_MODE_DETAILS_FIELD = SdkField
            .<StreamModeDetails> builder(MarshallingType.SDK_POJO).memberName("StreamModeDetails")
            .getter(getter(CreateStreamRequest::streamModeDetails)).setter(setter(Builder::streamModeDetails))
            .constructor(StreamModeDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamModeDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_NAME_FIELD,
            SHARD_COUNT_FIELD, STREAM_MODE_DETAILS_FIELD));

    private final String streamName;

    private final Integer shardCount;

    private final StreamModeDetails streamModeDetails;

    private CreateStreamRequest(BuilderImpl builder) {
        super(builder);
        this.streamName = builder.streamName;
        this.shardCount = builder.shardCount;
        this.streamModeDetails = builder.streamModeDetails;
    }

    /**
     * <p>
     * A name to identify the stream. The stream name is scoped to the Amazon Web Services account used by the
     * application that creates the stream. It is also scoped by Amazon Web Services Region. That is, two streams in two
     * different Amazon Web Services accounts can have the same name. Two streams in the same Amazon Web Services
     * account but in two different Regions can also have the same name.
     * </p>
     * 
     * @return A name to identify the stream. The stream name is scoped to the Amazon Web Services account used by the
     *         application that creates the stream. It is also scoped by Amazon Web Services Region. That is, two
     *         streams in two different Amazon Web Services accounts can have the same name. Two streams in the same
     *         Amazon Web Services account but in two different Regions can also have the same name.
     */
    public final String streamName() {
        return streamName;
    }

    /**
     * <p>
     * The number of shards that the stream will use. The throughput of the stream is a function of the number of
     * shards; more shards are required for greater provisioned throughput.
     * </p>
     * 
     * @return The number of shards that the stream will use. The throughput of the stream is a function of the number
     *         of shards; more shards are required for greater provisioned throughput.
     */
    public final Integer shardCount() {
        return shardCount;
    }

    /**
     * <p>
     * Indicates the capacity mode of the data stream. Currently, in Kinesis Data Streams, you can choose between an
     * <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data streams.
     * </p>
     * 
     * @return Indicates the capacity mode of the data stream. Currently, in Kinesis Data Streams, you can choose
     *         between an <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data streams.
     */
    public final StreamModeDetails streamModeDetails() {
        return streamModeDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(shardCount());
        hashCode = 31 * hashCode + Objects.hashCode(streamModeDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateStreamRequest)) {
            return false;
        }
        CreateStreamRequest other = (CreateStreamRequest) obj;
        return Objects.equals(streamName(), other.streamName()) && Objects.equals(shardCount(), other.shardCount())
                && Objects.equals(streamModeDetails(), other.streamModeDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateStreamRequest").add("StreamName", streamName()).add("ShardCount", shardCount())
                .add("StreamModeDetails", streamModeDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "ShardCount":
            return Optional.ofNullable(clazz.cast(shardCount()));
        case "StreamModeDetails":
            return Optional.ofNullable(clazz.cast(streamModeDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateStreamRequest, T> g) {
        return obj -> g.apply((CreateStreamRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KinesisRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateStreamRequest> {
        /**
         * <p>
         * A name to identify the stream. The stream name is scoped to the Amazon Web Services account used by the
         * application that creates the stream. It is also scoped by Amazon Web Services Region. That is, two streams in
         * two different Amazon Web Services accounts can have the same name. Two streams in the same Amazon Web
         * Services account but in two different Regions can also have the same name.
         * </p>
         * 
         * @param streamName
         *        A name to identify the stream. The stream name is scoped to the Amazon Web Services account used by
         *        the application that creates the stream. It is also scoped by Amazon Web Services Region. That is, two
         *        streams in two different Amazon Web Services accounts can have the same name. Two streams in the same
         *        Amazon Web Services account but in two different Regions can also have the same name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * The number of shards that the stream will use. The throughput of the stream is a function of the number of
         * shards; more shards are required for greater provisioned throughput.
         * </p>
         * 
         * @param shardCount
         *        The number of shards that the stream will use. The throughput of the stream is a function of the
         *        number of shards; more shards are required for greater provisioned throughput.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shardCount(Integer shardCount);

        /**
         * <p>
         * Indicates the capacity mode of the data stream. Currently, in Kinesis Data Streams, you can choose between an
         * <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data streams.
         * </p>
         * 
         * @param streamModeDetails
         *        Indicates the capacity mode of the data stream. Currently, in Kinesis Data Streams, you can choose
         *        between an <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data
         *        streams.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamModeDetails(StreamModeDetails streamModeDetails);

        /**
         * <p>
         * Indicates the capacity mode of the data stream. Currently, in Kinesis Data Streams, you can choose between an
         * <b>on-demand</b> capacity mode and a <b>provisioned</b> capacity mode for your data streams.
         * </p>
         * This is a convenience method that creates an instance of the {@link StreamModeDetails.Builder} avoiding the
         * need to create one manually via {@link StreamModeDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link StreamModeDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #streamModeDetails(StreamModeDetails)}.
         * 
         * @param streamModeDetails
         *        a consumer that will call methods on {@link StreamModeDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #streamModeDetails(StreamModeDetails)
         */
        default Builder streamModeDetails(Consumer<StreamModeDetails.Builder> streamModeDetails) {
            return streamModeDetails(StreamModeDetails.builder().applyMutation(streamModeDetails).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KinesisRequest.BuilderImpl implements Builder {
        private String streamName;

        private Integer shardCount;

        private StreamModeDetails streamModeDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateStreamRequest model) {
            super(model);
            streamName(model.streamName);
            shardCount(model.shardCount);
            streamModeDetails(model.streamModeDetails);
        }

        public final String getStreamName() {
            return streamName;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final Integer getShardCount() {
            return shardCount;
        }

        public final void setShardCount(Integer shardCount) {
            this.shardCount = shardCount;
        }

        @Override
        public final Builder shardCount(Integer shardCount) {
            this.shardCount = shardCount;
            return this;
        }

        public final StreamModeDetails.Builder getStreamModeDetails() {
            return streamModeDetails != null ? streamModeDetails.toBuilder() : null;
        }

        public final void setStreamModeDetails(StreamModeDetails.BuilderImpl streamModeDetails) {
            this.streamModeDetails = streamModeDetails != null ? streamModeDetails.build() : null;
        }

        @Override
        public final Builder streamModeDetails(StreamModeDetails streamModeDetails) {
            this.streamModeDetails = streamModeDetails;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateStreamRequest build() {
            return new CreateStreamRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
