/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache;


import org.jboss.cache.lock.IdentityLock;
import org.jboss.cache.config.Option;
import org.jgroups.blocks.MethodCall;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.transaction.Transaction;
import java.util.*;

/**
 * This is the value (key being the {@link GlobalTransaction}) in the transaction table
 * of TreeCache.
 * <br>A TransactionEntry maintains
 * <ul>
 * <li>Handle to local Transactions: there can be more than 1 local TX associated with a GlobalTransaction
 * <li>List of modifications (Modification)
 * <li>List of nodes that were created as part of lock acquisition. These nodes can be
 * safely deleted when a transaction is rolled back
 * <li>List of locks ({@link IdentityLock}) that have been acquired by
 * this transaction so far
 * </ul>
 *
 * @author <a href="mailto:bela@jboss.org">Bela Ban</a> Apr 14, 2003
 * @version $Revision: 2061 $
 */
public class TransactionEntry {

   static Log log = LogFactory.getLog(TransactionEntry.class);

   /**
    * Local transaction
    */
   protected Transaction ltx=null;
   protected Option option;

   /**
    * List<MethodCall> of modifications ({@link MethodCall}). They will be replicated on TX commit
    */
   protected List modification_list=new LinkedList();
   protected List cl_mod_list = new LinkedList();

   /**
    * List<MethodCall>. List of compensating {@link org.jgroups.blocks.MethodCall} objects
    * which revert the ones in <tt>modification_list</tt>. For each entry in the modification list,
    * we have a corresponding entry in this list. A rollback will simply iterate over this list in
    * reverse to undo the modifications. Note that these undo-ops will never be replicated.
    */
   protected List undo_list=new LinkedList();

   /**
    * List<IdentityLock> of locks acquired by the transaction ({@link IdentityLock})
    */
   protected List locks = new LinkedList();

   /**
    * A list of dummy uninitialised nodes created by the cache loader interceptor to load data for a
    * given node in this tx.
    */
   protected List dummyNodesCreatedByCacheLoader;

   /**
    * Constructs a new TransactionEntry.
    */
   public TransactionEntry() {
   }

   /**
    * Adds a modification to the modification list.
    */
   public void addModification(MethodCall m) {
      if (m == null) return;
      modification_list.add(m);
   }

    public void addCacheLoaderModification(MethodCall m)
    {
        if (m!=null) cl_mod_list.add(m);
    }


   /**
    * Returns all modifications.
    */
   public List getModifications() {
      return modification_list;
   }

   public List getCacheLoaderModifications()
   {
       return cl_mod_list;
   }

   /**
    * Adds an undo operation to the undo list.
    * @see #undoOperations
    */
   public void addUndoOperation(MethodCall m) {
      undo_list.add(m);
   }

   /**
    * Returns the undo operations in use.
    * Note:  This list may be concurrently modified.
    */
   public List getUndoOperations() {
      return undo_list;
   }

   /**
    * Sets the local transaction for this entry.
    */
   public void setTransaction(Transaction tx) {
      ltx=tx;
   }

   /**
    * Returns a local transaction associated with this TransactionEntry
    */
   public Transaction getTransaction() {
      return ltx;
   }

   /**
    * Adds a lock to the end of the lock list.
    */
   public void addLock(IdentityLock l) {
      if(l != null) {
         synchronized(locks) {
             if (!locks.contains(l))
             {
                 locks.add(l);
             }
         }
      }
   }

   /**
    * Add multiple locks to the lock list.
    * @param newLocks Collection<IdentityLock>
    */
   public void addLocks(Collection newLocks) {
      if(newLocks != null) {
         IdentityLock tmp;
         synchronized(locks) {
            for(Iterator it=newLocks.iterator(); it.hasNext();) {
               tmp=(IdentityLock)it.next();
                if (!locks.contains(tmp))
                {
                    locks.add(tmp);
                }
            }
         }
      }
   }

   /**
    * Returns the locks in use.
    * Note:  This list may be concurrently modified.
    */
   public List getLocks() {
      return locks;
   }

   /**
    * Calls {@link #releaseAllLocksFIFO}.
    * @deprecated don't think this is used anymore
    */
   public void releaseAllLocks(Object owner) {
      releaseAllLocksFIFO(owner);
      synchronized (locks) {
         locks.clear();
      }
   }

   /**
    * Releases all locks held by the owner, in reverse order of creation.
    * Clears the list of locks held.
    */
   public void releaseAllLocksLIFO(Object owner) {
      // I guess a copy would work as well
      // This seems fairly safe though
      synchronized (locks) {
         for (ListIterator i = locks.listIterator(locks.size()); i.hasPrevious();) {
            IdentityLock lock = (IdentityLock)i.previous();
             if (log.isTraceEnabled())
             {
                 log.trace("releasing lock for " + lock.getFqn() + " (" + lock + ")");
             }
             lock.release(owner);
         }
         locks.clear();
      }
   }

   /**
    * Releases all locks held by the owner, in order of creation.
    * Does not clear the list of locks held.
    */
   public void releaseAllLocksFIFO(Object owner) {
      // I guess a copy would work as well
      // This seems fairly safe though
      synchronized (locks) {
         for (Iterator i = locks.iterator(); i.hasNext();) {
            IdentityLock lock = (IdentityLock)i.next();
            lock.release(owner);
             if (log.isTraceEnabled())
             {
                 log.trace("releasing lock for " + lock.getFqn() + " (" + lock + ")");
             }
         }
      }
   }

   /**
    * Posts all undo operations to the TreeCache.
    */
   public void undoOperations(TreeCache cache) {
      ArrayList l;
      synchronized (undo_list) {
        l = new ArrayList(undo_list);
      }
      for (ListIterator i = l.listIterator(l.size()); i.hasPrevious();) {
         MethodCall undo_op = (MethodCall)i.previous();
         undo(undo_op, cache);
      }
   }

   private void undo(MethodCall undo_op, TreeCache cache) {
      try {
         Object retval = undo_op.invoke(cache);
          if (retval instanceof Throwable)
          {
              throw (Throwable) retval;
          }
      } catch (Throwable t) {
         log.error("undo operation failed, error=" + t);
         log.trace(t, t);
      }
   }

   /**
    * Returns debug information about this transaction.
    */
   public String toString() {
      StringBuffer sb=new StringBuffer();
      sb.append("\nmodification_list: ").append(modification_list);
      synchronized (undo_list) {
         sb.append("\nundo_list: ").append(undo_list);
      }
      synchronized (locks) {
         sb.append("\nlocks: ").append(locks);
      }
      return sb.toString();
   }

    public void loadUninitialisedNode(Fqn fqn)
    {
        if (dummyNodesCreatedByCacheLoader == null) dummyNodesCreatedByCacheLoader = new LinkedList();
        dummyNodesCreatedByCacheLoader.add(fqn);
    }

    public List getDummyNodesCreatedByCacheLoader()
    {
        return dummyNodesCreatedByCacheLoader;
    }

    /**
     * Sets a transaction-scope option override
     * @param o
     */
    public void setOption(Option o)
    {
        this.option = o;
    }

    /**
     * Retrieves a transaction scope option override
     */
    public Option getOption()
    {
        return this.option;
    }

}
