/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.marshall;

import org.jboss.invocation.MarshalledValueInputStream;
import org.jboss.serial.io.JBossObjectInputStreamSharedTree;
import org.jboss.serial.io.JBossObjectOutputStreamSharedTree;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;

/**
 * Factory class for creating object output and inut streams, switching between JDK defaults and JBoss Serialization classes.
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 * @author <a href="mailto:clebert.suconic@jboss.org">Clebert Suconic</a>
 */
public class ObjectSerializationFactory
{
    static class JBossObjectInputStreamOverride extends JBossObjectInputStreamSharedTree
    {

        public JBossObjectInputStreamOverride(InputStream input) throws IOException
        {
            super(input);
        }

        public Object readObjectOverride() throws IOException, ClassNotFoundException
        {
            ClassLoader older = this.getClassLoader();
            try
            {
                this.setClassLoader(Thread.currentThread().getContextClassLoader());
                return super.readObjectOverride();
            }
            finally
            {
                this.setClassLoader(older);
            }
        }

    }

    static boolean useJBossSerialization = false;

    static
    {
        String useJBossSerializationStr = System.getProperty("serialization.jboss", "true");
        useJBossSerialization = Boolean.valueOf(useJBossSerializationStr).booleanValue();
    }

    public static ObjectOutputStream createObjectOutputStream(OutputStream out) throws IOException
    {
        return useJBossSerialization ? new JBossObjectOutputStreamSharedTree(out) : new ObjectOutputStream(out);
    }

    public static ObjectInputStream createObjectInputStream(byte[] bytes) throws IOException
    {
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        if (useJBossSerialization)
        {
            return new JBossObjectInputStreamOverride(in);
        }
        else
        {
            return new MarshalledValueInputStream(in);
        }
    }

    public static boolean useJBossSerialization()
    {
        return useJBossSerialization;
    }
}
